/*
 * ================================================================
 * Please view:
 *
 * 	README for program information.
 * 	COPYING for distribution information.
 *
 * 	based on ident2 by Michael Bacarella (defile@nyct.net)
 *
 * ================================================================ 
 */

#include "xtelld.h"


/* ------------------------------------------------------------------
 * client_reply: 
 *	send a reply back to the client..
 * ------------------------------------------------------------------
 */
void client_reply(int sd, char *outcome)
{
    char buffer[1024];

    sprintf(buffer, "%s\r\n", outcome);
    write(sd, buffer, strlen(buffer));
}



/* ============================================================================
	LOCAL FUNCTIONS
============================================================================ */



/* ------------------------------------------------------------------
 * is_inetd_child:
 *	determine if we're running as a child of inetd
 *	or if we were started via a command line (or forked and 
 *	exec'd from a process not inetd)
 * ------------------------------------------------------------------
 */

#ifdef CYGWIN
/* under CYGWIN, xtelld has to be run as daemon,
and autodetection does not work.... */
static int is_inetd_child(void)
{
    return 0;
}

#else

static int is_inetd_child(void)
{
    struct sockaddr_in sin;
    int sinsize;

    sinsize = sizeof(struct sockaddr_in);

    /* if child of inetd, stdin (0) would be a socket */
    if (getsockname(0, (struct sockaddr *) &sin, &sinsize) == -1) {
	if (errno == ENOTSOCK)
	    return 0;
	else {			/* this is a serious error, crap! */
	    perror("getsockname");	/* print to stderr as well */
	    syslog(LOG_ERR, "getsockname: %s\n", strerror(errno));
	    exit(-1);
	}
    }
    return 1;
}

#endif

/* ------------------------------------------------------------------
 * usage:
 * ------------------------------------------------------------------
 */
static void usage(char *progname)
{
    fprintf(stderr, "usage: %s [options]\n"
	    "   --help     This information\n"
	    "   -?\n"
	    "   --alone    Force server to run standalone\n"
	    "   --inetd    Force server to act as service of inetd\n"
	    "   --version  Print version information and exit.\n"
	    "   -v\n"
	    "   -sX        Lifetime for spawned services (in seconds)\n"
	    "               ex: -s25  maintain connections for up to 25 seconds\n"
	    "   -mX        Spawn no more than X children services at a time\n"
	    "               ex: -m15  service no more than 15 requests at once\n"
	    "               note: ignored if inetd service\n"
	    "   -pX        Use port X\n"
	    "               default: 4224\n"
	    "   -n         Do not lookup addresses, use IP numbers instead\n",
	    progname);
    exit(0);
}



/* ------------------------------------------------------------------
 * print_version:
 *	wouldn't want to disappoint anyone.
 * ------------------------------------------------------------------
 */
static void print_version(void)
{
    fprintf(stderr, "xtell %s\n", VERSION);
    exit(0);
}



/* ==================================================================
 * -- MAIN --
 * execution can take two major paths. 
 * it can transform this process into a daemon, which binds to the 
 * auth port and services requests by forking..
 * ..or it will become a simple one shot inetd service handler.
 * ==================================================================
 */
int main(int argc, char *argv[])
{
    u_short i;
    char inetd_child = 0, stand_alone = 0;


    client_timeout = 3600;	/* 1 hour - should be more than enough */
    max_connections = 10;
    resolve_addr = 1;
    xtell_port = XTELL_DEFAULT_PORT;

    for (i = 1; i < argc; i++) {
	if (argv[i][0] == '-') {
	    switch (argv[i][1]) {
	    case 's':
		client_timeout = atoi(argv[i] + 2);
		break;

	    case 'm':
		max_connections = atoi(argv[i] + 2);
		break;

	    case 'p':
		xtell_port = atoi(argv[i] + 2);
		break;

	    case 'v':
		print_version();

	    case 'n':
		resolve_addr = 0;
		break;

	    case '?':
		usage(argv[0]);

	    case '-':
		if (strncmp("inetd", (argv[i] + 2), 5) == 0)
		    inetd_child = 1;

		else if (strncmp("alone", (argv[i] + 2), 5) == 0)
		    stand_alone = 1;

		else if (strncmp("version", (argv[i] + 2), 7) == 0)
		    print_version();

		else if (strcmp("help", (argv[i] + 2)) == 0)
		    usage(argv[0]);

		break;
	    }
	}
    }


    /* if nothing is forced... */
    if (stand_alone == 0 && inetd_child == 0) {
	if (is_inetd_child())	/* ...try to determine automagically */
	    inetd_child = 1;
    }

    if (inetd_child) {
	openlog("in.xtell", LOG_PID, LOG_AUTH);
	alarm(client_timeout);
	signal(SIGALRM, xtell_killtic);

	inetd_service(0, 1);

	killsock(0);
	killsock(1);
	closelog();
    } else
	daemon_service();

    exit(-1);

}
