/*
  Copyright (C) 2000-2008

  Code contributed by Greg Collecutt, Joseph Hope and Paul Cochrane

  This file is part of xmds.

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/*
  $Id: xmds_integrate_ark89ip.cc 1885 2008-03-18 15:24:56Z paultcochrane $
*/

/*! @file xmds_integrate_ark89ip.cc
  @brief Integrate element parsing classes and methods; ninth order Runge-Kutta in the interaction picture

  More detailed explanation...
*/

#include <xmds_common.h>
#include <xmds_integrate_ark89ip.h>
#include <xmds_vector.h>
#include <xmds_simulation.h>

// **************************************************************************
// **************************************************************************
//                              xmdsIntegrateARK89IP public
// **************************************************************************
// **************************************************************************

extern bool debugFlag;

long nxmdsIntegrateARK89IPs=0; //!< The number of xmds integrate ARK89IP objects

// **************************************************************************
xmdsIntegrateARK89IP::xmdsIntegrateARK89IP(
                                           const xmdsSimulation *const yourSimulation,
                                           const bool& yourVerboseMode) :
  xmdsIntegrate(yourSimulation, yourVerboseMode, false, true),
  xmdsIntegrateIP(yourSimulation, yourVerboseMode),
  xmdsIntegrateARK89(yourSimulation, yourVerboseMode) {
  if (debugFlag) {
    nxmdsIntegrateARK89IPs++;
    printf("xmdsIntegrateARK89IP::xmdsIntegrateARK89IP\n");
    printf("nxmdsIntegrateARK89IPs=%li\n", nxmdsIntegrateARK89IPs);
  }
}

// **************************************************************************
xmdsIntegrateARK89IP::~xmdsIntegrateARK89IP() {
  if (debugFlag) {
    nxmdsIntegrateARK89IPs--;
    printf("xmdsIntegrateARK89IP::~xmdsIntegrateARK89IP\n");
    printf("nxmdsIntegrateARK89IPs=%li\n", nxmdsIntegrateARK89IPs);
  }
}

// **************************************************************************
void xmdsIntegrateARK89IP::processElement(
                                          const Element *const yourElement) {
  if (debugFlag) {
    printf("xmdsIntegrateARK89IP::processElement\n");
  }


  if (verbose()) {
    printf("Processing integrate ARK89IP element ...\n");
  }


  xmdsIntegrate::processElement(yourElement);
  xmdsIntegrateARK89::processElement(yourElement);
  xmdsIntegrateIP::processElement(yourElement);

}

// **************************************************************************
// **************************************************************************
//                              xmdsIntegrateARK89IP protected
// **************************************************************************



// **************************************************************************
// **************************************************************************
//                              xmdsIntegrateARK89IP private
// **************************************************************************


// **************************************************************************
void xmdsIntegrateARK89IP::writePrototypes(
                                           FILE *const outfile) const {
  if (debugFlag) {
    printf("xmdsIntegrateARK89IP::writePrototypes\n");
  }

  const xmdsVector* mainVector;

  if (!simulation()->field()->getVector("main", mainVector)) {
    throw xmdsException("Internal error in xmdsIntegrateARKIP::writeResetRoutine: cannot find 'main' vector");
  }

  const char* typeName="";
  if (mainVector->vectorType()==COMPLEX) {
    typeName="complex";
  }
  else if (mainVector->vectorType()==DOUBLE) {
    typeName="double";
  }

  fprintf(outfile, "// ********************************************************\n");
  fprintf(outfile, "// segment %li (ARK89IP) prototypes\n", segmentNumber);
  fprintf(outfile, "\n");

  xmdsIntegrate::writePrototypes(outfile);
  xmdsIntegrateIP::writePrototypes(outfile);

  fprintf(outfile, "// integrate (ARK89IP) prototypes\n");
  fprintf(outfile, "\n");
  fprintf(outfile, "void _segment%li_reset(%s* _reset_to, double _step);\n", segmentNumber, typeName);

  xmdsIntegrateARK89::writePrototypes(outfile);
}


// **************************************************************************
void xmdsIntegrateARK89IP::writeRoutines(
                                         FILE *const outfile) const {
  if (debugFlag) {
    printf("xmdsIntegrateARK89IP::writeRoutines\n");
  }

  fprintf(outfile, "// ********************************************************\n");
  fprintf(outfile, "// segment %li (ARK89IP) routines\n", segmentNumber);
  fprintf(outfile, "\n");

  xmdsIntegrate::writeRoutines(outfile);
  xmdsIntegrateIP::writeRoutines(outfile);
  writeResetRoutine(outfile);
  xmdsIntegrateARK89::writeRoutines(outfile);
}



// **************************************************************************
void xmdsIntegrateARK89IP::writeResetRoutine(
                                             FILE *const outfile) const {
  if (debugFlag) {
    printf("xmdsIntegrateARK89IP::writeResetRoutine\n");
  }

  const char *const fieldName = simulation()->field()->name()->c_str();

  const xmdsVector* mainVector;

  if (!simulation()->field()->getVector("main", mainVector)) {
    throw xmdsException("Internal error in xmdsIntegrateARKIP::writeResetRoutine: cannot find 'main' vector");
  }

  const char* typeName="";
  if (mainVector->vectorType()==COMPLEX) {
    typeName="complex";
  }
  else if (mainVector->vectorType()==DOUBLE) {
    typeName="double";
  }

  fprintf(outfile, "/* **************************************************/\n");
  fprintf(outfile, "void _segment%li_reset(%s* _reset_to, double _step){\n", segmentNumber, typeName);
  fprintf(outfile, "\n");
  if (simulation()->parameters()->useOpenMP) {
    // loop iterations are independent, so parallelise!
    fprintf(outfile, "#ifdef _OPENMP\n"
            "#pragma omp parallel for\n"
            "#endif\n");
  }
  if (simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile, "for (long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n", fieldName);
  }
  else {
    fprintf(outfile, "for (long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n", fieldName, fieldName);
  }
  fprintf(outfile, "    _%s_main[_i1] = _reset_to[_i1];\n", fieldName);
  fprintf(outfile, "\n");

  if (usesKOperators()){
    fprintf(outfile, "_active_%s_main=_%s_main;\n", fieldName, fieldName);
    if (!Smallmemory())
      fprintf(outfile, "_segment%li_k_propagate(1);\n", segmentNumber);
    else
      fprintf(outfile, "_segment%li_k_propagate(-1.0*_step);\n", segmentNumber);
  }
  fprintf(outfile, "}\n");
  fprintf(outfile, "\n");

}

// **************************************************************************
void xmdsIntegrateARK89IP::writeSingleStepCode(
                                               FILE *const outfile,
                                               const stepCaseEnum& stepCase) const {
  if (debugFlag) {
    printf("xmdsIntegrateARK89IP::writeSingleStepCode\n");
  }

  const char *const fieldName = simulation()->field()->name()->c_str();
  const char *const propDim = simulation()->parameters()->propDimName.c_str();

  list<XMLString> tempVectorNamesList;
  tempVectorNamesList.push_back("main");

  const char* kStep="";
  if (!constantK()) {
    kStep="_step/2";
  }

  const char* indent = "  ";
  if (simulation()->parameters()->errorCheck) {
    indent = "    ";
  }

  const char* noiseVector="";
  if ((simulation()->parameters()->stochastic)&&(!noNoises())) {
    noiseVector=", _noise_vector";
  }

  if (usesKOperators()){
    if (!Smallmemory())
      fprintf(outfile, "%s_segment%li_calculate_k_operator_field(_step);\n", indent, segmentNumber);
    simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
  }
  else
    simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);

  fprintf(outfile, "%s// Step 1 \n", indent);

  fprintf(outfile, "\n");
  fprintf(outfile, "%s%s += _a[0]*_step;\n", indent, propDim);
  fprintf(outfile, "\n");

  if (simulation()->parameters()->useOpenMP) {
    // We begin a parallel region as we have two for loops following each other
    // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
            "#pragma omp parallel\n"
            "{\n"
            "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
            "#endif\n");
  }

  if (simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile, "%sfor (long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName);
  }
  else {
    fprintf(outfile, "%sfor (long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName, fieldName);
  }
  fprintf(outfile, "%s _akafield_%s_main[_i1] = _%s_main[_i1];\n", indent, fieldName, fieldName);
  if (simulation()->parameters()->useOpenMP) {
    // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
            "}\n"
            "#endif\n");
  }
  fprintf(outfile, "\n");

  if (usesKOperators()) { //a = D((1-a[0])dt) a
    fprintf(outfile, "%s_active_%s_main = _%s_main;\n", indent, fieldName, fieldName);
    fprintf(outfile, "\n");
    if (!Smallmemory()){
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(-1);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }
    else{
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(1.0*_step);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }
  }


  fprintf(outfile, "%s_active_%s_main = _akafield_%s_main;\n", indent, fieldName, fieldName);
  fprintf(outfile, "\n");

  fprintf(outfile, "%s// a_k=G[a_k, t]\n", indent);
  fprintf(outfile, "%s_segment%li_calculate_delta_a(_step, cycle%s);\n", indent, segmentNumber, noiseVector);
  fprintf(outfile, "\n");

  if (usesKOperators()) //a = D((1-a[0])dt) a
    if (!Smallmemory()){
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(-1);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }
    else{
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(1.0*_step);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }

  fprintf(outfile, "%s// Step 2 \n", indent);

  fprintf(outfile, "\n");
  fprintf(outfile, "%s%s += _a[1]*_step;\n", indent, propDim);
  fprintf(outfile, "\n");

  if (simulation()->parameters()->useOpenMP) {
    // We begin a parallel region as we have two for loops following each other
    // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
            "#pragma omp parallel\n"
            "{\n"
            "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
            "#endif\n");
  }

  if (simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile, "%sfor (long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName);
  }
  else {
    fprintf(outfile, "%sfor (long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName, fieldName);
  }
  fprintf(outfile, "%s _akbfield_%s_main[_i1] = _%s_main[_i1] + _b[1][0]*_akafield_%s_main[_i1];\n", indent, fieldName, fieldName, fieldName);
  if (simulation()->parameters()->useOpenMP) {
    // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
            "}\n"
            "#endif\n");
  }
  fprintf(outfile, "\n");

  fprintf(outfile, "%s_active_%s_main = _akbfield_%s_main;\n", indent, fieldName, fieldName);
  fprintf(outfile, "\n");

  if (usesKOperators()) //a = D(-(1-a[1])dt) a
    if (!Smallmemory()){
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(2);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
    }
    else{
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(-0.978260869565217*_step);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
    }

  fprintf(outfile, "%s// a_k=G[a_k, t]\n", indent);
  fprintf(outfile, "%s_segment%li_calculate_delta_a(_step, cycle%s);\n", indent, segmentNumber, noiseVector);
  fprintf(outfile, "\n");

  if (usesKOperators()) //a = D((1-a[1])dt) a
    if (!Smallmemory()){
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(-2);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }
    else{
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(0.978260869565217*_step);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }

  fprintf(outfile, "%s// Step 3 \n", indent);

  fprintf(outfile, "\n");
  fprintf(outfile, "%s%s += _a[2]*_step;\n", indent, propDim);
  fprintf(outfile, "\n");

  if (simulation()->parameters()->useOpenMP) {
    // We begin a parallel region as we have two for loops following each other
    // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
            "#pragma omp parallel\n"
            "{\n"
            "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
            "#endif\n");
  }

  if (simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile, "%sfor (long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName);
  }
  else {
    fprintf(outfile, "%sfor (long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName, fieldName);
  }
  fprintf(outfile, "%s _akcfield_%s_main[_i1] = _%s_main[_i1] + _b[2][0]*_akafield_%s_main[_i1] + _b[2][1]*_akbfield_%s_main[_i1];\n", indent, fieldName, fieldName, fieldName, fieldName);
  if (simulation()->parameters()->useOpenMP) {
    // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
            "}\n"
            "#endif\n");
  }
  fprintf(outfile, "\n");

  fprintf(outfile, "%s_active_%s_main = _akcfield_%s_main;\n", indent, fieldName, fieldName);
  fprintf(outfile, "\n");

  if (usesKOperators()) //a = D(-(1-a[2])dt) a
    if (!Smallmemory()){
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(3);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
    }
    else{
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(-0.903704189521999*_step);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
    }

  fprintf(outfile, "%s// a_k=G[a_k, t]\n", indent);
  fprintf(outfile, "%s_segment%li_calculate_delta_a(_step, cycle%s);\n", indent, segmentNumber, noiseVector);
  fprintf(outfile, "\n");

  if (usesKOperators()) //a = D((1-a[2])dt) a
    if (!Smallmemory()){
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(-3);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }
    else{
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(0.903704189521999*_step);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }

  fprintf(outfile, "%s// Step 4 \n", indent);

  fprintf(outfile, "\n");
  fprintf(outfile, "%s%s += _a[3]*_step;\n", indent, propDim);
  fprintf(outfile, "\n");

  if (simulation()->parameters()->useOpenMP) {
    // We begin a parallel region as we have two for loops following each other
    // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
            "#pragma omp parallel\n"
            "{\n"
            "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
            "#endif\n");
  }

  if (simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile, "%sfor (long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName);
  }
  else {
    fprintf(outfile, "%sfor (long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName, fieldName);
  }
  fprintf(outfile, "%s _akdfield_%s_main[_i1] = _%s_main[_i1] + _b[3][0]*_akafield_%s_main[_i1] + _b[3][1]*_akbfield_%s_main[_i1] + _b[3][2]*_akcfield_%s_main[_i1];\n", indent, fieldName, fieldName, fieldName, fieldName, fieldName);
  if (simulation()->parameters()->useOpenMP) {
    // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
            "}\n"
            "#endif\n");
  }
  fprintf(outfile, "\n");

  fprintf(outfile, "%s_active_%s_main = _akdfield_%s_main;\n", indent, fieldName, fieldName);
  fprintf(outfile, "\n");

  if (usesKOperators()) //a = D(-(1-a[3])dt) a
    if (!Smallmemory()){
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(4);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
    }
    else{
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(-0.855556284282999*_step);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
    }

  fprintf(outfile, "%s// a_k=G[a_k, t]\n", indent);
  fprintf(outfile, "%s_segment%li_calculate_delta_a(_step, cycle%s);\n", indent, segmentNumber, noiseVector);
  fprintf(outfile, "\n");

  if (usesKOperators()) //a = D((1-a[3])dt) a
    if (!Smallmemory()){
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(-4);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }
    else{
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(0.855556284282999*_step);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }

  fprintf(outfile, "%s// Step 5 \n", indent);

  fprintf(outfile, "\n");
  fprintf(outfile, "%s%s += _a[4]*_step;\n", indent, propDim);
  fprintf(outfile, "\n");

  if (simulation()->parameters()->useOpenMP) {
    // We begin a parallel region as we have two for loops following each other
    // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
            "#pragma omp parallel\n"
            "{\n"
            "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
            "#endif\n");
  }

  if (simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile, "%sfor (long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName);
  }
  else {
    fprintf(outfile, "%sfor (long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName, fieldName);
  }
  fprintf(outfile, "%s _akefield_%s_main[_i1] = _%s_main[_i1] + _b[4][0]*_akafield_%s_main[_i1] + _b[4][1]*_akbfield_%s_main[_i1] + _b[4][2]*_akcfield_%s_main[_i1] + _b[4][3]*_akdfield_%s_main[_i1];\n", indent, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName);
  if (simulation()->parameters()->useOpenMP) {
    // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
            "}\n"
            "#endif\n");
  }
  fprintf(outfile, "\n");

  fprintf(outfile, "%s_active_%s_main = _akefield_%s_main;\n", indent, fieldName, fieldName);
  fprintf(outfile, "\n");

  if (usesKOperators()) //a = D(-(1-a[4])dt) a
    if (!Smallmemory()){
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(5);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
    }
    else{
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(-0.477941176470588*_step);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
    }

  fprintf(outfile, "%s// a_k=G[a_k, t]\n", indent);
  fprintf(outfile, "%s_segment%li_calculate_delta_a(_step, cycle%s);\n", indent, segmentNumber, noiseVector);
  fprintf(outfile, "\n");

  if (usesKOperators()) //a = D((1-a[4])dt) a
    if (!Smallmemory()){
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(-5);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }
    else{
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(0.477941176470588*_step);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }

  fprintf(outfile, "%s// Step 6 \n", indent);

  fprintf(outfile, "\n");
  fprintf(outfile, "%s%s += _a[5]*_step;\n", indent, propDim);
  fprintf(outfile, "\n");

  if (simulation()->parameters()->useOpenMP) {
    // We begin a parallel region as we have two for loops following each other
    // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
            "#pragma omp parallel\n"
            "{\n"
            "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
            "#endif\n");
  }

  if (simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile, "%sfor (long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName);
  }
  else {
    fprintf(outfile, "%sfor (long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName, fieldName);
  }
  fprintf(outfile, "%s _akifield_%s_main[_i1] = _%s_main[_i1] + _b[5][0]*_akafield_%s_main[_i1] + _b[5][3]*_akdfield_%s_main[_i1] + _b[5][4]*_akefield_%s_main[_i1];\n", indent, fieldName, fieldName, fieldName, fieldName, fieldName);
  if (simulation()->parameters()->useOpenMP) {
    // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
            "}\n"
            "#endif\n");
  }
  fprintf(outfile, "\n");

  fprintf(outfile, "%s_active_%s_main = _akifield_%s_main;\n", indent, fieldName, fieldName);
  fprintf(outfile, "\n");

  if (usesKOperators()) //a = D(-(1-a[6])dt) a
    if (!Smallmemory()){
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(6);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
    }
    else{
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(-0.771575563871365*_step);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
    }

  fprintf(outfile, "%s// a_k=G[a_k, t]\n", indent);
  fprintf(outfile, "%s_segment%li_calculate_delta_a(_step, cycle%s);\n", indent, segmentNumber, noiseVector);
  fprintf(outfile, "\n");

  if (usesKOperators()) //a = D((1-a[6])dt) a
    if (!Smallmemory()){
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(-6);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }
    else{
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(0.771575563871365*_step);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }

  fprintf(outfile, "%s// Step 7 \n", indent);

  fprintf(outfile, "\n");
  fprintf(outfile, "%s%s += _a[6]*_step;\n", indent, propDim);
  fprintf(outfile, "\n");

  if (simulation()->parameters()->useOpenMP) {
    // We begin a parallel region as we have two for loops following each other
    // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
            "#pragma omp parallel\n"
            "{\n"
            "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
            "#endif\n");
  }

  if (simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile, "%sfor (long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName);
  }
  else {
    fprintf(outfile, "%sfor (long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName, fieldName);
  }
  fprintf(outfile, "%s _akjfield_%s_main[_i1] = _%s_main[_i1] + _b[6][0]*_akafield_%s_main[_i1] + _b[6][3]*_akdfield_%s_main[_i1] + _b[6][4]*_akefield_%s_main[_i1] + _b[6][5]*_akifield_%s_main[_i1];\n", indent, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName);
  if (simulation()->parameters()->useOpenMP) {
    // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
            "}\n"
            "#endif\n");
  }
  fprintf(outfile, "\n");

  fprintf(outfile, "%s_active_%s_main = _akjfield_%s_main;\n", indent, fieldName, fieldName);
  fprintf(outfile, "\n");

  if (usesKOperators()) //a = D(-(1-a[6])dt) a
    if (!Smallmemory()){
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(7);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
    }
    else{
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(-0.456396464100663*_step);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
    }

  fprintf(outfile, "%s// a_k=G[a_k, t]\n", indent);
  fprintf(outfile, "%s_segment%li_calculate_delta_a(_step, cycle%s);\n", indent, segmentNumber, noiseVector);
  fprintf(outfile, "\n");

  if (usesKOperators()) //a = D((1-a[6])dt) a
    if (!Smallmemory()){
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(-7);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }
    else{
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(0.456396464100663*_step);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }

  fprintf(outfile, "%s// Step 8 \n", indent);

  fprintf(outfile, "\n");
  fprintf(outfile, "%s%s += _a[7]*_step;\n", indent, propDim);
  fprintf(outfile, "\n");

  if (simulation()->parameters()->useOpenMP) {
    // We begin a parallel region as we have two for loops following each other
    // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
            "#pragma omp parallel\n"
            "{\n"
            "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
            "#endif\n");
  }

  if (simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile, "%sfor (long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName);
  }
  else {
    fprintf(outfile, "%sfor (long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName, fieldName);
  }
  fprintf(outfile, "%s _akbfield_%s_main[_i1] = _%s_main[_i1] + _b[7][0]*_akafield_%s_main[_i1] + _b[7][5]*_akifield_%s_main[_i1] + _b[7][6]*_akjfield_%s_main[_i1];\n", indent, fieldName, fieldName, fieldName, fieldName, fieldName);
  if (simulation()->parameters()->useOpenMP) {
    // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
            "}\n"
            "#endif\n");
  }
  fprintf(outfile, "\n");

  fprintf(outfile, "%s_active_%s_main = _akbfield_%s_main;\n", indent, fieldName, fieldName);
  fprintf(outfile, "\n");

  if (usesKOperators()) //a = D(-(1-a[7])dt) a
    if (!Smallmemory()){
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(8);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
    }
    else{
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(-0.356643356643357*_step);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
    }

  fprintf(outfile, "%s// a_k=G[a_k, t]\n", indent);
  fprintf(outfile, "%s_segment%li_calculate_delta_a(_step, cycle%s);\n", indent, segmentNumber, noiseVector);
  fprintf(outfile, "\n");

  if (usesKOperators()) //a = D((1-a[7])dt) a
    if (!Smallmemory()){
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(-8);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }
    else{
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(0.356643356643357*_step);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }

  fprintf(outfile, "%s// Step 9 \n", indent);

  fprintf(outfile, "\n");
  fprintf(outfile, "%s%s += _a[8]*_step;\n", indent, propDim);
  fprintf(outfile, "\n");

  if (simulation()->parameters()->useOpenMP) {
    // We begin a parallel region as we have two for loops following each other
    // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
            "#pragma omp parallel\n"
            "{\n"
            "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
            "#endif\n");
  }

  if (simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile, "%sfor (long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName);
  }
  else {
    fprintf(outfile, "%sfor (long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName, fieldName);
  }
  fprintf(outfile, "%s _akcfield_%s_main[_i1] = _%s_main[_i1] + _b[8][0]*_akafield_%s_main[_i1] + _b[8][5]*_akifield_%s_main[_i1] + _b[8][6]*_akjfield_%s_main[_i1]+ _b[8][7]*_akbfield_%s_main[_i1];\n", indent, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName);
  if (simulation()->parameters()->useOpenMP) {
    // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
            "}\n"
            "#endif\n");
  }
  fprintf(outfile, "\n");

  fprintf(outfile, "%s_active_%s_main = _akcfield_%s_main;\n", indent, fieldName, fieldName);
  fprintf(outfile, "\n");

  if (usesKOperators()) //a = D(-(1-a[8])dt) a
    if (!Smallmemory()){
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(9);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
    }
    else{
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(-0.517482517482518*_step);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
    }

  fprintf(outfile, "%s// a_k=G[a_k, t]\n", indent);
  fprintf(outfile, "%s_segment%li_calculate_delta_a(_step, cycle%s);\n", indent, segmentNumber, noiseVector);
  fprintf(outfile, "\n");

  if (usesKOperators()) //a = D((1-a[8])dt) a
    if (!Smallmemory()){
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(-9);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }
    else{
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(0.517482517482518*_step);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }

  fprintf(outfile, "%s// Step 10 \n", indent);

  fprintf(outfile, "\n");
  fprintf(outfile, "%s%s += _a[9]*_step;\n", indent, propDim);
  fprintf(outfile, "\n");

  if (simulation()->parameters()->useOpenMP) {
    // We begin a parallel region as we have two for loops following each other
    // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
            "#pragma omp parallel\n"
            "{\n"
            "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
            "#endif\n");
  }

  if (simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile, "%sfor (long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName);
  }
  else {
    fprintf(outfile, "%sfor (long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName, fieldName);
  }
  fprintf(outfile, "%s _akdfield_%s_main[_i1] = _%s_main[_i1] + _b[9][0]*_akafield_%s_main[_i1] + _b[9][5]*_akifield_%s_main[_i1] + _b[9][6]*_akjfield_%s_main[_i1]+ _b[9][7]*_akbfield_%s_main[_i1]+ _b[9][8]*_akcfield_%s_main[_i1];\n", indent, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName);
  if (simulation()->parameters()->useOpenMP) {
    // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
            "}\n"
            "#endif\n");
  }
  fprintf(outfile, "\n");

  fprintf(outfile, "%s_active_%s_main = _akdfield_%s_main;\n", indent, fieldName, fieldName);
  fprintf(outfile, "\n");

  if (usesKOperators()) //a = D(-(1-a[9])dt) a
    if (!Smallmemory()){
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(10);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
    }
    else{
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(-0.931818181818182*_step);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
    }

  fprintf(outfile, "%s// a_k=G[a_k, t]\n", indent);
  fprintf(outfile, "%s_segment%li_calculate_delta_a(_step, cycle%s);\n", indent, segmentNumber, noiseVector);
  fprintf(outfile, "\n");

  if (usesKOperators()) //a = D((1-a[9])dt) a
    if (!Smallmemory()){
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(-10);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }
    else{
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(0.931818181818182*_step);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }

  fprintf(outfile, "%s// Step 11 \n", indent);

  fprintf(outfile, "\n");
  fprintf(outfile, "%s%s += _a[10]*_step;\n", indent, propDim);
  fprintf(outfile, "\n");

  if (simulation()->parameters()->useOpenMP) {
    // We begin a parallel region as we have two for loops following each other
    // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
            "#pragma omp parallel\n"
            "{\n"
            "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
            "#endif\n");
  }

  if (simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile, "%sfor (long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName);
  }
  else {
    fprintf(outfile, "%sfor (long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName, fieldName);
  }
  fprintf(outfile, "%s _akefield_%s_main[_i1] = _%s_main[_i1] + _b[10][0]*_akafield_%s_main[_i1] + _b[10][5]*_akifield_%s_main[_i1] + _b[10][6]*_akjfield_%s_main[_i1]+ _b[10][7]*_akbfield_%s_main[_i1] + _b[10][8]*_akcfield_%s_main[_i1] + _b[10][9]*_akdfield_%s_main[_i1];\n", indent, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName);
  if (simulation()->parameters()->useOpenMP) {
    // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
            "}\n"
            "#endif\n");
  }
  fprintf(outfile, "\n");

  fprintf(outfile, "%s_active_%s_main = _akefield_%s_main;\n", indent, fieldName, fieldName);
  fprintf(outfile, "\n");

  if (usesKOperators()) //a = D(-(1-a[10])dt) a
    if (!Smallmemory()){
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(11);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
    }
    else{
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(-0.749391727493917*_step);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
    }

  fprintf(outfile, "%s// a_k=G[a_k, t]\n", indent);
  fprintf(outfile, "%s_segment%li_calculate_delta_a(_step, cycle%s);\n", indent, segmentNumber, noiseVector);
  fprintf(outfile, "\n");

  if (usesKOperators()) //a = D((1-a[10])dt) a
    if (!Smallmemory()){
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(-11);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }
    else{
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(0.749391727493917*_step);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }

  fprintf(outfile, "%s// Step 12 \n", indent);

  fprintf(outfile, "\n");
  fprintf(outfile, "%s%s += _a[11]*_step;\n", indent, propDim);
  fprintf(outfile, "\n");

  if (simulation()->parameters()->useOpenMP) {
    // We begin a parallel region as we have two for loops following each other
    // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
            "#pragma omp parallel\n"
            "{\n"
            "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
            "#endif\n");
  }

  if (simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile, "%sfor (long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName);
  }
  else {
    fprintf(outfile, "%sfor (long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName, fieldName);
  }
  fprintf(outfile, "%s _akffield_%s_main[_i1] = _%s_main[_i1] + _b[11][0]*_akafield_%s_main[_i1] + _b[11][5]*_akifield_%s_main[_i1] + _b[11][6]*_akjfield_%s_main[_i1]+ _b[11][7]*_akbfield_%s_main[_i1] + _b[11][8]*_akcfield_%s_main[_i1] + _b[11][9]*_akdfield_%s_main[_i1] + _b[11][10]*_akefield_%s_main[_i1];\n", indent, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName);
  if (simulation()->parameters()->useOpenMP) {
    // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
            "}\n"
            "#endif\n");
  }
  fprintf(outfile, "\n");

  fprintf(outfile, "%s_active_%s_main = _akffield_%s_main;\n", indent, fieldName, fieldName);
  fprintf(outfile, "\n");

  if (usesKOperators()) //a = D(-(1-a[11])dt) a
    if (!Smallmemory()){
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(12);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
    }
    else{
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(-0.332632840343994*_step);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
    }

  fprintf(outfile, "%s// a_k=G[a_k, t]\n", indent);
  fprintf(outfile, "%s_segment%li_calculate_delta_a(_step, cycle%s);\n", indent, segmentNumber, noiseVector);
  fprintf(outfile, "\n");

  if (usesKOperators()) //a = D((1-a[11])dt) a
    if (!Smallmemory()){
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(-12);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }
    else{
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(0.332632840343994*_step);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }

  fprintf(outfile, "%s// Step 13 \n", indent);

  fprintf(outfile, "\n");
  fprintf(outfile, "%s%s += _a[12]*_step;\n", indent, propDim);
  fprintf(outfile, "\n");

  if (simulation()->parameters()->useOpenMP) {
    // We begin a parallel region as we have two for loops following each other
    // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
            "#pragma omp parallel\n"
            "{\n"
            "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
            "#endif\n");
  }

  if (simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile, "%sfor (long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName);
  }
  else {
    fprintf(outfile, "%sfor (long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName, fieldName);
  }
  fprintf(outfile, "%s _akgfield_%s_main[_i1] = _%s_main[_i1] + _b[12][0]*_akafield_%s_main[_i1] + _b[12][5]*_akifield_%s_main[_i1] + _b[12][6]*_akjfield_%s_main[_i1]+ _b[12][7]*_akbfield_%s_main[_i1] + _b[12][8]*_akcfield_%s_main[_i1] + _b[12][9]*_akdfield_%s_main[_i1] + _b[12][10]*_akefield_%s_main[_i1] + _b[12][11]*_akffield_%s_main[_i1];\n", indent, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName);
  if (simulation()->parameters()->useOpenMP) {
    // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
            "}\n"
            "#endif\n");
  }
  fprintf(outfile, "\n");

  fprintf(outfile, "%s_active_%s_main = _akgfield_%s_main;\n", indent, fieldName, fieldName);
  fprintf(outfile, "\n");

  if (usesKOperators()) //a = D(-(1-a[12])dt) a
    if (!Smallmemory()){
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(13);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
    }
    else{
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(-0.144927536231884*_step);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
    }

  fprintf(outfile, "%s// a_k=G[a_k, t]\n", indent);
  fprintf(outfile, "%s_segment%li_calculate_delta_a(_step, cycle%s);\n", indent, segmentNumber, noiseVector);
  fprintf(outfile, "\n");

  if (usesKOperators()) //a = D((1-a[12])dt) a
    if (!Smallmemory()){
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(-13);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }
    else{
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(0.144927536231884*_step);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }

  fprintf(outfile, "%s// Step 14 \n", indent);

  fprintf(outfile, "\n");
  fprintf(outfile, "%s%s += _a[13]*_step;\n", indent, propDim);
  fprintf(outfile, "\n");

  if (simulation()->parameters()->useOpenMP) {
    // We begin a parallel region as we have two for loops following each other
    // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
            "#pragma omp parallel\n"
            "{\n"
            "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
            "#endif\n");
  }

  if (simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile, "%sfor (long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName);
  }
  else {
    fprintf(outfile, "%sfor (long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName, fieldName);
  }
  fprintf(outfile, "%s _akhfield_%s_main[_i1] = _%s_main[_i1] + _b[13][0]*_akafield_%s_main[_i1] + _b[13][5]*_akifield_%s_main[_i1] + _b[13][6]*_akjfield_%s_main[_i1]+ _b[13][7]*_akbfield_%s_main[_i1] + _b[13][8]*_akcfield_%s_main[_i1] + _b[13][9]*_akdfield_%s_main[_i1] + _b[13][10]*_akefield_%s_main[_i1] + _b[13][11]*_akffield_%s_main[_i1] + _b[13][12]*_akgfield_%s_main[_i1];\n", indent, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName);
  if (simulation()->parameters()->useOpenMP) {
    // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
            "}\n"
            "#endif\n");
  }
  fprintf(outfile, "\n");

  fprintf(outfile, "%s_active_%s_main = _akhfield_%s_main;\n", indent, fieldName, fieldName);
  fprintf(outfile, "\n");

  if (usesKOperators()) //a = D(-(1-a[13])dt) a
    if (!Smallmemory()){
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(14);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
    }
    else{
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(-0.102040816326531*_step);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
    }

  fprintf(outfile, "%s// a_k=G[a_k, t]\n", indent);
  fprintf(outfile, "%s_segment%li_calculate_delta_a(_step, cycle%s);\n", indent, segmentNumber, noiseVector);
  fprintf(outfile, "\n");

  if (usesKOperators()) //a = D((1-a[13])dt) a
    if (!Smallmemory()){
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(-14);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }
    else{
      fprintf(outfile, "%s// a_i=D(a_2*dt)[y1]\n", indent);
      fprintf(outfile, "%s_segment%li_k_propagate(0.102040816326531*_step);\n", indent, segmentNumber);
      fprintf(outfile, "\n");
      simulation()->field()->vectors2space(outfile, 0, tempVectorNamesList, indent);
    }

  fprintf(outfile, "%s// Step 15 and 16 combined to reduce memory use \n", indent);



  if (simulation()->parameters()->useOpenMP) {
    // We begin a parallel region as we have two for loops following each other
    // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
            "#pragma omp parallel\n"
            "{\n"
            "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
            "#endif\n");
  }

  if (simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile, "%sfor (long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName);
  }
  else {
    fprintf(outfile, "%sfor (long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName, fieldName);
  }
  fprintf(outfile, "%s _akifield_%s_main[_i1] = _%s_main[_i1] + _b[14][0]*_akafield_%s_main[_i1] + _b[14][5]*_akifield_%s_main[_i1] + _b[14][6]*_akjfield_%s_main[_i1]+ _b[14][7]*_akbfield_%s_main[_i1] + _b[14][8]*_akcfield_%s_main[_i1] + _b[14][9]*_akdfield_%s_main[_i1] + _b[14][10]*_akefield_%s_main[_i1] + _b[14][11]*_akffield_%s_main[_i1] + _b[14][12]*_akgfield_%s_main[_i1] + _b[14][13]*_akhfield_%s_main[_i1];\n", indent, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName);

  if (simulation()->parameters()->useOpenMP) {
    // Continuing the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
            "#pragma omp for\n"
            "#endif\n");
  }

  fprintf(outfile, "\n");

  if (simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile, "%sfor (long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName);
  }
  else {
    fprintf(outfile, "%sfor (long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName, fieldName);
  }
  fprintf(outfile, "%s _akjfield_%s_main[_i1] = (1-_b[15][5]/_b[14][5])*_%s_main[_i1] + (_b[15][0]-_b[14][0]*_b[15][5]/_b[14][5])*_akafield_%s_main[_i1] + (_b[15][5]/_b[14][5])*_akifield_%s_main[_i1] + (_b[15][6]-_b[14][6]*_b[15][5]/_b[14][5])*_akjfield_%s_main[_i1] + (_b[15][7]-_b[14][7]*_b[15][5]/_b[14][5])*_akbfield_%s_main[_i1] + (_b[15][8]-_b[14][8]*_b[15][5]/_b[14][5])*_akcfield_%s_main[_i1] + (_b[15][9]-_b[14][9]*_b[15][5]/_b[14][5])*_akdfield_%s_main[_i1] + (_b[15][10]-_b[14][10]*_b[15][5]/_b[14][5])*_akefield_%s_main[_i1] + (_b[15][11]-_b[14][11]*_b[15][5]/_b[14][5])*_akffield_%s_main[_i1] + (_b[15][12]-_b[14][12]*_b[15][5]/_b[14][5])*_akgfield_%s_main[_i1] + (_b[15][13]-_b[14][13]*_b[15][5]/_b[14][5])*_akhfield_%s_main[_i1];\n", indent, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName);


  if (simulation()->parameters()->useOpenMP) {
    // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
            "}\n"
            "#endif\n");
  }
  fprintf(outfile, "\n");



  fprintf(outfile, "\n");
  fprintf(outfile, "%s%s += _a[14]*_step;\n", indent, propDim);
  fprintf(outfile, "\n");

  fprintf(outfile, "%s_active_%s_main = _akifield_%s_main;\n", indent, fieldName, fieldName);
  fprintf(outfile, "\n");

  fprintf(outfile, "%s// a_k=G[a_k, t]\n", indent);
  fprintf(outfile, "%s_segment%li_calculate_delta_a(_step, cycle%s);\n", indent, segmentNumber, noiseVector);
  fprintf(outfile, "\n");

  fprintf(outfile, "\n");
  fprintf(outfile, "%s%s += _a[15]*_step;\n", indent, propDim);
  fprintf(outfile, "\n");

  fprintf(outfile, "%s_active_%s_main = _akjfield_%s_main;\n", indent, fieldName, fieldName);
  fprintf(outfile, "\n");

  fprintf(outfile, "%s// a_k=G[a_k, t]\n", indent);
  fprintf(outfile, "%s_segment%li_calculate_delta_a(_step, cycle%s);\n", indent, segmentNumber, noiseVector);
  fprintf(outfile, "\n");

  fprintf(outfile, "%s// Take full step\n", indent);
  fprintf(outfile, "%s\n", indent);


  fprintf(outfile, "%s// ai = a \n", indent);
  if (simulation()->parameters()->useOpenMP) {
    // We begin a parallel region as we have two for loops following each other
    // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
            "#pragma omp parallel\n"
            "{\n"
            "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
            "#endif\n");
  }

  if (simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile, "%sfor (long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName);
  }
  else {
    fprintf(outfile, "%sfor (long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName, fieldName);
  }
  fprintf(outfile, "%s _%s_init[_i1] = _%s_main[_i1];\n", indent, fieldName, fieldName);
  if (simulation()->parameters()->useOpenMP) {
    // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
            "}\n"
            "#endif\n");
  }
  fprintf(outfile, "\n");

  fprintf(outfile, "%s// a = a + etc \n", indent);
  if (simulation()->parameters()->useOpenMP) {
    // We begin a parallel region as we have two for loops following each other
    // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
            "#pragma omp parallel\n"
            "{\n"
            "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
            "#endif\n");
  }

  if (simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile, "%sfor (long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName);
  }
  else {
    fprintf(outfile, "%sfor (long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName, fieldName);
  }
  fprintf(outfile, "%s _%s_main[_i1] += _c[0]*_akafield_%s_main[_i1]+_c[7]*_akbfield_%s_main[_i1]+_c[8]*_akcfield_%s_main[_i1]+_c[9]*_akdfield_%s_main[_i1]+_c[10]*_akefield_%s_main[_i1]+_c[11]*_akffield_%s_main[_i1]+_c[12]*_akgfield_%s_main[_i1]+_c[13]*_akhfield_%s_main[_i1]+_c[14]*_akifield_%s_main[_i1]+_c[15]*_akjfield_%s_main[_i1];\n", indent, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName);
  if (simulation()->parameters()->useOpenMP) {
    // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
            "}\n"
            "#endif\n");
  }
  fprintf(outfile, "\n");

  fprintf(outfile, "%s// a* = a + etc \n", indent);
  if (simulation()->parameters()->useOpenMP) {
    // We begin a parallel region as we have two for loops following each other
    // This minimises the penalty of creating/destroying threads for silly implementations of OpenMP
    fprintf(outfile, "#ifdef _OPENMP\n"
            "#pragma omp parallel\n"
            "{\n"
            "#pragma omp for nowait\n" // we use nowait as the two loops are entirely independent
            "#endif\n");
  }

  if (simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile, "%sfor (long _i1=0; _i1<total_local_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName);
  }
  else {
    fprintf(outfile, "%sfor (long _i1=0; _i1<_%s_size*_%s_main_ncomponents; _i1++)\n", indent, fieldName, fieldName);
  }
  fprintf(outfile, "%s _akafield_%s_main[_i1] = _%s_init[_i1] + _cs[0]*_akafield_%s_main[_i1]+_cs[7]*_akbfield_%s_main[_i1]+_cs[8]*_akcfield_%s_main[_i1]+_cs[9]*_akdfield_%s_main[_i1]+_cs[10]*_akefield_%s_main[_i1]+_cs[11]*_akffield_%s_main[_i1]+_cs[12]*_akgfield_%s_main[_i1]+_cs[13]*_akhfield_%s_main[_i1]+_cs[14]*_akifield_%s_main[_i1]+_cs[15]*_akjfield_%s_main[_i1];\n", indent, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName, fieldName);
  if (simulation()->parameters()->useOpenMP) {
    // End the parallel region
    fprintf(outfile, "#ifdef _OPENMP\n"
            "}\n"
            "#endif\n");
  }
  fprintf(outfile, "\n");


  fprintf(outfile, "%s_active_%s_main = _%s_main;\n", indent, fieldName, fieldName);
  fprintf(outfile, "\n");
}

/*
 * Local variables:
 * c-indentation-style: bsd
 * c-basic-offset: 2
 * indent-tabs-mode: nil
 * End:
 *
 * vim: tabstop=2 expandtab shiftwidth=2:
 */
