/* -*- Mode: Vala; indent-tabs-mode: nil; tab-width: 2 -*- */
/*
    © 2010 Canonical Ltd
    Authors:
      Michael Terry <michael.terry@canonical.com>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, version 3 of the License.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

static string mode = null;
static bool show_version = false;
static bool verbose = false;
static string[] forces = null;
static string[] filenames = null;
static const OptionEntry[] options = {
  {"mode", 0, 0, OptionArg.STRING, ref mode, N_("gir (default) or vapi"), null},
  {"verbose", 0, 0, OptionArg.NONE, ref verbose, N_("Show verbose information"), null},
  {"version", 0, 0, OptionArg.NONE, ref show_version, N_("Show version"), null},
  {"force-pkg", 0, 0, OptionArg.STRING_ARRAY, ref forces, N_("Force the use of certain packages"), null},
  {"", 0, 0, OptionArg.FILENAME_ARRAY, ref filenames, null, null}, // remaining
  {null}
};

int main(string[] args)
{
  Intl.textdomain(Config.GETTEXT_PACKAGE);
  Intl.bindtextdomain(Config.GETTEXT_PACKAGE, Config.LOCALE_DIR);
  Intl.bind_textdomain_codeset(Config.GETTEXT_PACKAGE, "UTF-8");

  Environment.set_application_name(_("Vala Dependency Scanner"));

  OptionContext context = new OptionContext("[%s]".printf(_("FILES")));
  context.add_main_entries(options, Config.GETTEXT_PACKAGE);
  try {
    context.parse(ref args);
  } catch (Error e) {
    printerr("%s\n\n%s", e.message, context.get_help(true, null));
    return 1;
  }

  if (show_version) {
    print("%s %s\n", Environment.get_application_name(), Config.VERSION);
    return 0;
  }

  if (mode == null)
    mode = "vapi";
  if (mode != "gir" && mode != "vapi") {
    printerr("%s\n", _("Mode must be either gir or vapi"));
    return 1;
  }

  if (filenames == null) {
    printerr("%s\n", _("No filenames provided"));
    return 1;
  }

  var vinfo = new VapiInfo();

  var ctx = new Vala.CodeContext();
  Vala.CodeContext.push(ctx);

  for (int i = 0; filenames[i] != null; ++i) {
    var file = new Vala.SourceFile(ctx, Vala.SourceFileType.SOURCE, filenames[i], null, true);
    ctx.add_source_file(file);
  }

  var parser = new Vala.Parser();
  parser.parse(ctx);

  var catcher = new SymbolCatcher();
  catcher.catch(ctx, vinfo);

  var best_pkgs = choose_best_pkgs(catcher.packages);
  best_pkgs.remove("glib-2.0"); // always implicit

  // Sort the package list for cleanliness
  var pkg_list = new List<string>();
  foreach (string p in best_pkgs)
    pkg_list.insert_sorted(p, strcmp);

  foreach (weak string pkg in pkg_list)
    print("%s\n", pkg);

  return 0;
}

int compare_pkgs(string one, string two)
{
  int i;
  string suffix1 = null, suffix2 = null;

  for (i = 0; one[i] != 0; ++i) {
    if (one[i] != two[i]) {
      suffix1 = one.substring(i);
      suffix2 = two.substring(i);
      break;
    }
  }

  if (one[i] == 0)
    return 0;
  if (suffix1 != null && !suffix1[0].isdigit())
    return 0;
  if (suffix2 != null && !suffix2[0].isdigit())
    return 0;

  var num1 = double.parse(suffix1);
  var num2 = double.parse(suffix2);
  if (num1 < num2)
    return -1;
  else if (num1 > num2)
    return 1;
  else
    return 0;
}

Vala.Set<string> choose_best_pkgs(Vala.Set<string> pkgs)
{
  var rv = new Vala.HashSet<string>(str_hash, str_equal);

  // Pare down to just the highest versions in original list
  foreach (string pkg in pkgs) {
    bool add = true;
    foreach (string rvpkg in rv) {
      int cmp = compare_pkgs(pkg, rvpkg);
      if (cmp != 0) {
        if (cmp < 0) {
          add = false;
          break;
        } else {
          rv.remove(rvpkg);
          break;
        }
      }
    }
    if (add)
      rv.add(pkg);
  }

  // Now force packages specified on command line
  if (forces != null) {
    for (int i = 0; forces[i] != null; ++i) {
      bool add = true;
      // First, remove any conflicting packages
      foreach (string rvpkg in rv) {
        if (rvpkg == forces[i]) { // already have it!
          add = false;
          break;
        }
        int cmp = compare_pkgs(forces[i], rvpkg);
        if (cmp != 0) {
          rv.remove(rvpkg);
          break;
        }
      }
      if (add)
        rv.add(forces[i]);
    }
  }

  return rv;
}

public class VapiInfo : Object
{
  //       pkg              namespace        symbols
  Vala.Map<string, Vala.Map<string, Vala.Set<string>>> symbols;
  Vala.Set<string> namespaces;

  public bool is_namespace(string symbol)
  {
    return namespaces.contains(symbol);
  }

  public Vala.Set<string> pkgs_for_symbol(string symbol, Vala.Set<string> usings)
  {
    var rv = new Vala.HashSet<string>(str_hash, str_equal);
    foreach (string pkg in symbols.get_keys()) {
      Vala.Map<string, Vala.Set<string>> map = symbols.get(pkg);
      foreach (string ns in map.get_keys()) {
        if (!usings.contains(ns))
          continue;
        Vala.Set<string> syms = map.get(ns);
        if (syms.contains(symbol)) {
          if (verbose)
            printerr(_("Symbol '%s' found in package '%s'\n"), symbol, pkg);
          rv.add(pkg);
          break;
        }
      }
    }
    return rv;
  }

  construct {
    symbols = new Vala.HashMap<string, Vala.Map<string, Vala.Set<string>>>(str_hash, str_equal);
    namespaces = new Vala.HashSet<string>(str_hash, str_equal);
    foreach (string pkg in get_pkg_list())
      process_pkg(pkg);
  }

  void get_pkg_list_for_dir(string vapidir, ref Vala.Set<string> pkgs, string suffix)
  {
    try {
      string name;
      var dir = Dir.open(vapidir, 0);
      while ((name = dir.read_name()) != null) {
        if (name.has_suffix(suffix))
          pkgs.add(name.substring(0, name.length - suffix.length));
      }
    }
    catch (Error e) {
      warning("%s\n", e.message);
    }
  }

  Vala.Set<string> get_pkg_list()
  {
    Vala.Set<string> rv = new Vala.HashSet<string>(str_hash, str_equal);
    if (mode == "vapi") {
      var specific = Config.VAPI_VER_DIR;
      var generic = GLib.Path.build_filename(Config.DATA_DIR, "vala", "vapi");
      get_pkg_list_for_dir(specific, ref rv, ".vapi");
      get_pkg_list_for_dir(generic, ref rv, ".vapi");
    }
    else if (mode == "gir") {
      var girdir = GLib.Path.build_filename(Config.DATA_DIR, "gir-1.0");
      get_pkg_list_for_dir(girdir, ref rv, ".gir");
    }
    return rv;
  }

  bool is_symbol_from_pkg(Vala.Symbol symbol, string pkg)
  {
    if (mode == "gir") {
      var sym_pkgname = "%s-%s".printf(symbol.source_reference.file.gir_namespace,
                                       symbol.source_reference.file.gir_version);
      return sym_pkgname == pkg;
    }
    else
      return true;
  }

  void add_symbol(Vala.HashSet<string> symbols, Vala.Symbol symbol, string pkg)
  {
    if (is_symbol_from_pkg(symbol, pkg))
      symbols.add(symbol.name);
  }

  void process_pkg(string pkg)
  {
    var pkgmap = new Vala.HashMap<string, Vala.Set<string>>(str_hash, str_equal);

    var ctx = new Vala.CodeContext();
    Vala.CodeContext.push(ctx);

    ctx.profile = Vala.Profile.GOBJECT; // FIXME: should be selectable?
    ctx.add_external_package("glib-2.0");
    ctx.add_external_package("gobject-2.0");
    add_package(ctx, pkg);

    if (mode == "vapi") {
      var parser = new Vala.Parser();
      parser.parse(ctx);
    }
    else if (mode == "gir") {
      var parser = new Vala.GirParser();
      parser.parse(ctx);
    }

    foreach (Vala.Namespace ns in ctx.root.get_namespaces()) {
      var symbolset = new Vala.HashSet<string>(str_hash, str_equal);

      namespaces.add(ns.name);
      pkgmap.set(ns.name, symbolset);
      if (verbose)
        printerr("Reading namespace %s %s in package %s\n", ns.name, ns.source_reference.file.gir_namespace, pkg);

      // Now add all top-level symbols
      foreach (Vala.Class c in ns.get_classes())
        add_symbol(symbolset, c, pkg);
      foreach (Vala.Constant c in ns.get_constants())
        add_symbol(symbolset, c, pkg);
      foreach (Vala.Delegate d in ns.get_delegates())
        add_symbol(symbolset, d, pkg);
      foreach (Vala.Enum e in ns.get_enums())
        add_symbol(symbolset, e, pkg);
      foreach (Vala.ErrorDomain e in ns.get_error_domains())
        add_symbol(symbolset, e, pkg);
      foreach (Vala.Field f in ns.get_fields())
        add_symbol(symbolset, f, pkg);
      foreach (Vala.Interface i in ns.get_interfaces())
        add_symbol(symbolset, i, pkg);
      foreach (Vala.Method m in ns.get_methods())
        add_symbol(symbolset, m, pkg);
      foreach (Vala.Struct s in ns.get_structs())
        add_symbol(symbolset, s, pkg);
    }

    symbols.set(pkg, pkgmap);

    Vala.CodeContext.pop();
  }

  void add_package(Vala.CodeContext ctx, string pkg)
  {
    if (ctx.has_package(pkg))
      return;
    ctx.add_package(pkg);
    string package_path = null;
    if (mode == "vapi")
      package_path = ctx.get_vapi_path(pkg);
    else if (mode == "gir")
      package_path = ctx.get_gir_path(pkg);
    if (package_path != null)
      ctx.add_source_file(new Vala.SourceFile(ctx, Vala.SourceFileType.PACKAGE, package_path, null, false));
  }
}

public class SymbolCatcher : Vala.CodeVisitor {
  public Vala.Set<string> packages;

  Vala.CodeContext ctx;
  VapiInfo vinfo;

  public void catch(Vala.CodeContext ctx, VapiInfo vinfo)
  {
    this.packages = new Vala.HashSet<string>(str_hash, str_equal);
    this.ctx = ctx;
    this.vinfo = vinfo;
    this.ctx.root.accept(this);
  }

  void descend(Vala.CodeNode n) {
    //print("Descend: %s : %s : %s\n", n.to_string(), n.get_temp_name(), n.type_name);
    n.accept_children(this);
  }

  void add_symbol(string name, string? prev, Vala.List<Vala.UsingDirective>? usings)
  {
    Vala.Set<string> usings_set;

    if (prev != null && vinfo.is_namespace(name)) {
      usings_set = new Vala.HashSet<string>(str_hash, str_equal);
      usings_set.add(name);
      name = prev;
    }
    else {
      usings_set = new Vala.HashSet<string>(str_hash, str_equal);
      foreach (Vala.UsingDirective u in usings) {
        usings_set.add(u.namespace_symbol.name);
      }
    }

    var pkgs = vinfo.pkgs_for_symbol(name, usings_set);
    foreach (string pkg in pkgs)
      this.packages.add(pkg);
  }

  public override void visit_data_type (Vala.DataType data_type) {
    if (data_type is Vala.UnresolvedType) {
      var unsymbol = (data_type as Vala.UnresolvedType).unresolved_symbol;
      Vala.UnresolvedSymbol prev = null;
      while (unsymbol.inner != null) {
        prev = unsymbol;
        unsymbol = unsymbol.inner;
      }
      add_symbol(unsymbol.name, prev == null ? null : prev.name,
                 data_type.source_reference.using_directives);
    }
    descend(data_type);
  }

  public override void visit_member_access (Vala.MemberAccess expr) {
    // Unroll the Member access to the top
    Vala.MemberAccess inner = expr;
    Vala.MemberAccess prev = null;
    while (inner.inner != null && inner.inner is Vala.MemberAccess) {
      prev = inner;
      inner = inner.inner as Vala.MemberAccess;
    }
    add_symbol(inner.member_name, prev == null ? null : prev.member_name,
               expr.source_reference.using_directives);
    // Don't descend, even on original expr as that can cause crashes when
    // converting the expr to a string?
  }

  public override void visit_binary_expression (Vala.BinaryExpression expr) {
    // Will crash on descend for similar reason to MemberAccess
    //descend(expr);
  }

  public override void visit_unary_expression (Vala.UnaryExpression expr) {
    // Will crash on descend for similar reason to MemberAccess
    //descend(expr);
  }

  public override void visit_array_creation_expression (Vala.ArrayCreationExpression e) {descend(e);}
  public override void visit_assignment (Vala.Assignment a) {descend(a);}
  public override void visit_block (Vala.Block b) {descend(b);}
  public override void visit_catch_clause (Vala.CatchClause clause) {descend(clause);}
  public override void visit_class (Vala.Class cl) {descend(cl);}
  public override void visit_constant (Vala.Constant c) {descend(c);}
  public override void visit_constructor (Vala.Constructor c) {descend(c);}
  public override void visit_creation_method (Vala.CreationMethod m) {descend(m);}
  public override void visit_declaration_statement (Vala.DeclarationStatement stmt) {descend(stmt);}
  public override void visit_delegate (Vala.Delegate cb) {descend(cb);}
  public override void visit_destructor (Vala.Destructor d) {descend(d);}
  public override void visit_do_statement (Vala.DoStatement stmt) {descend(stmt);}
  public override void visit_element_access (Vala.ElementAccess expr) {descend(expr);}
  public override void visit_enum (Vala.Enum en) {descend(en);}
  public override void visit_error_domain (Vala.ErrorDomain ed) {descend(ed);}
  public override void visit_expression_statement (Vala.ExpressionStatement stmt) {descend(stmt);}
  public override void visit_field (Vala.Field f) {descend(f);}
  public override void visit_for_statement (Vala.ForStatement stmt) {descend(stmt);}
  public override void visit_foreach_statement (Vala.ForeachStatement stmt) {descend(stmt);}
  public override void visit_formal_parameter (Vala.Parameter p) {descend(p);}
  public override void visit_if_statement (Vala.IfStatement stmt) {descend(stmt);}
  public override void visit_initializer_list (Vala.InitializerList list) {descend(list);}
  public override void visit_interface (Vala.Interface iface) {descend(iface);}
  public override void visit_lambda_expression (Vala.LambdaExpression l) {descend(l);}
  public override void visit_list_literal (Vala.ListLiteral lit) {descend(lit);}
  public override void visit_local_variable (Vala.LocalVariable local) {descend(local);}
  public override void visit_loop (Vala.Loop stmt) {descend(stmt);}
  public override void visit_map_literal (Vala.MapLiteral lit) {descend(lit);}
  public override void visit_method (Vala.Method m) {descend(m);}
  public override void visit_method_call (Vala.MethodCall expr) {descend(expr);}
  public override void visit_namespace (Vala.Namespace ns) {descend(ns);}
  public override void visit_object_creation_expression (Vala.ObjectCreationExpression expr) {descend(expr);}
  public override void visit_property (Vala.Property prop) {descend(prop);}
  public override void visit_property_accessor (Vala.PropertyAccessor acc) {descend(acc);}
  public override void visit_reference_transfer_expression (Vala.ReferenceTransferExpression expr) {descend(expr);}
  public override void visit_return_statement (Vala.ReturnStatement stmt) {descend(stmt);}
  public override void visit_set_literal (Vala.SetLiteral lit) {descend(lit);}
  public override void visit_signal (Vala.Signal sig) {descend(sig);}
  public override void visit_slice_expression (Vala.SliceExpression expr) {descend(expr);}
  public override void visit_struct (Vala.Struct st) {descend(st);}
  public override void visit_switch_label (Vala.SwitchLabel label) {descend(label);}
  public override void visit_switch_section (Vala.SwitchSection section) {descend(section);}
  public override void visit_switch_statement (Vala.SwitchStatement stmt) {descend(stmt);}
  public override void visit_template (Vala.Template tmpl) {descend(tmpl);}
  public override void visit_throw_statement (Vala.ThrowStatement stmt) {descend(stmt);}
  public override void visit_try_statement (Vala.TryStatement stmt) {descend(stmt);}
  public override void visit_tuple (Vala.Tuple tuple) {descend(tuple);}
  public override void visit_using_directive (Vala.UsingDirective ns) {descend(ns);}
  public override void visit_while_statement (Vala.WhileStatement stmt) {descend(stmt);}
  public override void visit_yield_statement (Vala.YieldStatement stmt) {descend(stmt);}
}

