/*
 * Copyright (C) 2022, KylinSoft Co., Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * Authors: Nicole <buxiaoqing@kylinos.cn>
 *
 */

#include "contextmenu.h"
#include <QFileInfo>
#include <QProcess>
#include <KWindowSystem>

ContextMenu::ContextMenu(QWidget *parent) : QMenu(parent)
{
    const QByteArray id(PANEL_SETTINGS);
    if (QGSettings::isSchemaInstalled(PANEL_SETTINGS)) {
        m_gsettings = new QGSettings(id);
        m_gsettingsKeys = m_gsettings->keys();
    }
    //设置任务栏
    QAction *setPanel;
    setPanel = new QAction(this);
    setPanel->setText(tr("Set Panel"));
    this->addAction(setPanel);
    connect(setPanel, &QAction::triggered, [this](){
        QProcess::startDetached(QString("ukui-control-center -m ukccpanel"));
    });


    QAction * showtaskview = this->addAction(tr("Show Taskview"));
    showtaskview->setCheckable(true);
    if (m_gsettingsKeys.contains(SHOW_TASKVIEW_KEY)) {
        showtaskview->setChecked(m_gsettings->get(SHOW_TASKVIEW_KEY).toBool());
        connect(showtaskview, &QAction::triggered, [this] { showTaskView(); });
    }

    this->addAction(tr("Show Desktop"),
                      this,[=]{showDesktop();}
                    );

    this->addSeparator();

    if(QFileInfo::exists(QString("/usr/bin/ukui-system-monitor"))) {
        this->addAction(tr("Show System Monitor"),
                          this,[=] {systeMonitor();}
                        );
    }

    this->addSeparator();

    adjustPanel();

    QAction * m_lockAction  = this->addAction(tr("Lock This Panel"));
    m_lockAction->setCheckable(true);
    m_lockAction->setChecked(m_lockPanel);
    if (m_gsettingsKeys.contains(LOCK_PANEL_KEY)) {
        connect(m_lockAction, &QAction::triggered, [this] { m_gsettings->set(LOCK_PANEL_KEY,!m_gsettings->get(LOCK_PANEL_KEY).toBool());});
    }

    QAction *about;
    about = new QAction(this);
    about->setText(tr("About Kylin"));
    if (m_gsettingsKeys.contains(ABOUT_KYLIN_DISPLAY)) {
        if (m_gsettings->get(ABOUT_KYLIN_DISPLAY).toBool()) {
            this->addAction(about);
            connect(about,&QAction::triggered, [this] {
                QProcess::startDetached(QString("ukui-control-center -m About"));
            });
        }
    }
}

void ContextMenu::adjustPanel()
{
    if (m_gsettingsKeys.contains(LOCK_PANEL_KEY)) {
        m_lockPanel = m_gsettings->get(LOCK_PANEL_KEY).toBool();
    } else {
        m_lockPanel = false;
    }

    QAction *pmenuaction_s;
    QAction *pmenuaction_m;
    QAction *pmenuaction_l;

    pmenuaction_s = new QAction(this);
    pmenuaction_s->setText(tr("Small"));
    pmenuaction_m = new QAction(this);
    pmenuaction_m->setText(tr("Medium"));
    pmenuaction_l = new QAction(this);
    pmenuaction_l->setText(tr("Large"));


    QMenu *pmenu_panelsize;
    pmenu_panelsize = new QMenu(this);
    pmenu_panelsize->setTitle(tr("Adjustment Size"));
    pmenu_panelsize->addAction(pmenuaction_s);
    pmenu_panelsize->addAction(pmenuaction_m);
    pmenu_panelsize->addAction(pmenuaction_l);
    pmenu_panelsize->setWindowOpacity(0.9);
    this->addMenu(pmenu_panelsize);

    pmenuaction_s->setCheckable(true);
    pmenuaction_m->setCheckable(true);
    pmenuaction_l->setCheckable(true);
    if (m_gsettingsKeys.contains(PANEL_SIZE_KEY)) {
        pmenuaction_s->setChecked(m_gsettings->get(PANEL_SIZE_KEY).toInt()==PANEL_SIZE_SMALL);
        pmenuaction_m->setChecked(m_gsettings->get(PANEL_SIZE_KEY).toInt()==PANEL_SIZE_MEDIUM);
        pmenuaction_l->setChecked(m_gsettings->get(PANEL_SIZE_KEY).toInt()==PANEL_SIZE_LARGE);
    }

    connect(pmenuaction_s,&QAction::triggered,[this] {
        m_gsettings->set(PANEL_SIZE_KEY,PANEL_SIZE_SMALL);
        m_gsettings->set(ICON_SIZE_KEY,ICON_SIZE_SMALL);
    });
    connect(pmenuaction_m,&QAction::triggered,[this] {
        m_gsettings->set(PANEL_SIZE_KEY,PANEL_SIZE_MEDIUM);
        m_gsettings->set(ICON_SIZE_KEY,ICON_SIZE_MEDIUM);
    });
    connect(pmenuaction_l,&QAction::triggered,[this] {
        m_gsettings->set(PANEL_SIZE_KEY,PANEL_SIZE_LARGE);
        m_gsettings->set(ICON_SIZE_KEY,ICON_SIZE_LARGE);
    });
    pmenu_panelsize->setDisabled(m_lockPanel);

    QAction *pmenuaction_top;
    QAction *pmenuaction_bottom;
    QAction *pmenuaction_left;
    QAction *pmenuaction_right;
    pmenuaction_top = new QAction(this);
    pmenuaction_top->setText(tr("Up"));
    pmenuaction_bottom = new QAction(this);
    pmenuaction_bottom->setText(tr("Bottom"));
    pmenuaction_left = new QAction(this);
    pmenuaction_left->setText(tr("Left"));
    pmenuaction_right = new QAction(this);
    pmenuaction_right->setText(tr("Right"));
    QMenu *pmenu_positon;
    pmenu_positon = new QMenu(this);
    pmenu_positon->setTitle(tr("Adjustment Position"));
    pmenu_positon->addAction(pmenuaction_top);
    pmenu_positon->addAction(pmenuaction_bottom);
    pmenu_positon->addAction(pmenuaction_left);
    pmenu_positon->addAction(pmenuaction_right);
    this->addMenu(pmenu_positon);

    pmenuaction_top->setCheckable(true);
    pmenuaction_bottom->setCheckable(true);
    pmenuaction_left->setCheckable(true);
    pmenuaction_right->setCheckable(true);
    if (m_gsettingsKeys.contains(PANEL_POSITION_KEY)) {
        pmenuaction_top->setChecked(m_gsettings->get(PANEL_POSITION_KEY).toInt()==1);
        pmenuaction_bottom->setChecked(m_gsettings->get(PANEL_POSITION_KEY).toInt()==0);
        pmenuaction_left->setChecked(m_gsettings->get(PANEL_POSITION_KEY).toInt()==2);
        pmenuaction_right->setChecked(m_gsettings->get(PANEL_POSITION_KEY).toInt()==3);
    }

    connect(pmenuaction_top,&QAction::triggered, [this] { m_gsettings->set(PANEL_POSITION_KEY,1);});
    connect(pmenuaction_bottom,&QAction::triggered, [this] { m_gsettings->set(PANEL_POSITION_KEY,0);});
    connect(pmenuaction_left,&QAction::triggered, [this] { m_gsettings->set(PANEL_POSITION_KEY,2);});
    connect(pmenuaction_right,&QAction::triggered, [this] { m_gsettings->set(PANEL_POSITION_KEY,3);});
    pmenu_positon->setWindowOpacity(0.9);
    pmenu_positon->setDisabled(m_lockPanel);

    QAction * hidepanel = this->addAction(tr("Hide Panel"));
    hidepanel->setDisabled(m_lockPanel);
    hidepanel->setCheckable(true);
    if (m_gsettingsKeys.contains(HIDE_PANEL_KEY)) {
        hidepanel->setChecked(m_gsettings->get(HIDE_PANEL_KEY).toBool());
        connect(hidepanel, &QAction::triggered, [this] {
            m_gsettings->set(HIDE_PANEL_KEY,!m_gsettings->get(HIDE_PANEL_KEY).toBool());
        });
    }
}

/*右键　系统监视器选项*/
void ContextMenu::systeMonitor()
{
    QProcess *process = new QProcess(this);
    process->startDetached("/usr/bin/ukui-system-monitor");
    process->deleteLater();
}

/*右键　显示桌面选项*/
void ContextMenu::showDesktop()
{
    KWindowSystem::setShowingDesktop(!KWindowSystem::showingDesktop());
}

/*右键　显示任务视图　选项*/
void ContextMenu::showTaskView()
{
    if (m_gsettingsKeys.contains(SHOW_TASKVIEW_KEY)) {
        m_gsettings->set(SHOW_TASKVIEW_KEY,!m_gsettings->get(SHOW_TASKVIEW_KEY).toBool());
    }
}
