/*
 * Copyright (C) 2023, KylinSoft Co., Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
**/
#include "boot.h"

#include <QDebug>

Boot::Boot() : mFirstLoad(true)
{
    pluginName = tr("Boot");
    pluginType = COMMONINFO;
}

Boot::~Boot()
{
}

QString Boot::plugini18nName()
{
    return pluginName;
}

int Boot::pluginTypes()
{
    return pluginType;
}

QWidget *Boot::pluginUi()
{
    if (mFirstLoad) {
        mFirstLoad = false;
        bootWidget = new BootUi;

        bootDbus = new QDBusInterface("com.control.center.qt.systemdbus",
                                      "/",
                                      "com.control.center.interface",
                                      QDBusConnection::systemBus(), this);
        bootDbus->setTimeout(2147483647); // -1 为默认的25s超时;

        if (!bootDbus->isValid()) {
            qCritical() << "com.control.center.qt.systemdbus DBus error:" << bootDbus->lastError();
        } else {
            initBootStatus();
            initConnection();
        }
    }
    return bootWidget;
}

const QString Boot::name() const
{
    return QStringLiteral("Boot");
}

bool Boot::isShowOnHomePage() const
{
    return true;
}

QIcon Boot::icon() const
{
    return QIcon::fromTheme("ukui-bootmenu-symbolic");
}

bool Boot::isEnable() const
{
    return true;
}

void Boot::initConnection()
{

    initBootStatus();

    connect(bootWidget->grubSwitchButton(), &KSwitchButton::stateChanged, this, &Boot::bootSlot);

    connect(bootWidget->resetButton(), &QPushButton::clicked, this, &Boot::resetPasswdSlot);

}

void Boot::initBootStatus()
{
    if (bootDbus != nullptr) {
        QDBusReply<bool> ret = bootDbus->call("getGrupPasswdStatus");
        bootWidget->grubSwitchButton()->blockSignals(true);
        bootWidget->grubSwitchButton()->setChecked(ret);
        bootWidget->grubSwitchButton()->blockSignals(false);
    }

    bootWidget->resetButton()->setVisible(bootWidget->grubSwitchButton()->isChecked());
}

void Boot::bootSlot(bool checked)
{
    if (checked) {
        GrubVerify *dia = new GrubVerify(bootWidget);
        QPushButton *confirmBtn = dia->getConfirmBtn();
        connect(confirmBtn, &QPushButton::clicked, this, [=](){
            setGrubPasswd(dia->getPwd(), checked);
        });
        if (dia->exec() != QDialog::Accepted) {
            bootWidget->grubSwitchButton()->blockSignals(true);
            bootWidget->grubSwitchButton()->setChecked(!checked);
            bootWidget->grubSwitchButton()->blockSignals(false);
            bootWidget->resetButton()->setVisible(bootWidget->grubSwitchButton()->isChecked());
        }
    } else {
        setGrubPasswd("", checked);
    }

    bootWidget->resetButton()->setVisible(bootWidget->grubSwitchButton()->isChecked());
}

void Boot::setGrubPasswd(QString pwd, bool isOpen)
{
    inhibit("shutdown", "com.control.center.qt.systemdbus", "update-grub", "block");
    bootWidget->grubSwitchButton()->setEnabled(false);
    bootWidget->resetButton()->setEnabled(false);
    QString lang = qgetenv("LANG");
    QDBusPendingCall call = bootDbus->asyncCall("setGrupPasswd", "root", pwd, lang, isOpen);
    if (!call.isValid()) {
        qCritical() << "setGrupPasswd";
    }
    QDBusPendingCallWatcher *watcher = new QDBusPendingCallWatcher(call,this);
    connect(watcher,&QDBusPendingCallWatcher::finished,this, [=](QDBusPendingCallWatcher *call){
        QDBusPendingReply<bool> reply = *call;
        if (!reply.isValid()) {
             qCritical() << "setGrupPasswd:" << reply.error();
             uninhibit();
             bootWidget->grubSwitchButton()->blockSignals(true);
             bootWidget->grubSwitchButton()->setChecked(!isOpen);
             bootWidget->grubSwitchButton()->blockSignals(false);
             bootWidget->resetButton()->setVisible(bootWidget->grubSwitchButton()->isChecked());
        } else {
            bool status = reply.value();
            qCritical() << "setGrupPasswd:" << status;
            if (status) {
                uninhibit();
            } else {
                uninhibit();
                bootWidget->grubSwitchButton()->blockSignals(true);
                bootWidget->grubSwitchButton()->setChecked(!isOpen);
                bootWidget->grubSwitchButton()->blockSignals(false);
                bootWidget->resetButton()->setVisible(bootWidget->grubSwitchButton()->isChecked());
            }
        }
        bootWidget->grubSwitchButton()->setEnabled(true);
        bootWidget->resetButton()->setEnabled(true);
    });
}

bool Boot::inhibit(QString what, QString who, QString why, QString mode)
{
    QDBusMessage message;

    message = QDBusMessage::createMethodCall("org.freedesktop.login1",
                                             "/org/freedesktop/login1",
                                             "org.freedesktop.login1.Manager",
                                             QStringLiteral("Inhibit"));


    message.setArguments(QVariantList({what, who, why, mode}));

    QDBusPendingReply<QDBusUnixFileDescriptor> reply = QDBusConnection::systemBus().call(message);

    if (!reply.isValid()) {
        qCritical() << "inhibit failed!";
        return false;
    }
    reply.value().swap(m_inhibitFileDescriptor);
    qCritical() << "inhibit success!";
    return true;
}

void Boot::uninhibit()
{
    if (!m_inhibitFileDescriptor.isValid()) {
        return;
    }

    qCritical() << "uninhibit success!";
    m_inhibitFileDescriptor = QDBusUnixFileDescriptor();
}

void Boot::resetPasswdSlot()
{
    GrubVerify *dia =  new GrubVerify(bootWidget);
    QPushButton *confirmBtn = dia->getConfirmBtn();
    QString lang = qgetenv("LANG");
    connect(confirmBtn, &QPushButton::clicked, this, [=](){
        inhibit("shutdown", "com.control.center.qt.systemdbus", "update-grub", "block");
        bootWidget->grubSwitchButton()->setEnabled(false);
        bootWidget->resetButton()->setEnabled(false);
        QDBusPendingCall call = bootDbus->asyncCall("setGrupPasswd", "root", dia->getPwd(), lang, true);
        if (!call.isValid()) {
            qCritical() << "setGrupPasswd";
        }
        QDBusPendingCallWatcher *watcher = new QDBusPendingCallWatcher(call,this);
        connect(watcher,&QDBusPendingCallWatcher::finished,this, [=](QDBusPendingCallWatcher *call){
            QDBusPendingReply<bool> reply = *call;

            if (!reply.isValid()) {
                 qCritical() << "setGrupPasswd:" << "iserror";
                 uninhibit();
            } else {
                bool status = reply.value();
                qCritical() << "setGrupPasswd:" << status;
                if (status) {
                    uninhibit();
                } else {
                    uninhibit();
                }
            }
            bootWidget->grubSwitchButton()->setEnabled(true);
            bootWidget->resetButton()->setEnabled(true);
        });
    });
    dia->exec();
}

