#include "StatisticWorkers.h"

#include <U2Lang/ConfigurationEditor.h>
#include <U2Lang/WorkflowEnv.h>
#include <U2Lang/ActorPrototypeRegistry.h>
#include <U2Lang/CoreLibConstants.h>
#include <U2Lang/BaseSlots.h>
#include <U2Lang/BaseTypes.h>
#include <U2Lang/BasePorts.h>
#include <U2Lang/BaseActorCategories.h>
#include <U2Designer/DelegateEditors.h>
#include <U2Core/DNASequence.h>
#include <U2Core/DNAAlphabet.h>
#include <U2Core/AnnotationData.h>
#include <U2Core/FailTask.h>

namespace U2 {
namespace LocalWorkflow {

const QString DNAStatWorkerFactory::ACTOR_ID("dna-stats");

const QString GCCONTENT("gc-content");
const QString GC1CONTENT("gc1-content");
const QString GC2CONTENT("gc2-content");
const QString GC3CONTENT("gc3-content");

void DNAStatWorkerFactory::init() {
    QList<PortDescriptor*> portDescs; 
    QList<Attribute*> attribs;

    //accept sequence and annotated regions as input
    QMap<Descriptor, DataTypePtr> inputMap;
    QMap<Descriptor, DataTypePtr> outputMap;
    inputMap[ BaseSlots::DNA_SEQUENCE_SLOT() ] = BaseTypes::DNA_SEQUENCE_TYPE();
    outputMap[ BaseSlots::ANNOTATION_TABLE_SLOT() ] = BaseTypes::ANNOTATION_TABLE_TYPE();

    { //Create input port descriptors
        Descriptor inDesc( BasePorts::IN_SEQ_PORT_ID(), DNAStatWorker::tr("Input sequence"), 
            DNAStatWorker::tr("Sequence for which GC-content and GC3-content will be evaluated.") );
        Descriptor outDesc( BasePorts::OUT_ANNOTATIONS_PORT_ID(), DNAStatWorker::tr("Result annotation"), 
            DNAStatWorker::tr("Resulted annotations, with GC-content and GC3-content.") );

        portDescs << new PortDescriptor( inDesc, DataTypePtr(new MapDataType("filter.anns", inputMap)), /*input*/ true );
        portDescs << new PortDescriptor( outDesc, DataTypePtr(new MapDataType("filter.anns", outputMap)), /*input*/false, /*multi*/true );
    }

    attribs << new Attribute(Descriptor(GCCONTENT, DNAStatWorker::tr("GC-content"), DNAStatWorker::tr("Evaluate GC-content")), 
        BaseTypes::BOOL_TYPE(),false, true);
    attribs << new Attribute(Descriptor(GC1CONTENT, DNAStatWorker::tr("GC1-content"), DNAStatWorker::tr("Evaluate GC1-content")), 
        BaseTypes::BOOL_TYPE(),false, true);
    attribs << new Attribute(Descriptor(GC2CONTENT, DNAStatWorker::tr("GC2-content"), DNAStatWorker::tr("Evaluate GC2-content")), 
        BaseTypes::BOOL_TYPE(),false, true);
    attribs << new Attribute(Descriptor(GC3CONTENT, DNAStatWorker::tr("GC3-content"), DNAStatWorker::tr("Evaluate GC3-content")), 
        BaseTypes::BOOL_TYPE(),false, true);

    Descriptor desc( ACTOR_ID,
        DNAStatWorker::tr("DNA statistics"), 
        DNAStatWorker::tr("Evaluates statistics for DNA sequences") );
    ActorPrototype * proto = new IntegralBusActorPrototype( desc, portDescs, attribs );

    proto->setPrompter( new DNAStatWorkerPrompter() );
    proto->setEditor(new DelegateEditor(QMap<QString, PropertyDelegate*>()));
    WorkflowEnv::getProtoRegistry()->registerProto( BaseActorCategories::CATEGORY_STATISTIC(), proto );

    DomainFactory* localDomain = WorkflowEnv::getDomainRegistry()->getById( LocalDomainFactory::ID );
    localDomain->registerEntry( new DNAStatWorkerFactory() );
}

QString DNAStatWorkerPrompter::composeRichDoc() {
    return tr("Evaluates GC-Content and GC3-Content");
}


void DNAStatWorker::init() {
    input = ports.value(BasePorts::IN_SEQ_PORT_ID());
    output = ports.value(BasePorts::OUT_ANNOTATIONS_PORT_ID());
}

bool DNAStatWorker::isReady() {
    return input->hasMessage();
}

Task* DNAStatWorker::tick() {
    Message inputMessage = getMessageAndSetupScriptValues(input);
    QVariantMap qm = inputMessage.getData().toMap();
    DNASequence dna = qVariantValue<DNASequence>( qm.value(BaseSlots::DNA_SEQUENCE_SLOT().getId()) );

    if(!dna.alphabet->isNucleic()) {
        return new FailTask(tr("Sequence must be nucleotide"));
    }

    QList<SharedAnnotationData> res;
    SharedAnnotationData gcAnn(new AnnotationData());
    gcAnn->name = "statistics";
    gcAnn->location->regions << U2Region( 0, dna.seq.size());

    if(actor->getParameter(GCCONTENT)->getAttributeValue<bool>()) {
        float gcContent = calcGCContent(dna.seq);
        gcAnn->qualifiers.push_back(U2Qualifier("gc-content", QString::number(gcContent*100) + "%"));
    }

    if(actor->getParameter(GC1CONTENT)->getAttributeValue<bool>()) {
        float gc1Content = calcGC1Content(dna.seq);
        gcAnn->qualifiers.push_back(U2Qualifier("gc1-content", QString::number(gc1Content*100) + "%"));
    }

    if(actor->getParameter(GC2CONTENT)->getAttributeValue<bool>()) {
        float gc2Content = calcGC2Content(dna.seq);
        gcAnn->qualifiers.push_back(U2Qualifier("gc2-content", QString::number(gc2Content*100) + "%"));
    }

    if(actor->getParameter(GC3CONTENT)->getAttributeValue<bool>()) {
        float gc3Content = calcGC3Content(dna.seq);
        gcAnn->qualifiers.push_back(U2Qualifier("gc3-content", QString::number(gc3Content*100) + "%"));
    }

    if(gcAnn->qualifiers.isEmpty()) {
        return new FailTask(tr("No statistics was selected"));
    }

    res << gcAnn;

    QVariant v = qVariantFromValue<QList<SharedAnnotationData> >(res);
    output->put( Message(BaseTypes::ANNOTATION_TABLE_TYPE(), v) );
    if (input->isEnded()) {
        output->setEnded();
    }

    return NULL;
}

bool DNAStatWorker::isDone() {
    return input->isEnded();
}

float DNAStatWorker::calcGCContent(const QByteArray &seq) {
    int gc = 0;
    for(int i = 0; i < seq.size(); i++) {
        if(QRegExp("[cCgG]").exactMatch(QString(seq.at(i)))){
            gc++;
        }
    }
    float gcContent = (float)gc/seq.size();
    return gcContent;
}

float DNAStatWorker::calcGC1Content(const QByteArray &seq) {
    int gc1 = 0;
    for(int i = 0; i < seq.size(); i+=3) {
        if(QRegExp("[cCgG]").exactMatch(QString(seq.at(i)))){
            gc1++;
        }
    }
    float gc1Content = (float)gc1/seq.size()*3;
    return gc1Content;
}

float DNAStatWorker::calcGC2Content(const QByteArray &seq) {
    int gc2 = 0;
    for(int i = 1; i < seq.size(); i+=3) {
        if(QRegExp("[cCgG]").exactMatch(QString(seq.at(i)))){
            gc2++;
        }
    }
    float gc2Content = (float)gc2/seq.size()*3;
    return gc2Content;
}

float DNAStatWorker::calcGC3Content(const QByteArray &seq) {
    int gc3 = 0;
    for(int i = 2; i < seq.size(); i+=3) {
        if(QRegExp("[cCgG]").exactMatch(QString(seq.at(i)))){
            gc3++;
        }
    }
    float gc3Content = (float)gc3/seq.size()*3;
    return gc3Content;
}

} //LocalWorkflow
} //U2