# Copyright (C) 2006,2007 Daiki Ueno <ueno@unixuser.org>

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

require 'GD'

module Treil
  class GDWriter < Writer
    TTF = Pathname.new(__FILE__).dirname + 'boxfont2.ttf'

    def initialize(output, config)
      super
      @color = Hash.new
    end

    def do_write(image, rect)
      max_depth = @config['']['max_depth']
      rect.each do |r|
        next if max_depth && r.tree.depth > max_depth
        do_write_rect(image, r)
      end
      rect.each do |r|
        next if max_depth && r.tree.depth > max_depth
        if @config.name_visible?(r)
          do_write_name(image, r)
        end
      end
    end

    def allocate_color(color)
      r, g, b, a = color
      r = (r * 255).floor
      g = (g * 255).floor
      b = (b * 255).floor
      if a
        a = (a * 127).floor
        color = [r, g, b, a]
        @color[color] ||= GD::Image.trueColorAlpha(*color)
      else
        color = [r, g, b]
        @color[color] ||= GD::Image.trueColor(*color)
      end
      @color[color]
    end

    def do_write_rect(image, rect)
      return if rect.tree.depth == 0
      if rect.tree.leaf?
        image.filledRectangle(rect.x, rect.y,
			      rect.x + rect.width, rect.y + rect.height,
                              allocate_color(@config.background_color(rect)))
      end
      image.rectangle(rect.x, rect.y,
		      rect.x + rect.width, rect.y + rect.height,
                      allocate_color(@config.border_color(rect)))
    end

    def do_write_name(image, rect)
      name_color = allocate_color(@config.name_color(rect))
      name_font_size = @config.name_font_size(rect)
      error, brect = GD::Image.stringTTF(name_color, TTF.to_s, name_font_size,
					 0, 0, 0, rect.tree.name)
      width = brect[2] - brect[6]
      if width > rect.width
        name_font_size *= rect.width / (1.0 * width)
	error, brect = GD::Image.stringTTF(name_color, TTF.to_s, name_font_size,
					   0, 0, 0, rect.tree.name)
      end
      width = brect[2] - brect[6]
      height = brect[3] - brect[7]
      image.stringTTF(name_color, TTF.to_s, name_font_size, 0,
		      (rect.x + rect.width / 2) - width / 2,
                      (rect.y + rect.height / 2) - height / 2,
		      rect.tree.name)
    end
  end

  class GDJPEGWriter < GDWriter
    def self.accept?(output)
      Pathname.new(output).extname =~ /\A\.jpe?g\z/i
    end

    def write(rect)
      image = GD::Image.newTrueColor(rect.width, rect.height)
      do_write(image, rect)
      File.open(@output, 'w') do |file|
        image.jpeg(file, 95)
      end
    end

    register(self)
  end

  class GDPNGWriter < GDWriter
    def self.accept?(output)
      Pathname.new(output).extname =~ /\A\.png\z/i
    end

    def write(rect)
      image = GD::Image.newTrueColor(rect.width, rect.height)
      do_write(image, rect)
      File.open(@output, 'w') do |file|
        image.png(file)
      end
    end

    register(self)
  end
end
