// SPDX-License-Identifier: (GPL-2.0-or-later OR BSD-2-Clause)
/*
 * Traceshark - a visualizer for visualizing ftrace and perf traces
 * Copyright (C) 2015-2018, 2020  Viktor Rosendahl <viktor.rosendahl@gmail.com>
 *
 * This file is dual licensed: you can use it either under the terms of
 * the GPL, or the BSD license, at your option.
 *
 *  a) This program is free software; you can redistribute it and/or
 *     modify it under the terms of the GNU General Public License as
 *     published by the Free Software Foundation; either version 2 of the
 *     License, or (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public
 *     License along with this library; if not, write to the Free
 *     Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
 *     MA 02110-1301 USA
 *
 * Alternatively,
 *
 *  b) Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *     1. Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *     2. Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 *     THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 *     CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 *     INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 *     MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *     DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 *     CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *     SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *     NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *     LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 *     HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 *     CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 *     OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 *     EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef TTHREAD_H
#define TTHREAD_H

 /* We are going to piggyback on QThread */
#include <QThread>

#include <QList>
#include <QMap>
#include <QtCore>
#include <QString>

class TThread_;

/*
 * I hate this class. I would like to use QThread directly instead of inventing
 * my own thread class. However, QThread does not lend itself to the use of
 * templates since it inherits QObject, so here we go down the dirty road.
 */
class TThread {
	friend class TThread_;
public:
	TThread();
	TThread(const QString &name);
	virtual ~TThread();
	void exit(int returnCode = 0);
	bool isFinished() const;
#if QT_VERSION >= QT_VERSION_CHECK(5, 2, 0)
	bool isInterruptionRequested() const;
	void requestInterruption();
#endif
	bool isRunning() const;
	void setPriority(QThread::Priority priority);
	void setStackSize(uint stackSize);
	uint stackSize();
	void start();
	void terminate();
	bool wait(unsigned long time = ULONG_MAX);
	void quit();
	static void listThreads(QList<QThread*> &list);
protected:
	virtual void run()=0;
private:
	TThread_ *threadPtr;
	static QMap<TThread_*, TThread_*> threadMap;
};

class TThread_ : public QThread {
	Q_OBJECT
public:
	TThread_(TThread *thr, const QString &name);
protected:
	void run();
private:
	TThread *tThread;
	QString threadName;
};

#endif /* TTHREAD_H */
