/*
 * xcolors.c --
 *
 *	This file contains the routines used to map from X color names to RGB
 *	and pixel values.
 *
 * Copyright © 1996 Sun Microsystems, Inc.
 * Copyright © 2012 Jan Nijtmans
 *
 * See the file "license.terms" for information on usage and redistribution of
 * this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include "tkInt.h"

/*
 * Index array. For each of the characters 'a'-'y', this table gives the first
 * color starting with that character in the xColors table.
 */

static const unsigned char az[] = {
    0, 5, 13, 21, 45, 46, 50, 60, 62, 65, 66,
    67, 91, 106, 109, 115, 126, 127, 130, 144, 149, 150, 152, 155, 156, 158
};

/*
 * Define an array that defines the mapping from color names to RGB values.
 * Note that this array must be kept sorted alphabetically so that the
 * binary search used in XParseColor will succeed.
 *
 * Each color definition consists of exactly 32 characters, and starts with
 * the color name, but without its first character (that character can be
 * reproduced from the az index array). The last byte holds the number
 * of additional color variants. For example "azure1" up to "azure4" are
 * handled by the same table entry as "azure". From the last byte backwards,
 * each group of 3 bytes contain the rgb values of the main color and
 * the available variants.
 *
 * The colors gray and grey have more than 8 variants. gray1 up to gray8
 * are handled by this table, above that is handled especially.
 */

typedef char elem[32];

static const elem xColors[] = {
    /* Colors starting with 'a' */
    {'l', 'i', 'c', 'e', 'B', 'l', 'u', 'e', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xF0, 0xF8, 0xFF, 0x00},
    {'n', 't', 'i', 'q', 'u', 'e', 'W', 'h', 'i', 't', 'e', 0, 0, 0, 0, 0,
	    0x8B, 0x83, 0x78, 0xCD, 0xC0, 0xB0, 0xEE, 0xDF, 0xCC, 0xFF, 0xEF, 0xDB, 0xFA, 0xEB, 0xD7, 0x04},
    {'q', 'u', 'a', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x00, 0xFF, 0xFF, 0x00},
    {'q', 'u', 'a', 'm', 'a', 'r', 'i', 'n', 'e', 0, 0, 0, 0, 0, 0, 0,
	    0x45, 0x8B, 0x74, 0x66, 0xCD, 0xAA, 0x76, 0xEE, 0xC6, 0x7F, 0xFF, 0xD4, 0x7F, 0xFF, 0xD4, 0x04},
    {'z', 'u', 'r', 'e', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x83, 0x8B, 0x8B, 0xC1, 0xCD, 0xCD, 0xE0, 0xEE, 0xEE, 0xF0, 0xFF, 0xFF, 0xF0, 0xFF, 0xFF, 0x04},
    /* Colors starting with 'b' */
    {'e', 'i', 'g', 'e', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xF5, 0xF5, 0xDC, 0x00},
    {'i', 's', 'q', 'u', 'e', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x7D, 0x6B, 0xCD, 0xB7, 0x9E, 0xEE, 0xD5, 0xB7, 0xFF, 0xE4, 0xC4, 0xFF, 0xE4, 0xC4, 0x04},
    {'l', 'a', 'c', 'k', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x00, 0x00, 0x00, 0x00},
    {'l', 'a', 'n', 'c', 'h', 'e', 'd', 'A', 'l', 'm', 'o', 'n', 'd', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xFF, 0xEB, 0xCD, 0x00},
    {'l', 'u', 'e', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x00, 0x00, 0x8B, 0x00, 0x00, 0xCD, 0x00, 0x00, 0xEE, 0x00, 0x00, 0xFF, 0x00, 0x00, 0xFF, 0x04},
    {'l', 'u', 'e', 'V', 'i', 'o', 'l', 'e', 't', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8A, 0x2B, 0xE2, 0x00},
    {'r', 'o', 'w', 'n', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x23, 0x23, 0xCD, 0x33, 0x33, 0xEE, 0x3B, 0x3B, 0xFF, 0x40, 0x40, 0xA5, 0x2A, 0x2A, 0x04},
    {'u', 'r', 'l', 'y', 'w', 'o', 'o', 'd', 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x73, 0x55, 0xCD, 0xAA, 0x7D, 0xEE, 0xC5, 0x91, 0xFF, 0xD3, 0x9B, 0xDE, 0xB8, 0x87, 0x04},
    /* Colors starting with 'c' */
    {'a', 'd', 'e', 't', 'B', 'l', 'u', 'e', 0, 0, 0, 0, 0, 0, 0, 0,
	    0x53, 0x86, 0x8B, 0x7A, 0xC5, 0xCD, 0x8E, 0xE5, 0xEE, 0x98, 0xF5, 0xFF, 0x5F, 0x9E, 0xA0, 0x04},
    {'h', 'a', 'r', 't', 'r', 'e', 'u', 's', 'e', 0, 0, 0, 0, 0, 0, 0,
	    0x45, 0x8B, 0x00, 0x66, 0xCD, 0x00, 0x76, 0xEE, 0x00, 0x7F, 0xFF, 0x00, 0x7F, 0xFF, 0x00, 0x04},
    {'h', 'o', 'c', 'o', 'l', 'a', 't', 'e', 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x45, 0x13, 0xCD, 0x66, 0x1D, 0xEE, 0x76, 0x21, 0xFF, 0x7F, 0x24, 0xD2, 0x69, 0x1E, 0x04},
    {'o', 'r', 'a', 'l', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x3E, 0x2F, 0xCD, 0x5B, 0x45, 0xEE, 0x6A, 0x50, 0xFF, 0x72, 0x56, 0xFF, 0x7F, 0x50, 0x04},
    {'o', 'r', 'n', 'f', 'l', 'o', 'w', 'e', 'r', 'B', 'l', 'u', 'e', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x64, 0x95, 0xED, 0x00},
    {'o', 'r', 'n', 's', 'i', 'l', 'k', 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x88, 0x78, 0xCD, 0xC8, 0xB1, 0xEE, 0xE8, 0xCD, 0xFF, 0xF8, 0xDC, 0xFF, 0xF8, 0xDC, 0x04},
    {'r', 'i', 'm', 's', 'o', 'n', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xDC, 0x14, 0x3C, 0x00},
    {'y', 'a', 'n', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x00, 0x8B, 0x8B, 0x00, 0xCD, 0xCD, 0x00, 0xEE, 0xEE, 0x00, 0xFF, 0xFF, 0x00, 0xFF, 0xFF, 0x04},
    /* Colors starting with 'd' */
    {'a', 'r', 'k', 'B', 'l', 'u', 'e', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x00, 0x00, 0x8B, 0x00},
    {'a', 'r', 'k', 'C', 'y', 'a', 'n', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x00, 0x8B, 0x8B, 0x00},
    {'a', 'r', 'k', 'G', 'o', 'l', 'd', 'e', 'n', 'r', 'o', 'd', 0, 0, 0, 0,
	    0x8B, 0x65, 0x08, 0xCD, 0x95, 0x0C, 0xEE, 0xAD, 0x0E, 0xFF, 0xB9, 0x0F, 0xB8, 0x86, 0x0B, 0x04},
    {'a', 'r', 'k', 'G', 'r', 'a', 'y', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xA9, 0xA9, 0xA9, 0x00},
    {'a', 'r', 'k', 'G', 'r', 'e', 'e', 'n', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x00, 0x64, 0x00, 0x00},
    {'a', 'r', 'k', 'G', 'r', 'e', 'y', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xA9, 0xA9, 0xA9, 0x00},
    {'a', 'r', 'k', 'K', 'h', 'a', 'k', 'i', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xBD, 0xB7, 0x6B, 0x00},
    {'a', 'r', 'k', 'M', 'a', 'g', 'e', 'n', 't', 'a', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x00, 0x8B, 0x00},
    {'a', 'r', 'k', 'O', 'l', 'i', 'v', 'e', 'G', 'r', 'e', 'e', 'n', 0, 0, 0,
	    0x6E, 0x8B, 0x3D, 0xA2, 0xCD, 0x5A, 0xBC, 0xEE, 0x68, 0xCA, 0xFF, 0x70, 0x55, 0x6B, 0x2F, 0x04},
    {'a', 'r', 'k', 'O', 'r', 'a', 'n', 'g', 'e', 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x45, 0x00, 0xCD, 0x66, 0x00, 0xEE, 0x76, 0x00, 0xFF, 0x7F, 0x00, 0xFF, 0x8C, 0x00, 0x04},
    {'a', 'r', 'k', 'O', 'r', 'c', 'h', 'i', 'd', 0, 0, 0, 0, 0, 0, 0,
	    0x68, 0x22, 0x8B, 0x9A, 0x32, 0xCD, 0xB2, 0x3A, 0xEE, 0xBF, 0x3E, 0xFF, 0x99, 0x32, 0xCC, 0x04},
    {'a', 'r', 'k', 'R', 'e', 'd', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x00, 0x00, 0x00},
    {'a', 'r', 'k', 'S', 'a', 'l', 'm', 'o', 'n', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xE9, 0x96, 0x7A, 0x00},
    {'a', 'r', 'k', 'S', 'e', 'a', 'G', 'r', 'e', 'e', 'n', 0, 0, 0, 0, 0,
	    0x69, 0x8B, 0x69, 0x9B, 0xCD, 0x9B, 0xB4, 0xEE, 0xB4, 0xC1, 0xFF, 0xC1, 0x8F, 0xBC, 0x8F, 0x04},
    {'a', 'r', 'k', 'S', 'l', 'a', 't', 'e', 'B', 'l', 'u', 'e', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x48, 0x3D, 0x8B, 0x00},
    {'a', 'r', 'k', 'S', 'l', 'a', 't', 'e', 'G', 'r', 'a', 'y', 0, 0, 0, 0,
	    0x52, 0x8B, 0x8B, 0x79, 0xCD, 0xCD, 0x8D, 0xEE, 0xEE, 0x97, 0xFF, 0xFF, 0x2F, 0x4F, 0x4F, 0x04},
    {'a', 'r', 'k', 'S', 'l', 'a', 't', 'e', 'G', 'r', 'e', 'y', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x2F, 0x4F, 0x4F, 0x00},
    {'a', 'r', 'k', 'T', 'u', 'r', 'q', 'u', 'o', 'i', 's', 'e', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x00, 0xCE, 0xD1, 0x00},
    {'a', 'r', 'k', 'V', 'i', 'o', 'l', 'e', 't', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x94, 0x00, 0xD3, 0x00},
    {'e', 'e', 'p', 'P', 'i', 'n', 'k', 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x0A, 0x50, 0xCD, 0x10, 0x76, 0xEE, 0x12, 0x89, 0xFF, 0x14, 0x93, 0xFF, 0x14, 0x93, 0x04},
    {'e', 'e', 'p', 'S', 'k', 'y', 'B', 'l', 'u', 'e', 0, 0, 0, 0, 0, 0,
	    0x00, 0x68, 0x8B, 0x00, 0x9A, 0xCD, 0x00, 0xB2, 0xEE, 0x00, 0xBF, 0xFF, 0x00, 0xBF, 0xFF, 0x04},
    {'i', 'm', 'G', 'r', 'a', 'y', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x69, 0x69, 0x69, 0x00},
    {'i', 'm', 'G', 'r', 'e', 'y', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x69, 0x69, 0x69, 0x00},
    {'o', 'd', 'g', 'e', 'r', 'B', 'l', 'u', 'e', 0, 0, 0, 0, 0, 0, 0,
	    0x10, 0x4E, 0x8B, 0x18, 0x74, 0xCD, 0x1C, 0x86, 0xEE, 0x1E, 0x90, 0xFF, 0x1E, 0x90, 0xFF, 0x04},
    /* Colors starting with 'e' (placeholder) */
    {0xFF, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
    /* Colors starting with 'f' */
    {'i', 'r', 'e', 'b', 'r', 'i', 'c', 'k', 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x1A, 0x1A, 0xCD, 0x26, 0x26, 0xEE, 0x2C, 0x2C, 0xFF, 0x30, 0x30, 0xB2, 0x22, 0x22, 0x04},
    {'l', 'o', 'r', 'a', 'l', 'W', 'h', 'i', 't', 'e', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xFF, 0xFA, 0xF0, 0x00},
    {'o', 'r', 'e', 's', 't', 'G', 'r', 'e', 'e', 'n', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x22, 0x8B, 0x22, 0x00},
    {'u', 'c', 'h', 's', 'i', 'a', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xFF, 0x00, 0xFF, 0x00},
    /* Colors starting with 'g' */
    {'a', 'i', 'n', 's', 'b', 'o', 'r', 'o', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xDC, 0xDC, 0xDC, 0x00},
    {'h', 'o', 's', 't', 'W', 'h', 'i', 't', 'e', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xF8, 0xF8, 0xFF, 0x00},
    {'o', 'l', 'd', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x75, 0x00, 0xCD, 0xAD, 0x00, 0xEE, 0xC9, 0x00, 0xFF, 0xD7, 0x00, 0xFF, 0xD7, 0x00, 0x04},
    {'o', 'l', 'd', 'e', 'n', 'r', 'o', 'd', 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x69, 0x14, 0xCD, 0x9B, 0x1D, 0xEE, 0xB4, 0x22, 0xFF, 0xC1, 0x25, 0xDA, 0xA5, 0x20, 0x04},
    {'r', 'a', 'y', 0,
	    0x14, 0x14, 0x14, 0x12, 0x12, 0x12, 0x0F, 0x0F, 0x0F, 0x0D, 0x0D, 0x0D, 0x0A, 0x0A, 0x0A, 0x08, 0x08, 0x08, 0x05, 0x05, 0x05, 0x03, 0x03, 0x03, 0x80, 0x80, 0x80, 0x08},
    {'r', 'a', 'y', '0', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x00, 0x00, 0x00, 0x00},
    {'r', 'e', 'e', 'n', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x00, 0x8B, 0x00, 0x00, 0xCD, 0x00, 0x00, 0xEE, 0x00, 0x00, 0xFF, 0x00, 0x00, 0x80, 0x00, 0x04},
    {'r', 'e', 'e', 'n', 'Y', 'e', 'l', 'l', 'o', 'w', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xAD, 0xFF, 0x2F, 0x00},
    {'r', 'e', 'y', 0,
	    0x14, 0x14, 0x14, 0x12, 0x12, 0x12, 0x0F, 0x0F, 0x0F, 0x0D, 0x0D, 0x0D, 0x0A, 0x0A, 0x0A, 0x08, 0x08, 0x08, 0x05, 0x05, 0x05, 0x03, 0x03, 0x03, 0x80, 0x80, 0x80, 0x08},
    {'r', 'e', 'y', '0', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x00, 0x00, 0x00, 0x00},
    /* Colors starting with 'h' */
    {'o', 'n', 'e', 'y', 'd', 'e', 'w', 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x83, 0x8B, 0x83, 0xC1, 0xCD, 0xC1, 0xE0, 0xEE, 0xE0, 0xF0, 0xFF, 0xF0, 0xF0, 0xFF, 0xF0, 0x04},
    {'o', 't', 'P', 'i', 'n', 'k', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x3A, 0x62, 0xCD, 0x60, 0x90, 0xEE, 0x6A, 0xA7, 0xFF, 0x6E, 0xB4, 0xFF, 0x69, 0xB4, 0x04},
    /* Colors starting with 'i' */
    {'n', 'd', 'i', 'a', 'n', 'R', 'e', 'd', 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x3A, 0x3A, 0xCD, 0x55, 0x55, 0xEE, 0x63, 0x63, 0xFF, 0x6A, 0x6A, 0xCD, 0x5C, 0x5C, 0x04},
    {'n', 'd', 'i', 'g', 'o', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x4B, 0x00, 0x82, 0x00},
    {'v', 'o', 'r', 'y', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x8B, 0x83, 0xCD, 0xCD, 0xC1, 0xEE, 0xEE, 0xE0, 0xFF, 0xFF, 0xF0, 0xFF, 0xFF, 0xF0, 0x04},
    /* Colors starting with 'j' (placeholder) */
    {0xFF, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
    /* Colors starting with 'k' */
    {'h', 'a', 'k', 'i', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x86, 0x4E, 0xCD, 0xC6, 0x73, 0xEE, 0xE6, 0x85, 0xFF, 0xF6, 0x8F, 0xF0, 0xE6, 0x8C, 0x04},
    /* Colors starting with 'l' */
    {'a', 'v', 'e', 'n', 'd', 'e', 'r', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xE6, 0xE6, 0xFA, 0x00},
    {'a', 'v', 'e', 'n', 'd', 'e', 'r', 'B', 'l', 'u', 's', 'h', 0, 0, 0, 0,
	    0x8B, 0x83, 0x86, 0xCD, 0xC1, 0xC5, 0xEE, 0xE0, 0xE5, 0xFF, 0xF0, 0xF5, 0xFF, 0xF0, 0xF5, 0x04},
    {'a', 'w', 'n', 'G', 'r', 'e', 'e', 'n', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x7C, 0xFC, 0x00, 0x00},
    {'e', 'm', 'o', 'n', 'C', 'h', 'i', 'f', 'f', 'o', 'n', 0, 0, 0, 0, 0,
	    0x8B, 0x89, 0x70, 0xCD, 0xC9, 0xA5, 0xEE, 0xE9, 0xBF, 0xFF, 0xFA, 0xCD, 0xFF, 0xFA, 0xCD, 0x04},
    {'i', 'g', 'h', 't', 'B', 'l', 'u', 'e', 0, 0, 0, 0, 0, 0, 0, 0,
	    0x68, 0x83, 0x8B, 0x9A, 0xC0, 0xCD, 0xB2, 0xDF, 0xEE, 0xBF, 0xEF, 0xFF, 0xAD, 0xD8, 0xE6, 0x04},
    {'i', 'g', 'h', 't', 'C', 'o', 'r', 'a', 'l', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xF0, 0x80, 0x80, 0x00},
    {'i', 'g', 'h', 't', 'C', 'y', 'a', 'n', 0, 0, 0, 0, 0, 0, 0, 0,
	    0x7A, 0x8B, 0x8B, 0xB4, 0xCD, 0xCD, 0xD1, 0xEE, 0xEE, 0xE0, 0xFF, 0xFF, 0xE0, 0xFF, 0xFF, 0x04},
    {'i', 'g', 'h', 't', 'G', 'o', 'l', 'd', 'e', 'n', 'r', 'o', 'd', 0, 0, 0,
	    0x8B, 0x81, 0x4C, 0xCD, 0xBE, 0x70, 0xEE, 0xDC, 0x82, 0xFF, 0xEC, 0x8B, 0xEE, 0xDD, 0x82, 0x04},
    {'i', 'g', 'h', 't', 'G', 'o', 'l', 'd', 'e', 'n', 'r', 'o', 'd', 'Y', 'e', 'l', 'l', 'o', 'w', 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xFA, 0xFA, 0xD2, 0x00},
    {'i', 'g', 'h', 't', 'G', 'r', 'a', 'y', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xD3, 0xD3, 0xD3, 0x00},
    {'i', 'g', 'h', 't', 'G', 'r', 'e', 'e', 'n', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x90, 0xEE, 0x90, 0x00},
    {'i', 'g', 'h', 't', 'G', 'r', 'e', 'y', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xD3, 0xD3, 0xD3, 0x00},
    {'i', 'g', 'h', 't', 'P', 'i', 'n', 'k', 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x5F, 0x65, 0xCD, 0x8C, 0x95, 0xEE, 0xA2, 0xAD, 0xFF, 0xAE, 0xB9, 0xFF, 0xB6, 0xC1, 0x04},
    {'i', 'g', 'h', 't', 'S', 'a', 'l', 'm', 'o', 'n', 0, 0, 0, 0, 0, 0,
	    0x8B, 0x57, 0x42, 0xCD, 0x81, 0x62, 0xEE, 0x95, 0x72, 0xFF, 0xA0, 0x7A, 0xFF, 0xA0, 0x7A, 0x04},
    {'i', 'g', 'h', 't', 'S', 'e', 'a', 'G', 'r', 'e', 'e', 'n', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x20, 0xB2, 0xAA, 0x00},
    {'i', 'g', 'h', 't', 'S', 'k', 'y', 'B', 'l', 'u', 'e', 0, 0, 0, 0, 0,
	    0x60, 0x7B, 0x8B, 0x8D, 0xB6, 0xCD, 0xA4, 0xD3, 0xEE, 0xB0, 0xE2, 0xFF, 0x87, 0xCE, 0xFA, 0x04},
    {'i', 'g', 'h', 't', 'S', 'l', 'a', 't', 'e', 'B', 'l', 'u', 'e', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x84, 0x70, 0xFF, 0x00},
    {'i', 'g', 'h', 't', 'S', 'l', 'a', 't', 'e', 'G', 'r', 'a', 'y', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x77, 0x88, 0x99, 0x00},
    {'i', 'g', 'h', 't', 'S', 'l', 'a', 't', 'e', 'G', 'r', 'e', 'y', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x77, 0x88, 0x99, 0x00},
    {'i', 'g', 'h', 't', 'S', 't', 'e', 'e', 'l', 'B', 'l', 'u', 'e', 0, 0, 0,
	    0x6E, 0x7B, 0x8B, 0xA2, 0xB5, 0xCD, 0xBC, 0xD2, 0xEE, 0xCA, 0xE1, 0xFF, 0xB0, 0xC4, 0xDE, 0x04},
    {'i', 'g', 'h', 't', 'Y', 'e', 'l', 'l', 'o', 'w', 0, 0, 0, 0, 0, 0,
	    0x8B, 0x8B, 0x7A, 0xCD, 0xCD, 0xB4, 0xEE, 0xEE, 0xD1, 0xFF, 0xFF, 0xE0, 0xFF, 0xFF, 0xE0, 0x04},
    {'i', 'm', 'e', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x00, 0xFF, 0x00, 0x00},
    {'i', 'm', 'e', 'G', 'r', 'e', 'e', 'n', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x32, 0xCD, 0x32, 0x00},
    {'i', 'n', 'e', 'n', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xFA, 0xF0, 0xE6, 0x00},
    /* Colors starting with 'm' */
    {'a', 'g', 'e', 'n', 't', 'a', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x00, 0x8B, 0xCD, 0x00, 0xCD, 0xEE, 0x00, 0xEE, 0xFF, 0x00, 0xFF, 0xFF, 0x00, 0xFF, 0x04},
    {'a', 'r', 'o', 'o', 'n', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x1C, 0x62, 0xCD, 0x29, 0x90, 0xEE, 0x30, 0xA7, 0xFF, 0x34, 0xB3, 0x80, 0x00, 0x00, 0x04},
    {'e', 'd', 'i', 'u', 'm', 'A', 'q', 'u', 'a', 'm', 'a', 'r', 'i', 'n', 'e', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x66, 0xCD, 0xAA, 0x00},
    {'e', 'd', 'i', 'u', 'm', 'B', 'l', 'u', 'e', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x00, 0x00, 0xCD, 0x00},
    {'e', 'd', 'i', 'u', 'm', 'O', 'r', 'c', 'h', 'i', 'd', 0, 0, 0, 0, 0,
	    0x7A, 0x37, 0x8B, 0xB4, 0x52, 0xCD, 0xD1, 0x5F, 0xEE, 0xE0, 0x66, 0xFF, 0xBA, 0x55, 0xD3, 0x04},
    {'e', 'd', 'i', 'u', 'm', 'P', 'u', 'r', 'p', 'l', 'e', 0, 0, 0, 0, 0,
	    0x5D, 0x47, 0x8B, 0x89, 0x68, 0xCD, 0x9F, 0x79, 0xEE, 0xAB, 0x82, 0xFF, 0x93, 0x70, 0xDB, 0x04},
    {'e', 'd', 'i', 'u', 'm', 'S', 'e', 'a', 'G', 'r', 'e', 'e', 'n', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x3C, 0xB3, 0x71, 0x00},
    {'e', 'd', 'i', 'u', 'm', 'S', 'l', 'a', 't', 'e', 'B', 'l', 'u', 'e', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x7B, 0x68, 0xEE, 0x00},
    {'e', 'd', 'i', 'u', 'm', 'S', 'p', 'r', 'i', 'n', 'g', 'G', 'r', 'e', 'e', 'n', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x00, 0xFA, 0x9A, 0x00},
    {'e', 'd', 'i', 'u', 'm', 'T', 'u', 'r', 'q', 'u', 'o', 'i', 's', 'e', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x48, 0xD1, 0xCC, 0x00},
    {'e', 'd', 'i', 'u', 'm', 'V', 'i', 'o', 'l', 'e', 't', 'R', 'e', 'd', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xC7, 0x15, 0x85, 0x00},
    {'i', 'd', 'n', 'i', 'g', 'h', 't', 'B', 'l', 'u', 'e', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x19, 0x19, 0x70, 0x00},
    {'i', 'n', 't', 'C', 'r', 'e', 'a', 'm', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xF5, 0xFF, 0xFA, 0x00},
    {'i', 's', 't', 'y', 'R', 'o', 's', 'e', 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x7D, 0x7B, 0xCD, 0xB7, 0xB5, 0xEE, 0xD5, 0xD2, 0xFF, 0xE4, 0xE1, 0xFF, 0xE4, 0xE1, 0x04},
    {'o', 'c', 'c', 'a', 's', 'i', 'n', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xFF, 0xE4, 0xB5, 0x00},
    /* Colors starting with 'n' */
    {'a', 'v', 'a', 'j', 'o', 'W', 'h', 'i', 't', 'e', 0, 0, 0, 0, 0, 0,
	    0x8B, 0x79, 0x5E, 0xCD, 0xB3, 0x8B, 0xEE, 0xCF, 0xA1, 0xFF, 0xDE, 0xAD, 0xFF, 0xDE, 0xAD, 0x04},
    {'a', 'v', 'y', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x00, 0x00, 0x80, 0x00},
    {'a', 'v', 'y', 'B', 'l', 'u', 'e', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x00, 0x00, 0x80, 0x00},
    /* Colors starting with 'o' */
    {'l', 'd', 'L', 'a', 'c', 'e', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xFD, 0xF5, 0xE6, 0x00},
    {'l', 'i', 'v', 'e', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x80, 0x80, 0x00, 0x00},
    {'l', 'i', 'v', 'e', 'D', 'r', 'a', 'b', 0, 0, 0, 0, 0, 0, 0, 0,
	    0x69, 0x8B, 0x22, 0x9A, 0xCD, 0x32, 0xB3, 0xEE, 0x3A, 0xC0, 0xFF, 0x3E, 0x6B, 0x8E, 0x23, 0x04},
    {'r', 'a', 'n', 'g', 'e', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x5A, 0x00, 0xCD, 0x85, 0x00, 0xEE, 0x9A, 0x00, 0xFF, 0xA5, 0x00, 0xFF, 0xA5, 0x00, 0x04},
    {'r', 'a', 'n', 'g', 'e', 'R', 'e', 'd', 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x25, 0x00, 0xCD, 0x37, 0x00, 0xEE, 0x40, 0x00, 0xFF, 0x45, 0x00, 0xFF, 0x45, 0x00, 0x04},
    {'r', 'c', 'h', 'i', 'd', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x47, 0x89, 0xCD, 0x69, 0xC9, 0xEE, 0x7A, 0xE9, 0xFF, 0x83, 0xFA, 0xDA, 0x70, 0xD6, 0x04},
    /* Colors starting with 'p' */
    {'a', 'l', 'e', 'G', 'o', 'l', 'd', 'e', 'n', 'r', 'o', 'd', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xEE, 0xE8, 0xAA, 0x00},
    {'a', 'l', 'e', 'G', 'r', 'e', 'e', 'n', 0, 0, 0, 0, 0, 0, 0, 0,
	    0x54, 0x8B, 0x54, 0x7C, 0xCD, 0x7C, 0x90, 0xEE, 0x90, 0x9A, 0xFF, 0x9A, 0x98, 0xFB, 0x98, 0x04},
    {'a', 'l', 'e', 'T', 'u', 'r', 'q', 'u', 'o', 'i', 's', 'e', 0, 0, 0, 0,
	    0x66, 0x8B, 0x8B, 0x96, 0xCD, 0xCD, 0xAE, 0xEE, 0xEE, 0xBB, 0xFF, 0xFF, 0xAF, 0xEE, 0xEE, 0x04},
    {'a', 'l', 'e', 'V', 'i', 'o', 'l', 'e', 't', 'R', 'e', 'd', 0, 0, 0, 0,
	    0x8B, 0x47, 0x5D, 0xCD, 0x68, 0x89, 0xEE, 0x79, 0x9F, 0xFF, 0x82, 0xAB, 0xDB, 0x70, 0x93, 0x04},
    {'a', 'p', 'a', 'y', 'a', 'W', 'h', 'i', 'p', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xFF, 0xEF, 0xD5, 0x00},
    {'e', 'a', 'c', 'h', 'P', 'u', 'f', 'f', 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x77, 0x65, 0xCD, 0xAF, 0x95, 0xEE, 0xCB, 0xAD, 0xFF, 0xDA, 0xB9, 0xFF, 0xDA, 0xB9, 0x04},
    {'e', 'r', 'u', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xCD, 0x85, 0x3F, 0x00},
    {'i', 'n', 'k', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x63, 0x6C, 0xCD, 0x91, 0x9E, 0xEE, 0xA9, 0xB8, 0xFF, 0xB5, 0xC5, 0xFF, 0xC0, 0xCB, 0x04},
    {'l', 'u', 'm', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x66, 0x8B, 0xCD, 0x96, 0xCD, 0xEE, 0xAE, 0xEE, 0xFF, 0xBB, 0xFF, 0xDD, 0xA0, 0xDD, 0x04},
    {'o', 'w', 'd', 'e', 'r', 'B', 'l', 'u', 'e', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xB0, 0xE0, 0xE6, 0x00},
    {'u', 'r', 'p', 'l', 'e', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x55, 0x1A, 0x8B, 0x7D, 0x26, 0xCD, 0x91, 0x2C, 0xEE, 0x9B, 0x30, 0xFF, 0x80, 0x00, 0x80, 0x04},
    /* Colors starting with 'q' (placeholder)*/
    {0xFF, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
    /* Colors starting with 'r' */
    {'e', 'd', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x00, 0x00, 0xCD, 0x00, 0x00, 0xEE, 0x00, 0x00, 0xFF, 0x00, 0x00, 0xFF, 0x00, 0x00, 0x04},
    {'o', 's', 'y', 'B', 'r', 'o', 'w', 'n', 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x69, 0x69, 0xCD, 0x9B, 0x9B, 0xEE, 0xB4, 0xB4, 0xFF, 0xC1, 0xC1, 0xBC, 0x8F, 0x8F, 0x04},
    {'o', 'y', 'a', 'l', 'B', 'l', 'u', 'e', 0, 0, 0, 0, 0, 0, 0, 0,
	    0x27, 0x40, 0x8B, 0x3A, 0x5F, 0xCD, 0x43, 0x6E, 0xEE, 0x48, 0x76, 0xFF, 0x41, 0x69, 0xE1, 0x04},
    /* Colors starting with 's' */
    {'a', 'd', 'd', 'l', 'e', 'B', 'r', 'o', 'w', 'n', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x45, 0x13, 0x00},
    {'a', 'l', 'm', 'o', 'n', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x4C, 0x39, 0xCD, 0x70, 0x54, 0xEE, 0x82, 0x62, 0xFF, 0x8C, 0x69, 0xFA, 0x80, 0x72, 0x04},
    {'a', 'n', 'd', 'y', 'B', 'r', 'o', 'w', 'n', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xF4, 0xA4, 0x60, 0x00},
    {'e', 'a', 'G', 'r', 'e', 'e', 'n', 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x2E, 0x8B, 0x57, 0x43, 0xCD, 0x80, 0x4E, 0xEE, 0x94, 0x54, 0xFF, 0x9F, 0x2E, 0x8B, 0x57, 0x04},
    {'e', 'a', 's', 'h', 'e', 'l', 'l', 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x86, 0x82, 0xCD, 0xC5, 0xBF, 0xEE, 0xE5, 0xDE, 0xFF, 0xF5, 0xEE, 0xFF, 0xF5, 0xEE, 0x04},
    {'i', 'e', 'n', 'n', 'a', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x47, 0x26, 0xCD, 0x68, 0x39, 0xEE, 0x79, 0x42, 0xFF, 0x82, 0x47, 0xA0, 0x52, 0x2D, 0x04},
    {'i', 'l', 'v', 'e', 'r', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xC0, 0xC0, 0xC0, 0x00},
    {'k', 'y', 'B', 'l', 'u', 'e', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x4A, 0x70, 0x8B, 0x6C, 0xA6, 0xCD, 0x7E, 0xC0, 0xEE, 0x87, 0xCE, 0xFF, 0x87, 0xCE, 0xEB, 0x04},
    {'l', 'a', 't', 'e', 'B', 'l', 'u', 'e', 0, 0, 0, 0, 0, 0, 0, 0,
	    0x47, 0x3C, 0x8B, 0x69, 0x59, 0xCD, 0x7A, 0x67, 0xEE, 0x83, 0x6F, 0xFF, 0x6A, 0x5A, 0xCD, 0x04},
    {'l', 'a', 't', 'e', 'G', 'r', 'a', 'y', 0, 0, 0, 0, 0, 0, 0, 0,
	    0x6C, 0x7B, 0x8B, 0x9F, 0xB6, 0xCD, 0xB9, 0xD3, 0xEE, 0xC6, 0xE2, 0xFF, 0x70, 0x80, 0x90, 0x04},
    {'l', 'a', 't', 'e', 'G', 'r', 'e', 'y', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x70, 0x80, 0x90, 0x00},
    {'n', 'o', 'w', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x89, 0x89, 0xCD, 0xC9, 0xC9, 0xEE, 0xE9, 0xE9, 0xFF, 0xFA, 0xFA, 0xFF, 0xFA, 0xFA, 0x04},
    {'p', 'r', 'i', 'n', 'g', 'G', 'r', 'e', 'e', 'n', 0, 0, 0, 0, 0, 0,
	    0x00, 0x8B, 0x45, 0x00, 0xCD, 0x66, 0x00, 0xEE, 0x76, 0x00, 0xFF, 0x7F, 0x00, 0xFF, 0x7F, 0x04},
    {'t', 'e', 'e', 'l', 'B', 'l', 'u', 'e', 0, 0, 0, 0, 0, 0, 0, 0,
	    0x36, 0x64, 0x8B, 0x4F, 0x94, 0xCD, 0x5C, 0xAC, 0xEE, 0x63, 0xB8, 0xFF, 0x46, 0x82, 0xB4, 0x04},
    /* Colors starting with 't' */
    {'a', 'n', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x5A, 0x2B, 0xCD, 0x85, 0x3F, 0xEE, 0x9A, 0x49, 0xFF, 0xA5, 0x4F, 0xD2, 0xB4, 0x8C, 0x04},
    {'e', 'a', 'l', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x00, 0x80, 0x80, 0x00},
    {'h', 'i', 's', 't', 'l', 'e', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x7B, 0x8B, 0xCD, 0xB5, 0xCD, 0xEE, 0xD2, 0xEE, 0xFF, 0xE1, 0xFF, 0xD8, 0xBF, 0xD8, 0x04},
    {'o', 'm', 'a', 't', 'o', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x36, 0x26, 0xCD, 0x4F, 0x39, 0xEE, 0x5C, 0x42, 0xFF, 0x63, 0x47, 0xFF, 0x63, 0x47, 0x04},
    {'u', 'r', 'q', 'u', 'o', 'i', 's', 'e', 0, 0, 0, 0, 0, 0, 0, 0,
	    0x00, 0x86, 0x8B, 0x00, 0xC5, 0xCD, 0x00, 0xE5, 0xEE, 0x00, 0xF5, 0xFF, 0x40, 0xE0, 0xD0, 0x04},
    /* Colors starting with 'u' (placeholder) */
    {0xFF, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
    /* Colors starting with 'v' */
    {'i', 'o', 'l', 'e', 't', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xEE, 0x82, 0xEE, 0x00},
    {'i', 'o', 'l', 'e', 't', 'R', 'e', 'd', 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x22, 0x52, 0xCD, 0x32, 0x78, 0xEE, 0x3A, 0x8C, 0xFF, 0x3E, 0x96, 0xD0, 0x20, 0x90, 0x04},
    /* Colors starting with 'w' */
    {'h', 'e', 'a', 't', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x7E, 0x66, 0xCD, 0xBA, 0x96, 0xEE, 0xD8, 0xAE, 0xFF, 0xE7, 0xBA, 0xF5, 0xDE, 0xB3, 0x04},
    {'h', 'i', 't', 'e', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xFF, 0xFF, 0xFF, 0x00},
    {'h', 'i', 't', 'e', 'S', 'm', 'o', 'k', 'e', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0xF5, 0xF5, 0xF5, 0x00},
    /* Colors starting with 'x' (placeholder) */
    {0xFF, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
    /* Colors starting with 'y' */
    {'e', 'l', 'l', 'o', 'w', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x8B, 0x8B, 0x00, 0xCD, 0xCD, 0x00, 0xEE, 0xEE, 0x00, 0xFF, 0xFF, 0x00, 0xFF, 0xFF, 0x00, 0x04},
    {'e', 'l', 'l', 'o', 'w', 'G', 'r', 'e', 'e', 'n', 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	    0x9A, 0xCD, 0x32, 0x00}
};

/*
 *----------------------------------------------------------------------
 *
 * XParseColor --
 *
 *	Partial implementation of X color name parsing interface.
 *
 * Results:
 *	Returns non-zero on success.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 *
 * This only handles hex-strings without 0x prefix. Luckily, that's just what
 * we need.
 */

static Tcl_WideInt
parseHex64bit(
    const char *spec,
    char **p)
{
    Tcl_WideInt result = 0;
    char c;
    while ((c = *spec)) {
	if ((c >= '0') && (c <= '9')) {
	    c -= '0';
	} else if ((c >= 'A') && (c <= 'F')) {
	    c += (10 - 'A');
	} else if ((c >= 'a') && (c <= 'f')) {
	    c += (10 - 'a');
	} else {
	    break;
	}
	result = (result << 4) + c;
	++spec;
    }
    *p = (char *) spec;
    return result;
}

static int
colorcmp(
    const char *spec,
    const char *pname,
    int *special)
{
    int r;
    int c, d;
    int notequal = 0;
    int num = 0;

    do {
	d = *pname++;
	c = (*spec == ' ');
	if (c) {
	    spec++;
	}
	if ((unsigned)(d - 'A') <= (unsigned)('Z' - 'A')) {
	    d += 'a' - 'A';
	} else if (c) {
	    /*
	     * A space doesn't match a lowercase, but we don't know yet
	     * whether we should return a negative or positive number. That
	     * depends on what follows.
	     */

	    notequal = 1;
	}
	c = *spec++;
	if ((unsigned)(c - 'A') <= (unsigned)('Z' - 'A')) {
	    c += 'a' - 'A';
	} else if (((unsigned)(c - '1') <= (unsigned)('9' - '1'))) {
	    if (d == '0') {
		d += 10;
	    } else if (!d) {
		num = c - '0';
		while ((unsigned)((c = *spec++) - '0') <= (unsigned)('9' - '0')) {
		    num = num * 10 + c - '0';
		}
	    }
	}
	r = c - d;
    } while (!r && d);

    if (!r && notequal) {
	/*
	 * Strings are equal, but difference in spacings only. We should still
	 * report not-equal, so "burly wood" is not a valid color.
	 */

	r = 1;
    }
    *special = num;
    return r;
}

#define RED(p)		((unsigned char) (p)[0])
#define GREEN(p)	((unsigned char) (p)[1])
#define BLUE(p)		((unsigned char) (p)[2])
#define US(expr)	((unsigned short) (expr))

Status
XParseColor(
    TCL_UNUSED(Display *),
    TCL_UNUSED(Colormap),
    const char *spec,
    XColor *colorPtr)
{
    if (spec[0] == '#') {
	char *p;
	Tcl_WideInt value = parseHex64bit(++spec, &p);

	/*
	 * If *p does not point to the end of the string, there were invalid
	 * digits in the spec. Ergo, it is not a valid color string.
	 * (Bug f0188aca9e)
	 */

	if (*p != '\0') {
	    return 0;
	}

	switch ((int)(p-spec)) {
	case 3:
	    colorPtr->red = US(((value >> 8) & 0xf) * 0x1111);
	    colorPtr->green = US(((value >> 4) & 0xf) * 0x1111);
	    colorPtr->blue = US((value & 0xf) * 0x1111);
	    break;
	case 6:
	    colorPtr->red = US(((value >> 16) & 0xff) | ((value >> 8) & 0xff00));
	    colorPtr->green = US(((value >> 8) & 0xff) | (value & 0xff00));
	    colorPtr->blue = US((value & 0xff) | (value << 8));
	    break;
	case 9:
	    colorPtr->red = US(((value >> 32) & 0xf) | ((value >> 20) & 0xfff0));
	    colorPtr->green = US(((value >> 20) & 0xf) | ((value >> 8) & 0xfff0));
	    colorPtr->blue = US(((value >> 8) & 0xf) | (value << 4));
	    break;
	case 12:
	    colorPtr->red = US(value >> 32);
	    colorPtr->green = US(value >> 16);
	    colorPtr->blue = US(value);
	    break;
	default:
	    return 0;
	}
    } else {
	/*
	 * Perform a binary search on the sorted array of colors.
	 * size = current size of search range
	 * p    = pointer to current element being considered.
	 */

	int size, num;
	const elem *p;
	const char *q;
	int r = (spec[0] - 'A') & 0xdf;

	if (r >= (int) sizeof(az) - 1) {
	    return 0;
	}
	size = az[r + 1] - az[r];
	p = &xColors[(az[r + 1] + az[r]) >> 1];
	r = colorcmp(spec + 1, *p, &num);

	while (r != 0) {
	    if (r < 0) {
		size = (size >> 1);
		p -= ((size + 1) >> 1);
	    } else {
		--size;
		size = (size >> 1);
		p += ((size + 2) >> 1);
	    }
	    if (!size) {
		return 0;
	    }
	    r = colorcmp(spec + 1, *p, &num);
	}
	if (num > (*p)[31]) {
	    if (((*p)[31] != 8) || num > 100) {
		return 0;
	    }
	    num = (num * 255 + 50) / 100;
	    if ((num == 230) || (num == 128)) {
		/*
		 * Those two entries have a deviation i.r.t the table.
		 */

		num--;
	    }
	    num |= (num << 8);
	    colorPtr->red = colorPtr->green = colorPtr->blue = num;
	} else {
	    q = *p + 28 - num * 3;
	    colorPtr->red = ((RED(q) << 8) | RED(q));
	    colorPtr->green = ((GREEN(q) << 8) | GREEN(q));
	    colorPtr->blue = ((BLUE(q) << 8) | BLUE(q));
	}
    }
    colorPtr->pixel = TkpGetPixel(colorPtr);
    colorPtr->flags = DoRed|DoGreen|DoBlue;
    colorPtr->pad = 0;
    return 1;
}

/*
 * Local Variables:
 * mode: c
 * c-basic-offset: 4
 * fill-column: 78
 * End:
 */
