/* ScummVM - Graphic Adventure Engine
 *
 * ScummVM is the legal property of its developers, whose names
 * are too numerous to list here. Please refer to the COPYRIGHT
 * file distributed with this source distribution.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

/*
 * This code is based on Broken Sword 2.5 engine
 *
 * Copyright (c) Malte Thiesen, Daniel Queteschiner and Michael Elsdoerfer
 *
 * Licensed under GNU GPL v2
 *
 */

#ifndef SWORD25_BITMAP_H
#define SWORD25_BITMAP_H

#include "sword25/kernel/common.h"
#include "sword25/gfx/renderobject.h"

namespace Sword25 {

class Bitmap : public RenderObject {
protected:
	Bitmap(RenderObjectPtr<RenderObject> parentPtr, TYPES type, uint handle = 0);

public:

	virtual ~Bitmap();

	/**
	    @brief Setzt den Alphawert des Bitmaps.
	    @param Alpha der neue Alphawert der Bitmaps (0 = keine Deckung, 255 = volle Deckung).
	    @remark Diese Methode darf nur aufgerufen werden, wenn die Methode IsAlphaAllowed() true zurckgibt.
	*/
	void setAlpha(int alpha);

	/**
	    @brief Setzt die Modulationfarbe der Bitmaps.
	    @param Color eine 24-Bit Farbe, die die Modulationsfarbe des Bitmaps festlegt.
	    @remark Diese Methode darf nur aufgerufen werden, wenn die Methode IsColorModulationAllowed() true zurckgibt.
	*/
	void setModulationColor(uint modulationColor);

	/**
	    @brief Setzt den Skalierungsfaktor des Bitmaps.
	    @param ScaleFactor der Faktor um den das Bitmap in beide Richtungen gestreckt werden soll.
	    @remark Diese Methode darf nur aufgerufen werden, wenn die Methode IsScalingAllowed() true zurckgibt.
	*/
	void setScaleFactor(float scaleFactor);

	/**
	    @brief Setzt den Skalierungsfaktor der Bitmap auf der X-Achse.
	    @param ScaleFactor der Faktor um den die Bitmap in Richtungen der X-Achse gestreckt werden soll. Dieser Wert muss positiv sein.
	    @remark Diese Methode darf nur aufgerufen werden, wenn die Methode IsScalingAllowed() true zurckgibt.
	*/
	void setScaleFactorX(float scaleFactorX);

	/**
	    @brief Setzt den Skalierungsfaktor der Bitmap auf der Y-Achse.
	    @param ScaleFactor der Faktor um den die Bitmap in Richtungen der Y-Achse gestreckt werden soll. Dieser Wert muss positiv sein.
	    @remark Diese Methode darf nur aufgerufen werden, wenn die Methode IsScalingAllowed() true zurckgibt.
	*/
	void setScaleFactorY(float scaleFactorY);

	/**
	    @brief Legt fest, ob das Bild an der X-Achse gespiegelt werden soll.
	*/
	void setFlipH(bool flipH);

	/**
	    @brief Legt fest, ob das Bild an der Y-Achse gespiegelt werden soll.
	*/
	void setFlipV(bool flipV);

	/**
	    @brief Gibt den aktuellen Alphawert des Bildes zurck.
	    @remark Diese Methode darf nur aufgerufen werden, wenn die Methode IsAlphaAllowed() true zurckgibt.
	*/
	int getAlpha() {
		return _modulationColor >> 24;
	}

	/**
	    @brief Gibt die aktuelle 24bit RGB Modulationsfarde des Bildes zurck.
	    @remark Diese Methode darf nur aufgerufen werden, wenn die Methode IsColorModulationAllowed() true zurckgibt.
	*/
	int getModulationColor() {
		return _modulationColor & 0x00ffffff;
	}

	/**
	    @brief Gibt den Skalierungsfakter des Bitmaps auf der X-Achse zurck.
	    @remark Diese Methode darf nur aufgerufen werden, wenn die Methode IsScalingAllowed() true zurckgibt.
	*/
	float getScaleFactorX() const {
		return _scaleFactorX;
	}

	/**
	    @brief Gibt den Skalierungsfakter des Bitmaps auf der Y-Achse zurck.
	    @remark Diese Methode darf nur aufgerufen werden, wenn die Methode IsScalingAllowed() true zurckgibt.
	*/
	float getScaleFactorY() const {
		return _scaleFactorY;
	}

	/**
	    @brief Gibt zurck, ob das Bild an der X-Achse gespiegelt angezeigt wird.
	*/
	bool isFlipH() {
		return _flipH;
	}

	/**
	    @brief Gibt zurck, ob das Bild an der Y-Achse gespiegelt angezeigt wird.
	*/
	bool isFlipV() {
		return _flipV;
	}

	// -----------------------------------------------------------------------------
	// Die folgenden Methoden mssen alle BS_Bitmap-Klassen implementieren
	// -----------------------------------------------------------------------------

	/**
	    @brief Liest einen Pixel des Bildes.
	    @param X die X-Koordinate des Pixels.
	    @param Y die Y-Koordinate des Pixels
	    @return Gibt den 32-Bit Farbwert des Pixels an der bergebenen Koordinate zurck.
	    @remark Diese Methode sollte auf keine Fall benutzt werden um grere Teile des Bildes zu lesen, da sie sehr langsam ist. Sie ist
	            eher dafr gedacht einzelne Pixel des Bildes auszulesen.
	*/
	virtual uint getPixel(int x, int y) const = 0;

	/**
	    @brief Fllt den Inhalt des Bildes mit Pixeldaten.
	    @param Pixeldata ein Vector der die Pixeldaten enthlt. Sie mssen in dem Farbformat des Bildes vorliegen und es mssen gengend Daten
	           vorhanden sein, um das ganze Bild zu fllen.
	    @param Offset der Offset in Byte im Pixeldata-Vector an dem sich der erste zu schreibende Pixel befindet.<br>
	                  Der Standardwert ist 0.
	    @param Stride der Abstand in Byte zwischen dem Zeilenende und dem Beginn einer neuen Zeile im Pixeldata-Vector.<br>
	                  Der Standardwert ist 0.
	    @return Gibt false zurck, falls der Aufruf fehlgeschlagen ist.
	    @remark Ein Aufruf dieser Methode ist nur erlaubt, wenn IsSetContentAllowed() true zurckgibt.
	*/
	virtual bool    setContent(const byte *pixeldata, uint size, uint offset = 0, uint stride = 0) = 0;

	virtual bool    isScalingAllowed() const = 0;
	virtual bool    isAlphaAllowed() const = 0;
	virtual bool    isColorModulationAllowed() const = 0;
	virtual bool    isSetContentAllowed() const = 0;

	virtual bool    persist(OutputPersistenceBlock &writer);
	virtual bool    unpersist(InputPersistenceBlock &reader);

protected:
	bool  _flipH;
	bool  _flipV;
	float _scaleFactorX;
	float _scaleFactorY;
	uint  _modulationColor;
	int   _originalWidth;
	int   _originalHeight;
};

} // End of namespace Sword25

#endif
