/*
 *                            COPYRIGHT
 *
 *  sch-rnd - modular/flexible schematics editor - Allegro netlist export
 *  Copyright (C) 2025 Tibor 'Igor2' Palinkas
 *  Copyright (C) 2025 Aron Barath
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 31 Milk Street, # 960789 Boston, MA 02196 USA.
 *
 *  Contact:
 *    Project page: http://repo.hu/projects/sch-rnd
 *    contact lead developer: http://www.repo.hu/projects/sch-rnd/contact.html
 *    mailing list: http://www.repo.hu/projects/sch-rnd/contact.html
 */


#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <librnd/core/compat_misc.h>
#include <librnd/core/safe_fs.h>
#include <librnd/core/plugins.h>
#include <librnd/core/error.h>
#include <libcschem/config.h>
#include <libcschem/plug_io.h>

#include <plugins/lib_netlist_exp/lib_netlist_exp.h>

static csch_plug_io_t eallegro_net;

static int allegro_export_prio(const char *fn, const char *fmt, csch_plug_io_type_t type)
{
	if (type != CSCH_IOTYP_NETLIST)
		return 0;
	if (rnd_strcasecmp(fmt, "allegro") == 0)
		return 100;
	return 0;
}

static const char allegro_illegal_chars[] = "( )!/;,.";

/* print a stray name */
static void allegro_print_name(FILE* const f, const char* const str)
{
	if(str)
	{
		if(strpbrk(str, allegro_illegal_chars))
		{
			/* allegro does not escape, let the user know the output is broken */
			rnd_message(RND_MSG_ERROR, "allegro: Broken output! String contains one of: %s, but it cannot be escaped: %s\n",
				allegro_illegal_chars, str);
		}

		fputs(str, f);
	}
}

/* print a quoted name */
static void allegro_print_quoted(FILE* const f, const char* const str)
{
	if(str)
	{
		if(strchr(str, '\''))
		{
			rnd_message(RND_MSG_ERROR, "allegro: Broken output! String contains apostrophe (\'), but it cannot be escaped: %s\n", str);
		}

		fputc('\'', f);
		fputs(str, f);
		fputc('\'', f);
	}
}

/* Export abstract components; exports refdes, footprint, value and device */
static void allegro_export_comps(FILE *f, csch_abstract_t *abs)
{
	htsp_entry_t *e;

	fprintf(f, "$PACKAGES\n");

	for(e = htsp_first(&abs->comps); e != NULL; e = htsp_next(&abs->comps, e)) {
		csch_acomp_t *comp = e->value;
		const char *refdes = sch_nle_get_refdes(comp);
		const char *fp, *dev, *val;

		if (refdes == NULL)
			continue;

		if (comp->hdr.omit)
			continue;

		/* Get main symbol attributes the safe way, considering alternate names;
		   these should be always exported and are usually hardwiared in the
		   netlist format. */
		fp = sch_nle_get_alt_attr(&comp->hdr.attr, SCH_NLE_ALTATTR_FOOTPRINT);
		val = sch_nle_get_alt_attr(&comp->hdr.attr, SCH_NLE_ALTATTR_VALUE);
		dev = sch_nle_get_alt_attr(&comp->hdr.attr, SCH_NLE_ALTATTR_DEVICE);

		if(fp==NULL)
		{
			/* just warn the user */
			rnd_message(RND_MSG_ERROR, "allegro: no footprint is set for \"%s\"\n", refdes);

			/* if footprint is unknown, it can be set to empty string */
			fp = "";
		}

		if(val && dev)
		{
			/* cool, we have both */
		}
		else
		if(val)
		{
			dev = val;
		}
		else
		if(dev)
		{
			val = dev;
		}
		else
		{
			dev = val = "x";
		}

		allegro_print_quoted(f, fp);
		fputs("! ", f);
		allegro_print_quoted(f, dev);
		fputs("! ", f);
		allegro_print_quoted(f, val);
		fputs("; ", f);
		allegro_print_name(f, refdes);
		fputc('\n', f);
	}
}

/* Export abstract nets; exports net's name and a list of refdes-pinnum */
/* pairs connected */
static void allegro_export_nets(FILE *f, csch_abstract_t *abs)
{
	htsp_entry_t *e;
	long n;

	fprintf(f, "$NETS\n");

	for(e = htsp_first(&abs->nets); e != NULL; e = htsp_next(&abs->nets, e)) {
		csch_anet_t *net = e->value;
		const char *netname = sch_nle_get_netname(net);
		int net_exported = 0; /* net had at least one connection so it was written */

		if (net->hdr.omit) continue;

		for(n = 0; n < net->conns.used; n++) {
			csch_aport_t *port = net->conns.array[n];
			const char *refdes = NULL, *pinnums;

			if (port->hdr.type != CSCH_ATYPE_PORT) {
				rnd_message(RND_MSG_ERROR, "allegro: invalid connection (object type)\n");
				continue;
			}

			pinnums = sch_nle_get_pinnum(port);
			if (pinnums == NULL) {
				rnd_message(RND_MSG_ERROR, "allegro: can't determine port's pin number\n");
				continue;
			}

			if (port->parent != NULL) {
				refdes = sch_nle_get_refdes(port->parent);
				if (port->parent->hdr.omit)
					continue; /* omit component */
			}
			if (refdes == NULL) {
				/* This is not an error: no refdes means: do not export (e.g. gnd) */
/*				rnd_message(RND_MSG_ERROR, "allegro: can't determine port's parent component refdes\n");*/
				continue;
			}

			/* split up pinnum at space and create one or more conn lines connecting
			   each pin to the given net */
			SCH_NLE_FOREACH_PINNUM(pinnums, my_num,
				{
					if(!net_exported)
					{
						net_exported = 1;
						fputc('/', f);
						allegro_print_quoted(f, netname);
						fputc(';', f);
					}

					fputc(',', f);
					fputc('\n', f);
					fputc('\t', f);
					allegro_print_name(f, refdes);
					fputc('.', f);
					allegro_print_name(f, my_num);
				}
			);
		}

		/* If the net got exported, then close the net export */
		if(net_exported)
		{
			fputc('\n', f);
		}
	}
}


/* Export netlist from the abstract model */
static int allegro_export_project_abst(const char *fn, const char *fmt, csch_abstract_t *abs, rnd_hid_attr_val_t *options)
{
	TODO("get hidlib as an arg")
	rnd_design_t *hidlib = NULL;
	FILE *f = rnd_fopen(hidlib, fn, "w");
	if (f == NULL)
		return -1;

	fprintf(f, "(generated by sch-rnd)\n");
	allegro_export_comps(f, abs);
	allegro_export_nets(f, abs);
	fprintf(f, "$END\n");

	fclose(f);
	return 0;
}

#include "hid_impl.c"

int pplg_check_ver_export_allegro(int ver_needed) { return 0; }

void pplg_uninit_export_allegro(void)
{
	csch_plug_io_unregister(&eallegro_net);
	rnd_export_remove_opts_by_cookie(allegro_cookie);
	rnd_hid_remove_hid(&allegro_hid);
}

int pplg_init_export_allegro(void)
{
	RND_API_CHK_VER;

	eallegro_net.name = "export to Allegro";
	eallegro_net.export_prio = allegro_export_prio;
	eallegro_net.export_project_abst = allegro_export_project_abst;
	eallegro_net.ext_export_project = ".net";
	csch_plug_io_register(&eallegro_net);


	rnd_hid_nogui_init(&allegro_hid);

	allegro_hid.struct_size = sizeof(rnd_hid_t);
	allegro_hid.name = "allegro";
	allegro_hid.description = "Exports project's Allegro netlist";
	allegro_hid.exporter = 1;

	allegro_hid.get_export_options = allegro_get_export_options;
	allegro_hid.do_export = allegro_do_export;
	allegro_hid.parse_arguments = allegro_parse_arguments;
	allegro_hid.argument_array = allegro_values;

	allegro_hid.usage = allegro_usage;

	rnd_hid_register_hid(&allegro_hid);
	rnd_hid_load_defaults(&allegro_hid, allegro_options, NUM_OPTIONS);


	return 0;
}

