/*
 * Robot Testing Framework
 *
 * Copyright (C) 2015-2019 Istituto Italiano di Tecnologia (IIT)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


#include <robottestingframework/Asserter.h>
#include <robottestingframework/WebProgressListener.h>
#include <robottestingframework/impl/WebProgressListener_impl.h>

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <sstream>

using namespace std;
using namespace robottestingframework;


#define BLUE "<font color=\"blue\" style=\"font-weight: bold;\">"
#define GREEN "<font color=\"green\" style=\"font-weight: bold;\">"
#define RED "<font color=\"red\" style=\"font-weight: bold;\">"
#define GRAY "<font color=\"gray\" style=\"font-weight: bold;\">"
#define ENDC "</font>"

#define MSG_ERROR RED "[ERROR]&nbsp;" ENDC
#define MSG_FAIL RED "[FAIL]&nbsp;&nbsp;" ENDC
#define MSG_REPORT GREEN "[INFO]&nbsp;&nbsp;" ENDC

static std::string html_page = "<!DOCTYPE html>\n"
                               "<html>\n"
                               "<head>\n"
                               "<style>\n"
                               ".font_h4 {\n"
                               "font: normal normal normal 36px/1.3em Arial,'ms pgothic',dotum,helvetica,sans-serif;\n"
                               "color: #1C1C1C; font-weight: bold; }\n"
                               "\n"
                               ".font_1 {\n"
                               "font: normal normal bold 12px/1.3em Arial,'ms pgothic',dotum,helvetica,sans-serif;\n"
                               "color: #808080; }\n"
                               "\n"
                               ".my_window {\n"
                               "position:absolute;\n"
                               "left:20px; right:20px;\n"
                               "background: #efefef; background-color: #efefef;\n"
                               "border: solid 1px #0088CB;\n"
                               "-webkit-border-radius: 4px; -moz-border-radius: 4px; border-radius: 4px;\n"
                               "font-family: 'Spinnaker' !important; list-style: none;\n"
                               "margin: 0 0 10px; padding: 5px; position: absolute;\n"
                               "box-shadow: 0 5px 15px 0 rgba(0,0,0,.4); }\n"
                               "</style>\n"
                               "\n"
                               "<script>\n"
                               "function update() {\n"
                               "    var xmlUpdate;\n"
                               "    if (window.XMLHttpRequest) {\n"
                               "      xmlUpdate=new XMLHttpRequest();\n"
                               "    }\n"
                               "    else {\n"
                               "        xmlUpdate=new ActiveXObject(\"Microsoft.XMLHTTP\");\n"
                               "    }\n"
                               "    xmlUpdate.onreadystatechange=function() {\n"
                               "        if (xmlUpdate.readyState==4 && xmlUpdate.status==200) {\n"
                               "            document.getElementById(\"result\").innerHTML=xmlUpdate.responseText;\n"
                               "             if(document.getElementById(\"autoscroll\").checked == true) {\n"
                               "               document.getElementById(\"result\").scrollTop = document.getElementById(\"result\").scrollHeight;\n"
                               "             }\n"
                               "        }\n"
                               "    }\n"
                               "    xmlUpdate.open(\"GET\",document.URL+\"update\",true);\n"
                               "    xmlUpdate.send();\n"
                               "\n"
                               "    var xmlStatus;\n"
                               "    if (window.XMLHttpRequest) {\n"
                               "        xmlStatus=new XMLHttpRequest();\n"
                               "    }\n"
                               "    else {\n"
                               "        xmlStatus=new ActiveXObject(\"Microsoft.XMLHTTP\");\n"
                               "    }\n"
                               "    xmlStatus.onreadystatechange=function() {\n"
                               "        if (xmlStatus.readyState==4 && xmlStatus.status==200) {\n"
                               "            var status = JSON.parse(xmlStatus.responseText);\n"
                               "            document.getElementById(\"suite_name\").innerHTML=status.name;\n"
                               "            var c = document.getElementById(\"progress\");\n"
                               "            var ctx = c.getContext(\"2d\");\n"
                               "            ctx.clearRect(0, 0, c.width, c.height);\n"
                               "            var xpos = 20;\n"
                               "            for (i = 0; i < status.testStatus.length; i++) {\n"
                               "                ctx.beginPath();\n"
                               "                ctx.arc(xpos, c.height/2, 7, 0, 2 * Math.PI);\n"
                               "                ctx.lineWidth = 2;\n"
                               "                switch (status.testStatus[i]) {\n"
                               "                case 0:\n"
                               "                    ctx.strokeStyle=\"#B0C0D0\";\n"
                               "                    ctx.fillStyle=\"#BBCCDD\";\n"
                               "                    break;\n"
                               "                case 1:\n"
                               "                    ctx.strokeStyle=\"#708090\";\n"
                               "                    ctx.fillStyle=\"#778899\";\n"
                               "                    break;\n"
                               "                case 2:\n"
                               "                    ctx.strokeStyle=\"#FF0000\";\n"
                               "                    ctx.fillStyle=\"#DC143C\";\n"
                               "                    break;\n"
                               "                case 3:\n"
                               "                    ctx.strokeStyle=\"#008000\";\n"
                               "                    ctx.fillStyle=\"#228B22\";\n"
                               "                    break;\n"
                               "                default:\n"
                               "                    break;\n"
                               "                }\n"
                               "                ctx.stroke();\n"
                               "                ctx.fill();\n"
                               "                xpos = xpos + 20;\n"
                               "            }\n"
                               "        }\n"
                               "    }\n"
                               "    xmlStatus.open(\"GET\",document.URL+\"status\",true);\n"
                               "    xmlStatus.send();\n"
                               "\n"
                               "    setTimeout(update, 1000);\n"
                               "}\n"
                               "setTimeout(update, 1000);\n"
                               "</script>\n"
                               "</head>\n"
                               "<body bgcolor=\"#efefef\">\n"
                               "<div class=\"my_window\" style=\"height:52px\">\n"
                               "<div style=\"position:relative; padding:5px; float:left;\"> <img src=\"data:image/png;base64,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\"/> </div>\n"
                               "<div style=\"position:relative; padding:5px; float:center;\"> <h4 id=\"suite_name\" style=\"text-align: center; margin:0;\" class=\"font_h4\">Robot Testing Framework</h4></div>\n"
                               "</div>\n"
                               "<div class=\"my_window\" style=\"top:75px; height:90%; padding:5px\"><div id=\"result\" style=\"top:0px; height:96%; overflow-y:scroll; background:#ffffff;\"></div>\n"
                               "<table style=\"width:100%\"><tr><td width=\"90%\"><canvas id=\"progress\" width=\"1000px\" height=\"20px\" style=\"width:1000px; height:20px\" ></canvas></td><td align=\"right\"><input align=\"right\" type=\"checkbox\" id=\"autoscroll\" checked/> auto scroll</div></td></tr></table>\n"
                               //"<div style=\"top:94%; position:absolute; padding-left:20px;\">\n"
                               //"<p class=\"font_1\">Copyright (C) 2015-2019 Istituto Italiano di Tecnologia (IIT)</p>\n"
                               //"</div>\n"
                               "</body>\n"
                               "</html>\n";

WebProgressListenerImpl& WebProgressListenerImpl::create(unsigned int port,
                                                         bool verbose)
{
    static WebProgressListenerImpl instance(port, verbose);
    return instance;
}

WebProgressListenerImpl::WebProgressListenerImpl(unsigned int port,
                                                 bool verbose)
{
    suite_size = 0;
    this->verbose = verbose;
    this->port = port;
    server = mg_create_server(this, WebProgressListenerImpl::handler);
    if (server != nullptr) {
        std::string port_str = Asserter::format("%d", port);
        mg_set_option(server,
                      "listening_port",
                      port_str.c_str());
        shouldStop = false;
        updater = new std::thread(update, this);
    }
}

WebProgressListenerImpl::~WebProgressListenerImpl()
{
    shouldStop = true;
    // stop the pooling thread
    if (updater != nullptr) {
        updater->join();
        delete updater;
        updater = nullptr;
    }
    // delete the web server
    if (server != nullptr) {
        // ensure the last message delivery (?)
        mg_poll_server(server, 1000);
        mg_destroy_server(&server);
        server = nullptr;
    }
}

void WebProgressListenerImpl::update(void* param)
{
    auto* web = (WebProgressListenerImpl*)param;
    while (!web->shouldStop) {
        mg_poll_server(web->server, 1000);
    }
}

int WebProgressListenerImpl::handler(struct mg_connection* conn,
                                     enum mg_event ev)
{
    auto* web = (WebProgressListenerImpl*)conn->server_param;
    if (ev == MG_REQUEST) {
        if (strcmp(conn->uri, "/update") == 0) {
            web->critical.lock();
            std::string data = web->result;
            web->critical.unlock();
            mg_send_header(conn, "Content-Type", "text/turtle");
            mg_send_header(conn, "Access-Control-Allow-Origin", "*");
            //mg_send_header(conn, "Content-Location", "mydata.ttl");
            mg_printf_data(conn, "%s", data.c_str());
        } else if (strcmp(conn->uri, "/status") == 0) {
            web->critical.lock();
            std::stringstream ss;
            ss << R"({"name":")" << web->suite_name;
            ss << R"(","testStatus":[)";
            for (auto it = web->testStatus.begin(); it != web->testStatus.end(); ++it) {
                ss << (int)(*it);
                if (std::next(it) != web->testStatus.end()) {
                    ss << ",";
                }
            }
            for (int i = web->testStatus.size(); i < web->suite_size; ++i) {
                ss << ",0";
            }
            ss << "]}";

            web->critical.unlock();
            mg_send_header(conn, "Content-Type", "text/turtle");
            mg_send_header(conn, "Access-Control-Allow-Origin", "*");
            mg_printf_data(conn, "%s", ss.str().c_str());
        } else {
            mg_send_data(conn, html_page.c_str(), strlen(html_page.c_str()));
        }
        return MG_TRUE;
    }
    if (ev == MG_AUTH) {
        return MG_TRUE;
    }
    return MG_FALSE;
}

std::string WebProgressListenerImpl::encode(const std::string& data)
{
    std::string buffer;
    buffer.reserve(data.size());
    for (size_t pos = 0; pos != data.size(); ++pos) {
        switch (data[pos]) {
        case '&':
            buffer.append("&amp;");
            break;
        case '\"':
            buffer.append("&quot;");
            break;
        case '\'':
            buffer.append("&apos;");
            break;
        case '<':
            buffer.append("&lt;");
            break;
        case '>':
            buffer.append("&gt;");
            break;
        default:
            buffer.append(&data[pos], 1);
            break;
        }
    }
    return buffer;
}

void WebProgressListenerImpl::addReport(const Test* test,
                                        TestMessage msg)
{
    string text = Asserter::format("<br> %s (%s) %s : %s.",
                                   MSG_REPORT,
                                   encode(test->getName()).c_str(),
                                   encode(msg.getMessage()).c_str(),
                                   encode(msg.getDetail()).c_str());
    critical.lock();
    result += text;
    critical.unlock();
    //if(verbose && msg.getSourceLineNumber() != 0)
    //    cout<<GRAY<<msg.getSourceFileName()<<" at "<<msg.getSourceLineNumber()<<"."<<ENDC<<endl<<endl;
}

void WebProgressListenerImpl::addError(const Test* test,
                                       TestMessage msg)
{
    string text = Asserter::format("<br> %s (%s) %s : %s.",
                                   MSG_ERROR,
                                   encode(test->getName()).c_str(),
                                   encode(msg.getMessage()).c_str(),
                                   encode(msg.getDetail()).c_str());
    critical.lock();
    result += text;
    critical.unlock();
}

void WebProgressListenerImpl::addFailure(const Test* test,
                                         TestMessage msg)
{
    string text = Asserter::format("<br> %s (%s) %s : %s.",
                                   MSG_FAIL,
                                   encode(test->getName()).c_str(),
                                   encode(msg.getMessage()).c_str(),
                                   encode(msg.getDetail()).c_str());
    critical.lock();
    result += text;
    critical.unlock();
}

void WebProgressListenerImpl::startTest(const Test* test)
{
    string text = Asserter::format("<br> %s Test case %s started... %s",
                                   BLUE,
                                   encode(test->getName()).c_str(),
                                   ENDC);
    critical.lock();
    // if there is no test suite, use the test case's name
    if (suite_name.empty()) {
        suite_name = test->getName();
    }
    testStatus.push_back(TestStatus::Running);
    result += text;
    critical.unlock();
}

void WebProgressListenerImpl::endTest(const Test* test)
{
    string text = Asserter::format("<br> %s Test case %s %s %s",
                                   BLUE,
                                   encode(test->getName()).c_str(),
                                   (test->succeeded()) ? "passed!" : "failed!",
                                   ENDC);
    critical.lock();
    testStatus.back() = (test->succeeded()) ? TestStatus::Success : TestStatus::Failed;
    result += text;
    critical.unlock();
}

void WebProgressListenerImpl::startTestSuite(const Test* test)
{
    string text = Asserter::format("<br> %s Test suite %s started... %s",
                                   BLUE,
                                   encode(test->getName()).c_str(),
                                   ENDC);
    critical.lock();
    result += text;
    suite_name = test->getName();
    auto suite = dynamic_cast<const TestSuite*>(test);
    if (suite != nullptr) {
        suite_size = suite->size();
    }
    critical.unlock();
}

void WebProgressListenerImpl::endTestSuite(const Test* test)
{
    string text = Asserter::format("<br> %s Test suite %s %s %s",
                                   BLUE,
                                   encode(test->getName()).c_str(),
                                   (test->succeeded()) ? "passed!" : "failed!",
                                   ENDC);
    critical.lock();
    result += text;
    critical.unlock();
}

void WebProgressListenerImpl::startTestRunner()
{
    string text = Asserter::format("<br> %s Starting test runner. %s",
                                   BLUE,
                                   ENDC);
    critical.lock();
    result += text;
    critical.unlock();
}

void WebProgressListenerImpl::endTestRunner()
{
    string text = Asserter::format("<br> %s Ending test runner. %s",
                                   BLUE,
                                   ENDC);
    critical.lock();
    result += text;
    critical.unlock();
}


/**
 * WebProgressListener
 */
WebProgressListener::WebProgressListener(unsigned int port,
                                         bool verbose)
{
    implement = &WebProgressListenerImpl::create(port, verbose);
}

WebProgressListener::~WebProgressListener() = default;

void WebProgressListener::addReport(const Test* test,
                                    TestMessage msg)
{
    ((WebProgressListener*)implement)->addReport(test, msg);
}

void WebProgressListener::addError(const Test* test,
                                   TestMessage msg)
{
    ((WebProgressListener*)implement)->addError(test, msg);
}

void WebProgressListener::addFailure(const Test* test,
                                     TestMessage msg)
{
    ((WebProgressListener*)implement)->addFailure(test, msg);
}

void WebProgressListener::startTest(const Test* test)
{
    ((WebProgressListener*)implement)->startTest(test);
}

void WebProgressListener::endTest(const Test* test)
{
    ((WebProgressListener*)implement)->endTest(test);
}

void WebProgressListener::startTestSuite(const Test* test)
{
    ((WebProgressListener*)implement)->startTestSuite(test);
}

void WebProgressListener::endTestSuite(const Test* test)
{
    ((WebProgressListener*)implement)->endTestSuite(test);
}

void WebProgressListener::startTestRunner()
{
    ((WebProgressListener*)implement)->startTestRunner();
}

void WebProgressListener::endTestRunner()
{
    ((WebProgressListener*)implement)->endTestRunner();
}
