///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
#include "rheolef/config.h"

#ifdef _RHEOLEF_HAVE_UMFPACK

#include "rheolef/ssk.h"
#include "rheolef/csr.h"
#include "rheolef/vec.h"
using namespace rheolef;
using namespace std;

extern "C" {
#include <umfpack.h>
}

// umfpack error handler
static
void 
treat_error (int status)
{
	if (status == UMFPACK_OK) return;
	if (status == UMFPACK_WARNING_singular_matrix) {
	     warning_macro("singular matrix");
	     return;
	}
	string msg;
	switch (status) {
	    case UMFPACK_ERROR_n_nonpositive:
	     msg = "size is less or equal to zero"; break;
	    case UMFPACK_ERROR_invalid_matrix:
	     msg = "invalid matrix"; break;
	    case UMFPACK_ERROR_out_of_memory:
	     msg = "out of memory"; break;
	    case UMFPACK_ERROR_argument_missing:
	     msg = "argument missing"; break;
	    case UMFPACK_ERROR_internal_error:
	     msg = "internal error"; break;
	    case UMFPACK_ERROR_invalid_Symbolic_object:
	     msg = "invalid Symbolic object"; break;
	    case UMFPACK_ERROR_different_pattern:
	     msg = "different pattern"; break;
	    case UMFPACK_ERROR_invalid_Numeric_object:
	     msg = "invalid Numeric object"; break;
	    default:
	     fatal_macro("unexpected umfpack error status = " << status);
        }
        error_macro (msg);
}
// ------------------------------------------------------------
struct internal_umfpack_rep {

    typedef vector<int>::size_type size_type;
    typedef double       element_type;

    void   *_numeric;
    int    *_ia_p;
    int    *_ja_p;
    double *_a_p;
    int     _n;

    void constructor();
    void destructor();
    void factorize (
        size_type                            n,
        vector<size_type>::const_iterator    ia,
        vector<size_type>::const_iterator    ja,
        vector<element_type>::const_iterator a);
  
    void solve (
        vector<element_type>::const_iterator b,
        vector<element_type>::iterator       x);
};

void
internal_umfpack_rep::constructor()
{
    _numeric = 0;
    _ia_p = 0;
    _ja_p = 0;
    _a_p  = 0;
    _n    = 0;
}
void
internal_umfpack_rep::destructor()
{
    if (_numeric) umfpack_di_free_numeric (&_numeric); 
    if (_ia_p) delete_tab_macro (_ia_p); 
    if (_ja_p) delete_tab_macro (_ja_p); 
    if (_a_p)  delete_tab_macro (_a_p); 
    _n    = 0;
}
void
internal_umfpack_rep::factorize (
    size_type                            n,
    vector<size_type>::const_iterator    ia,
    vector<size_type>::const_iterator    ja,
    vector<element_type>::const_iterator a)
{
    int nnz = ia[n];
    if (n == 0 || nnz == 0) {
        // empty matrix
        return;
    }
    double *null = (double *) NULL ;
    void *symbolic;
    double info [UMFPACK_INFO];
    _ia_p = new_tab_macro (int,    n+1);
    _ja_p = new_tab_macro (int,    nnz);
    _a_p  = new_tab_macro (double, nnz);
    copy (ia, ia+n+1, _ia_p);
    copy (ja, ja+nnz, _ja_p);
    copy ( a,  a+nnz,  _a_p);
    _n = int(n);
    umfpack_di_symbolic (_n, _n, _ia_p, _ja_p, _a_p, &symbolic, null, info);
    treat_error (int(info [UMFPACK_STATUS]));
    umfpack_di_numeric (_ia_p, _ja_p, _a_p, symbolic, &_numeric, null, info);
    treat_error (int(info [UMFPACK_STATUS]));
    umfpack_di_free_symbolic (&symbolic) ;
}
void
internal_umfpack_rep::solve (
    vector<element_type>::const_iterator b,
    vector<element_type>::iterator       x)
{
    if (_n == 0) {
        // empty matrix
        return;
    }
    double *b_p = new_tab_macro (double, _n);
    double *x_p = new_tab_macro (double, _n);
    copy (b,  b+_n,  b_p);
    // parano :
    fill (x_p, x_p+_n, numeric_limits<double>::max());
    double info [UMFPACK_INFO];
    double *null = (double *) NULL ;
    umfpack_di_solve (UMFPACK_At, _ia_p, _ja_p, _a_p, x_p, b_p, _numeric, null, info) ;
    treat_error (int(info [UMFPACK_STATUS]));
    copy (x_p,  x_p+_n,  x);
    delete_tab_macro (x_p); 
    delete_tab_macro (b_p); 
}
// ------------------------------------------------------------

inline
internal_umfpack_rep&
getrep(void *p)
{
    return *((internal_umfpack_rep*)p);
}
template<class T>
umfpack_rep<T>::umfpack_rep ()
{
    _p = new_macro(internal_umfpack_rep);
    getrep(_p).constructor();
}
template<class T>
umfpack_rep<T>::~umfpack_rep ()
{
    getrep(_p).destructor();
    delete_macro(((internal_umfpack_rep*)_p));
}
template<class T>
umfpack_rep<T>::umfpack_rep (const csr<T>& a)
{
    _p = new_macro(internal_umfpack_rep);
    getrep(_p).factorize (a.nrow(), a.ia().begin(), a.ja().begin(), a.a().begin());
}
template<class T>
void
umfpack_rep<T>::factorize()
{
    // already factorized: nothing to do
}
template<class T>
void
umfpack_rep<T>::solve(const vec<T>& b, vec<T>& x) const
{
    if (x.size() == 0) return; // spooles core dump otherwise
    getrep(_p).solve(b.begin(),x.begin());
}
template<class T>
typename umfpack_rep<T>::size_type
umfpack_rep<T>::nnz () const
{
    fatal_macro("umfpack_rep::nnz: not implemented");
    return 0;
}
template<class T>
typename umfpack_rep<T>::size_type
umfpack_rep<T>::nrow () const
{
    fatal_macro("umfpack_rep::nrow: not implemented");
    return 0;
}
template<class T>
typename umfpack_rep<T>::size_type
umfpack_rep<T>::ncol () const
{
    fatal_macro("umfpack_rep::ncol: not implemented");
    return 0;
}

namespace rheolef { 
// instanciation in library
template class ssk<double>;
template class umfpack_rep<double>;
template ssk<double> ldlt (const csr<double>&);
template ssk<double> lu   (const csr<double>&);
}// namespace rheolef

#endif // _RHEOLEF_HAVE_SPOOLES
