# ifndef _SKIT_SSK_ALGORITHM_H
# define _SKIT_SSK_ALGORITHM_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
namespace rheolef { 
//
// SSK: Symmetric SKyline format
//
// algorithm-oriented generic library
//
// author: Pierre.Saramito@imag.fr
//
// date: 4 september 1997
//
//@!\vfill\listofalgorithms\vfill\newpage

/*@! 
 \begin{quote}
 \begin{algorithm}[h]
  \caption{Compute effective non-null elements in SSK}
  \begin{algorithmic}
    \INPUT {values of the matrix in SSK format}
      asky(0:nnzsky-1)\\
    \ENDINPUT
    \OUTPUT {non-null elements}
      nnz
    \ENDOUTPUT
    \BEGIN 
      nnz := 0 \\
      \FORTO{i := 0}{nnzsky-1}
        \IF{asky(i) $<>$ 0}
	  nnz++;
        \ENDIF
      \ENDFOR
    \END
  \end{algorithmic}
 \end{algorithm}
 \end{quote}
 \vfill
*/
template <
    class InputIteratorValue,
    class Size>
void
nnz_skyline (
    InputIteratorValue  iter_asky,
    InputIteratorValue  last_asky,
    Size&               nnz)
{
    nnz = 0;
    while (iter_asky != last_asky)
      if (*iter_asky++ != 0) 
	nnz++;
}
/*@! 
 \vfill
 \pagebreak
 \mbox{}
 \vfill
 \begin{quote}
 \begin{algorithm}[h]
  \caption{Compute SSK area from CSR}
  \begin{algorithmic}
    \INPUT {symmetric matrix in CSR format with column index in increasing order}
      ia(0:n), ja(0:nnz-1)\\
    \ENDINPUT
    \OUTPUT {the SSK area}
      nnzsky
    \ENDOUTPUT
    \BEGIN 
      nnzsky := 0 \\
      \FORTO{i := 0}{n-1}
        nnzsky += i - ja(ia(i)) + 1 \\
      \ENDFOR
    \END
  \end{algorithmic}
 \end{algorithm}
 \end{quote}
 \vfill
*/
template <
    class InputIteratorIndex,
    class RandomAccessIteratorIndex,
    class Size>
void
skyline_area (
    InputIteratorIndex        iter_ia,
    InputIteratorIndex        last_ia,
    RandomAccessIteratorIndex rand_ja,
    Size&                     nnzsky)
{
    nnzsky = 0;
    Size i = 0;
    InputIteratorIndex prec_ia = iter_ia++;
    while (iter_ia != last_ia) {
        nnzsky += (i++) - rand_ja [*prec_ia] + 1;
        prec_ia = iter_ia++;
    }
}
/*@! 
 \vfill
 \pagebreak
 \mbox{}
 \vfill
 \begin{quote}
 \begin{algorithm}[h]
  \caption{Compute the skyline area and set indexes}
  \begin{algorithmic}
    \INPUT {symmetric matrix in CSR format}
      ia(0:n), ja(0:nnz-1)\\
    \ENDINPUT
    \OUTPUT {the skiline area and indexes}
      nnzsky, isky(0:n)
    \ENDOUTPUT
    \BEGIN 
      isky(0) := 0 \\
      nnzsky := 0 \\
      \FORTO{i := 0}{n-1}
        nnzsky += i - ja(ia(i)) + 1 \\
        isky(i+1) := nnzsky
      \ENDFOR
    \END
  \end{algorithmic}
 \end{algorithm}
\end{quote}
\vfill
*/

template <
    class InputIteratorIndex,
    class RandomAccessIteratorIndex,
    class OutputIteratorIndex,
    class Size>
void
_csr_to_ssk_area_and_set_indexes (
    InputIteratorIndex        iter_ia,
    InputIteratorIndex        last_ia,
    RandomAccessIteratorIndex rand_ja,
    OutputIteratorIndex       iter_isky,
    Size&                     nnzsky)
{
    nnzsky = 0;
    Size i = 0;
    InputIteratorIndex prec_ia = iter_ia++;
    *iter_isky++ = 0;
    while (iter_ia != last_ia) {
        nnzsky += (i++) - rand_ja [*prec_ia] + 1;
        *iter_isky++ = nnzsky;
        prec_ia = iter_ia++;
    }
}
/*@! 
 \vfill \pagebreak \mbox{} \vfill \begin{quote}
 \begin{algorithm}[h]
  \caption{Set non-null values from CSR to SSK}
  \begin{algorithmic}
    \INPUT {symmetric matrix in CSR format}
      ia(0:n), ja(0:nnz-1), a(0:nnz-1)\\
    \ENDINPUT
    \OUTPUT {the matrix in SSK format}
      isky(0:n), asky(0:nnzsky-1)
    \ENDOUTPUT
    \BEGIN 
      p := 0 \\
      \FORTO{i := 0}{n-1}
        \WHILE{p $<$ ia(i+1)}
	  j := ja(p) \\
	  \IF{j $\leq$ i}
	      asky(isky(i+1)-i-1+j) := a(p)
          \ENDIF
	  p++
        \ENDWHILE
      \ENDFOR
    \END
  \end{algorithmic}
 \end{algorithm}
 \end{quote} \vfill \pagebreak \mbox{} \vfill
 */
template<
    class InputIteratorIndex1,
    class InputIteratorIndex2,
    class InputIteratorIndex3,
    class InputIteratorValue,
    class RandomAccessIteratorvalue,
    class Size,
    class Difference>
void
_csr_to_ssk_set_values(
    InputIteratorIndex1        iter_ia,
    InputIteratorIndex1        last_ia,
    InputIteratorIndex2        iter_ja,
    InputIteratorValue         iter_a,
    InputIteratorIndex3        iter_isky,
    RandomAccessIteratorvalue  rand_asky,
    Size&                      nnz,
    Difference                 _signed_tag)
{
    nnz = 0;
    Size i = 0;
    iter_ia++;
    while (iter_ia != last_ia) {
        Size pf = *iter_ia++;
        Difference qi = (*++iter_isky) - (i++) - 1;
        while (nnz < pf) {
            Size j = *iter_ja++;
            if (j <= i) rand_asky [qi + j] = *iter_a;
            ++iter_a;
            ++nnz;
        }
    }
}
/*@! 
 \vfill
 \pagebreak
 \mbox{}
 \vfill
 \begin{quote}
 \begin{algorithm}[h]
  \caption{Compute effective non-null elements and set CSR indexes}
  \begin{algorithmic}
    \INPUT {the matrix in SSK format}
      isky(0:n), asky(0:nnzsky-1)
    \ENDINPUT
    \OUTPUT {the CSR indexes}
      nnz, ia(0:n)
    \ENDINPUT
    \BEGIN 
      ia(0) := 0 \\
      nnz := 0 \\
      q := 0 \\
      \FORTO {i := 0} {n-1}
        \WHILE {q $<$ isky(i+1)}
          \IF {asky(q) $<>$ 0}
	      nnz++
          \ENDIF
	  q++
        \ENDWHILE
	ia(i+1) := nnz
      \ENDFOR
    \END
  \end{algorithmic}
 \end{algorithm}
\end{quote}
\vfill
*/
template <
    class InputIteratorIndex,
    class InputIteratorValue,
    class OutputIteratorIndex,
    class Size,
    class T>
void
_ssk_to_csr_nnz_and_set_indexes (
    InputIteratorIndex       iter_isky,
    InputIteratorIndex       last_isky,
    InputIteratorValue       iter_asky,
    OutputIteratorIndex      iter_ia,
    Size&                    nnz,
    const T&)
{
    nnz = *iter_ia++ = 0;
    Size q = *iter_isky++;
    while (iter_isky != last_isky) {
        Size qf = *iter_isky++;
	while (q < qf) {
	    if (*iter_asky++ != T(0)) 
	        nnz++;
            q++;
	}
	*iter_ia++ = nnz;
    }
}
/*@! 
 \vfill
 \pagebreak
 \mbox{}
 \vfill
 \begin{quote}
 \begin{algorithm}[h]
  \caption{Set non-null values from SSK to SSR}
  \begin{algorithmic}
    \INPUT {the matrix in SSK format and the SSR indexes}
      isky(0:n), asky(0:nnzsky-1), ia(0:n)
    \ENDINPUT
    \OUTPUT {the symmetric matrix in SSR indexes}
      ja(0:nnz-1), a(0:nnz-1)
    \ENDINPUT
    \BEGIN 
      p := 0 \\
      q := 0 \\
      \FORTO {i := 0} {n-1}
        \WHILE {q $<$ isky(i+1)}
          \IF {asky(q) $<>$ 0}
	      ja(p) := i - isky(i+1) + q - 1 \\
	      a(p) := asky(q) \\
	      p++
          \ENDIF
	  q++
        \ENDWHILE
      \ENDFOR
    \END
  \end{algorithmic}
 \end{algorithm}
\end{quote}
\vfill
 \pagebreak
 \mbox{}
 \vfill
*/
template <
    class InputIteratorIndex1,
    class InputIteratorIndex2,
    class InputIteratorValue,
    class OutputIteratorIndex,
    class OutputIteratorValue,
    class Size,
    class Difference,
    class T>
void
_ssk_to_csr_set_values(
    InputIteratorIndex1      iter_isky,
    InputIteratorIndex1      last_isky,
    InputIteratorValue       iter_asky,
    InputIteratorIndex2      iter_ia,
    OutputIteratorIndex      iter_ja,
    OutputIteratorValue      iter_a,
    Size&                    nnzsky,
    Difference               _signed_tag,
    const T&)
{
    nnzsky = *iter_isky++;
    Size p = 0;
    Size i = 0;
    iter_ia++;
    while (iter_isky != last_isky) {
        Size qf = *iter_isky++;
	Difference qi  = qf - (i++) - 1;
	while (nnzsky < qf) {
	  if (*iter_asky != T(0)) {
	      *iter_ja++ = nnzsky - qi;
	      *iter_a++  = *iter_asky;
	      p++;
	  }
	  iter_asky++;
          nnzsky++;
	}
	*iter_ia++ = p;
    }
}
/*@! 
 \vfill
 \pagebreak
 \mbox{}
 \vfill
 \begin{quote}
 \begin{algorithm}[h]
  \caption{Inplace $A=LDL^t$ matrix factorization in SSK format}
  \begin{algorithmic}
    \INPUT {indexes of the first elements of the rows}
      isky(0:n)
    \ENDINPUT
    \INOUT {values}
      asky(0:nnzsky-1)
    \ENDINOUT
    \mbox{}\\
    \begin{tabular}{lccr}
     {\bf alias} & colmin(i) & = & isky(i) - isky(i+1) + i + 1 \\
     {\bf alias} & a(i,j)    & = & asky(j + isky(i+1) - i - 1)
    \end{tabular}
    \mbox{}\\
    \mbox{}\\
    \BEGIN 
      \FORTO {i := 0} {n-1}
	\FORTO {j := colmin(i)} {i-1}
	    \FORTO {k := max(colmin(i), colmin(j))} {j-1}
		a(i,j) -= a(i,k)*a(j,k)
            \ENDFOR
        \ENDFOR
	\FORTO {j := colmin(i)} {i-1}
 	    $\lambda$ := a(i,j)/a(j,j) \\
	    a(i,i) -= $\lambda$*a(i,j) \\
	    a(i,j) := $\lambda$
        \ENDFOR
      \ENDFOR
    \END
  \end{algorithmic}
 \end{algorithm}
\end{quote}
\vfill
 \pagebreak
 \mbox{}
 \vfill
*/
template<
    class RandomAccessIteratorIndex,
    class RandomAccessIteratorValue,
    class Size,
    class T>
struct
ssk_factorize_ldlt_struct
{
  inline Size colmin (Size i) { return isky[i] - isky[i+1] + i + 1; }
  inline T& a (Size i, Size j) { return asky[j + isky[i+1] - i - 1]; }

  ssk_factorize_ldlt_struct (
    RandomAccessIteratorIndex isky1,
    RandomAccessIteratorValue asky1,
    Size                      n)

  : isky(isky1), asky(asky1)
  {
    for (Size i = 0; i < n; i++) {

	Size jmin = colmin(i);
	for (Size j = jmin; j < i; j++) {
	    T s = 0;
	    for (Size k = max(jmin, colmin(j)); k < j; k++)
		s += a(i,k)*a(j,k);
            a(i,j) -= s;
	}
	for (Size j = jmin; j < i; j++) {
 	    T c = a(i,j)/a(j,j);
	    a(i,i) -= c*a(i,j);
	    a(i,j) = c;
	}
    }
  }
  RandomAccessIteratorIndex isky;
  RandomAccessIteratorValue asky;
};
template<
    class RandomAccessIteratorIndex,
    class RandomAccessIteratorValue,
    class Size,
    class T>
inline
void 
ssk_factorize_ldlt (
    RandomAccessIteratorIndex isky,
    RandomAccessIteratorValue asky,
    Size                      n,
    T                         _value_tag)
{
    ssk_factorize_ldlt_struct
        <RandomAccessIteratorIndex, RandomAccessIteratorValue, Size, T> 
        (isky, asky, n);
}
/*@! 
 \vfill
 \pagebreak
 \mbox{}
 \vfill
 \begin{quote}
 \begin{algorithm}[h]
  \caption{Solve $A \, x = b$ using permuted $LDL^t=PAP^t$ in SSK format}
  \begin{algorithmic}
    \INPUT {The $LDL^t$ factorization, the right-hand side and the permutation}
      isky(0:n), asky(0:nnzsky-1), b(0:n-1), p(0:n-1)
    \ENDINPUT
    \OUTPUT {The solution}
      x(0:n-1)
    \ENDINOUT
    \mbox{}\\
    \begin{tabular}{lccr}
     {\bf alias} & colmin(i) & = & isky(i) - isky(i+1) + i + 1 \\
     {\bf alias} & a(i,j)    & = & asky(j + isky(i+1) - i - 1)
    \end{tabular}
    \mbox{}\\
    \LOCAL {use a temporary (in-place permutation is slower)}
      \~x(0:n-1)
    \ENDLOCAL
    \mbox{}\\
    \mbox{}\\
    \BEGIN 
      {\em solve the lower unitary triangular system with permuted r.h.s.} \\
      \FORTO {i := 0} {n-1}
	\~x(i) := b(p(i)) \\
	\FORTO {j := colmin(i)} {i-1}
	    \~x(i) -= a(i,j)*\~x(j)
        \ENDFOR
      \ENDFOR
      {\em solve the diagonal system} \\
      \FORTO {i := 0} {n-1}
	\~x(i) /= a(i,i)
      \ENDFOR
      {\em solve the upper unitary triangular system} \\
      \FORTOSTEP {i := n-1} {1} {-1}
	\FORTO {j := colmin(i)} {i-1}
	    \~x(j) -= a(i,j)*\~x(i)
        \ENDFOR
      \ENDFOR
      {\em permut back to the solution} \\
      \FORTO {i := 0} {n-1}
	x(p(i)) := \~x(i)
      \ENDFOR
    \END
  \end{algorithmic}
 \end{algorithm}
\end{quote}
\vfill
 \pagebreak
 \mbox{}
 \vfill
*/
template<
    class RandomAccessIterator1,
    class RandomAccessIterator2,
    class RandomAccessIterator3,
    class RandomAccessIterator4,
    class RandomAccessIterator5,
    class Size,
    class T>
struct
ssk_solve_ldlt_struct
{
  ssk_solve_ldlt_struct (
    RandomAccessIterator1 isky1,
    RandomAccessIterator2 asky1,
    RandomAccessIterator3 b,
    RandomAccessIterator5 x,
    Size                  n)

  : isky(isky1), asky(asky1)
  {
    for (Size i = 0; i < n; i++) {
	T s = b[i];
	for (Size j = colmin(i); j < i; j++)
	    s -= a(i,j)*x[j];
	x[i] = s;
    }
    for (Size i = 0; i < n; i++) {
	x[i] /= a(i,i);
    }
    for (Size i = n-1; i >= 1; i--) {
    	T c = x[i];
	for (Size j = colmin(i); j < i; j++)
	    x[j] -= a(i,j)*c;
    }
  }
  ssk_solve_ldlt_struct (
    RandomAccessIterator1 isky1,
    RandomAccessIterator2 asky1,
    RandomAccessIterator3 b,
    RandomAccessIterator4 p,
    RandomAccessIterator5 x,
    Size                  n)

  : isky(isky1), asky(asky1)
  {
    std::vector<T> bb(n), xx(n);

    for (Size i = 0; i < n; i++)
	bb[p[i]] = b[i];
    for (Size i = 0; i < n; i++) {
	T s = bb[i];
	for (Size j = colmin(i); j < i; j++)
	    s -= a(i,j)*xx[j];
	xx[i] = s;
    }
    for (Size i = 0; i < n; i++) {
	xx[i] /= a(i,i);
    }
    for (Size i = n-1; i >= 1; i--) {
    	T c = xx[i];
	for (Size j = colmin(i); j < i; j++)
	    xx[j] -= a(i,j)*c;
    }
    for (Size i = 0; i < n; i++)
	x[i] = xx[p[i]];
  }
  inline Size colmin (Size i)        { return isky[i] - isky[i+1] + i + 1; }
  inline const T& a (Size i, Size j) { return asky [j + isky[i+1] - i - 1]; }
  RandomAccessIterator1  isky;
  RandomAccessIterator2 asky;
};
template<
    class RandomAccessIterator1,
    class RandomAccessIterator2,
    class RandomAccessIterator3,
    class RandomAccessIterator5,
    class Size,
    class T>
inline
void 
ssk_solve_ldlt (
    RandomAccessIterator1 isky,
    RandomAccessIterator2 asky,
    RandomAccessIterator3 b,
    RandomAccessIterator5 x,
    Size                  n,
    T                    _value_tag)
{
    ssk_solve_ldlt_struct
        <RandomAccessIterator1, RandomAccessIterator2, 
	 RandomAccessIterator3, const unsigned int *, 
	 RandomAccessIterator5, Size, T> 
        (isky, asky, b, x, n);
}
template<
    class RandomAccessIterator1,
    class RandomAccessIterator2,
    class RandomAccessIterator3,
    class RandomAccessIterator4,
    class RandomAccessIterator5,
    class Size,
    class T>
inline
void 
ssk_solve_ldlt (
    RandomAccessIterator1 isky,
    RandomAccessIterator2 asky,
    RandomAccessIterator3 b,
    RandomAccessIterator4 p,
    RandomAccessIterator5 x,
    Size                  n,
    T                     _value_tag)
{
    ssk_solve_ldlt_struct
        <RandomAccessIterator1, RandomAccessIterator2, 
	 RandomAccessIterator3, RandomAccessIterator4, 
	 RandomAccessIterator5, Size, T> 
        (isky, asky, b, p, x, n);
}
//@!\vfill
}// namespace rheolef
# endif // _SKIT_SSK_ALGORITHM_H

