#ifndef _RHEO_NUMBERING_H
#define _RHEO_NUMBERING_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================
#include "rheolef/geo_element.h"
#include "rheolef/geo_size.h"
#include "rheolef/basis.h"
#include "rheolef/disarray.h"

namespace rheolef { 

// this file "numbering.h" is included by "geo.h",
// but the "numbering" refers to the "geo_basic" class
// thus, we use a forward declaration:
template <class T, class M> class geo_basic;


template <class T, class M>
class numbering_rep {
public:
  typedef size_t size_type;
  virtual ~numbering_rep() {}
  virtual std::string name() const = 0;
  virtual size_type degree () const = 0;
  virtual void set_degree (size_type degree) const = 0;
  virtual bool is_continuous() const = 0;
  virtual bool has_compact_support_inside_element() const = 0;
  const basis_basic<T>& get_basis() const { return _basis; }
  virtual size_type     ndof (const geo_size& gs, size_type map_dim) const = 0;
  virtual size_type dis_ndof (const geo_size& gs, size_type map_dim) const = 0;
  virtual void      dis_idof (const geo_size& gs, const geo_element& K, std::vector<size_type>& dis_idof) const = 0;
  virtual void      set_ios_permutations (const geo_basic<T,M>& omega,
    		disarray<size_type,M>&   idof2ios_dis_idof,
    		disarray<size_type,M>&   ios_idof2dis_idof) const = 0;

  static numbering_rep* make_ptr (std::string name);
protected:
  mutable basis_basic<T> _basis;
};

/*Class:numbering
NAME: @code{numbering} - global degree of freedom numbering
@cindex  numbering, global degree of freedom
@cindex  polynomial basis
@clindex numbering
SYNOPSYS:
  @noindent
  The @code{numbering} class defines methods that furnish global
  numbering of degrees of freedom. This numbering depends upon
  the degrees of polynoms on elements and upon the continuity
  requirement at inter-element boundary. For instance the
  "P1" continuous finite element approximation has one degree
  of freedom per vertice of the mesh, while its discontinuous
  counterpart has dim(basis) times the number of elements of the
  mesh, where dim(basis) is the size of the local finite element basis.

AUTHORS:
    LMC-IMAG, 38041 Grenoble cedex 9, France
   | Pierre.Saramito@imag.fr
DATE:   7 january 2004
End:
*/

//<numbering:
template <class T, class M = rheo_default_memory_model>
class numbering : public smart_pointer<numbering_rep<T,M> > {
public:

// typedefs:

  typedef numbering_rep<T,M> rep;
  typedef smart_pointer<rep> base;
  typedef size_t             size_type;

// allocators:

  numbering (std::string name = "");
  numbering (numbering_rep<T,M>* ptr);
  ~numbering() {}

// accessors & modifiers:

  bool is_initialized() const { return base::operator->() != 0; }
  std::string name() const;
  size_type degree () const;
  void set_degree (size_type degree) const;
  bool is_continuous()    const;
  bool is_discontinuous() const { return !is_continuous(); }
  bool has_compact_support_inside_element()    const;
  const basis_basic<T>& get_basis() const { return base::data().get_basis(); }

  size_type     ndof   (const geo_size& gs, size_type map_dim) const;
  size_type dis_ndof   (const geo_size& gs, size_type map_dim) const;
  void      dis_idof   (const geo_size& gs, const geo_element& K, std::vector<size_type>& dis_idof) const;
  void set_ios_permutations (const class geo_basic<T,M>& omega,
    		disarray<size_type,M>&   idof2ios_dis_idof,
    		disarray<size_type,M>&   ios_idof2dis_idof) const;

// comparator:
 
  bool operator== (const numbering<T,M>& y) const {
	if (! is_initialized() && ! y.is_initialized()) return true;
	if (! is_initialized() || ! y.is_initialized()) return false;
        return name() == y.name();
  }
// i/o:

  void dump(std::ostream& out = std::cerr) const;
};
//>numbering:
// -----------------------------------------------------------
// inlined
// -----------------------------------------------------------
template <class T, class M>
inline
numbering<T,M>::numbering(std::string name)
 : base (rep::make_ptr(name))
{
}
template <class T, class M>
inline
numbering<T,M>::numbering (numbering_rep<T,M>* ptr)
 : base (ptr)
{
}
template <class T, class M>
inline
std::string
numbering<T,M>::name() const
{
  return base::data().name();
}
template <class T, class M>
inline
typename numbering<T,M>::size_type
numbering<T,M>::degree () const
{
    return base::data().degree();
}
template <class T, class M>
inline
void 
numbering<T,M>::set_degree (size_type degree) const
{
    base::data().set_degree(degree);
}
template <class T, class M>
inline
bool
numbering<T,M>::is_continuous () const   
{
    return base::data().is_continuous();
}
template <class T, class M>
inline
bool
numbering<T,M>::has_compact_support_inside_element () const   
{
    return base::data().has_compact_support_inside_element();
}
template <class T, class M>
inline
void
numbering<T,M>::dump(std::ostream& out) const
{
    out << "numbering " << name() << std::endl;
}
template <class T, class M>
inline
typename numbering<T,M>::size_type
numbering<T,M>::ndof (
  	const geo_size& gs,
        size_type       map_dim) const
{
    return base::data().ndof (gs, map_dim);
}
template <class T, class M>
inline
typename numbering<T,M>::size_type
numbering<T,M>::dis_ndof (
  	const geo_size& gs,
        size_type       map_dim) const
{
    return base::data().dis_ndof (gs, map_dim);
}
template <class T, class M>
inline
void
numbering<T,M>::dis_idof (
  	const geo_size&       gs,
	const geo_element&    K, 
	std::vector<size_type>&    dis_idof) const
{
    return base::data().dis_idof (gs, K, dis_idof);
}
template <class T, class M>
inline
void
numbering<T,M>::set_ios_permutations (
	const class geo_basic<T,M>& omega,
    	disarray<size_type,M>&   idof2ios_dis_idof,
    	disarray<size_type,M>&   ios_idof2dis_idof) const
{
    return base::data().set_ios_permutations (omega, idof2ios_dis_idof, ios_idof2dis_idof);
}
}// namespace rheolef
#endif // _RHEO_NUMBERING_H
