///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "rheolef/quadrature.h"
#include "rheolef/reference_element_face_transformation.h"
namespace rheolef {
using namespace std;

static const char* static_family_name[quadrature_option_type::max_family+1] = {
	"gauss"           ,
	"gauss_lobatto"   ,
	"gauss_radau"	  ,
	"middle_edge"     ,
	"superconvergent" ,
	"equispaced"      ,
	"undefined"
};
static 
quadrature_option_type::family_type
family_name2type (string name)
{
  for (size_t i =  0; i < quadrature_option_type::max_family; ++i) {
    if (static_family_name[i] == name) {
      return (quadrature_option_type::family_type) (i);
    }
  }
  error_macro ("unexpected quadrature family name `" << name << "'");
  return quadrature_option_type::max_family;
}
void
quadrature_option_type::set_family (string name)
{
  set_family (family_name2type (name));
}
string
quadrature_option_type::get_family_name() const
{
  check_macro (_family >= 0 && _family <= max_family,
	"unexpected quadrature family number = " << _family);
  return static_family_name[_family];
}
// ----------------------------------------------------------------------------
template<class T>
void
quadrature_on_geo<T>::initialize (reference_element hat_K,  quadrature_option_type opt)
{
    base::resize (0);
    switch (hat_K.variant()) {
      case reference_element::p:  init_point       (opt); break;
      case reference_element::e:  init_edge        (opt); break;
      case reference_element::t:  init_triangle    (opt); break;
      case reference_element::q:  init_square      (opt); break;
      case reference_element::T:  init_tetrahedron (opt); break;
      case reference_element::P:  init_prism       (opt); break;
      case reference_element::H:  init_hexahedron  (opt); break;
      default:
	fatal_macro ("quadrature formula on element type `"
	  << hat_K.name() << "' are not yet implemented.");
    }
}
// ----------------------------------------------------------------------------
template<class T>
quadrature_rep<T>::quadrature_rep (quadrature_option_type opt)
 : _options(opt),
   _quad(),
   _initialized           (reference_element::max_variant, false)
{
}
template<class T>
quadrature_rep<T>::quadrature_rep (const quadrature_rep<T>& q)
 : _options               (q._options),
   _quad                  (q._quad),
   _initialized           (q._initialized)
{}
template<class T>
const quadrature_rep<T>&
quadrature_rep<T>::operator= (const quadrature_rep<T>& q) {
       _options                = q._options;
       _quad                   = q._quad;
       _initialized            = q._initialized;
       return *this;
}
template<class T>
void 
quadrature_rep<T>::_initialize (reference_element hat_K) const
{
    reference_element::variant_type K_type = hat_K.variant();
    _quad[K_type].initialize (hat_K, _options);
    _initialized [K_type] = true;
}
#ifdef TO_CLEAN
// was used by DG: integration on internal sides, for e.g. jump(u), etc
// 
// quadrature on a side S of an element K:
// the transformation from reference side hat_S into reference element tilde_K and its side tilde_S
// is linear:
// => linear-transform quadrature node and rescale weights
//
template<class T>
void 
quadrature_rep<T>::side_initialize (
    reference_element             tilde_K,
    const side_information_type&  sid) const
{
  reference_element::variant_type K_type  = tilde_K.variant();
  reference_element::variant_type S_type  = sid.hat.variant();
  if (!_initialized_side_cache [S_type]) {
    _quad_side_cache [S_type].initialize (sid.hat, _options);
    _initialized_side_cache [S_type] = true;
  }
  _quad[K_type].resize (_quad_side_cache[S_type].size());
  T coef = measure(tilde_K)/sid.hat.side_measure(sid.loc_isid);
  for (size_type q = 0, nq = _quad.size(); q < nq; ++q) {
    const point_basic<T>& hat_xq = _quad_side_cache[S_type][q].x;
    const T&              hat_wq = _quad_side_cache[S_type][q].w;
    _quad[K_type][q].x = reference_element_face_transformation (tilde_K, sid, hat_xq);
    _quad[K_type][q].w = coef*hat_wq;
  }
  _initialized [K_type] = true;
}
#endif // TO_CLEAN
template<class T>
typename quadrature_rep<T>::const_iterator 
quadrature_rep<T>::begin (reference_element hat_K) const
{
    reference_element::variant_type K_type = hat_K.variant();
    if (!_initialized [K_type]) _initialize (hat_K);
    return _quad[K_type].begin();
}
template<class T>
typename quadrature_rep<T>::const_iterator 
quadrature_rep<T>::end (reference_element hat_K) const
{
    reference_element::variant_type K_type = hat_K.variant();
    if (!_initialized [K_type]) _initialize (hat_K);
    return _quad[K_type].end();
}
template<class T>
typename quadrature_rep<T>::size_type 
quadrature_rep<T>::size (reference_element hat_K) const
{
    reference_element::variant_type K_type = hat_K.variant();
    if (!_initialized [K_type]) _initialize (hat_K);
    return _quad[K_type].size();
}
template<class T>
const weighted_point<T>&
quadrature_rep<T>::operator() (reference_element hat_K, size_type q) const
{
    reference_element::variant_type K_type = hat_K.variant();
    if (!_initialized [K_type]) _initialize (hat_K);
    return _quad[K_type][q];
}
template<class T>
ostream&
operator<< (ostream& out, const quadrature_on_geo<T>& x)
{
  out << setprecision (numeric_limits<T>::digits10)
      << x.size() << endl;
  for (size_t r = 0; r < x.size(); r++)
    out << x[r].x << "\t" << x[r].w << endl;
  return out;
}
template<class T>
ostream&
operator<< (ostream& out, const quadrature_rep<T>& x)
{
  out << "quadrature" << endl
      << x._options.get_family_name() << " " << x._options.get_order()  << endl;
  for (size_t i = 0; i != size_t(reference_element::max_variant); i++) {
    reference_element::variant_type K_type = reference_element::variant_type(i);
    if (! x._initialized [K_type]) continue;
    reference_element hat_K (K_type);
    out << "reference_element " << hat_K.name() << endl
        << x._quad[K_type];
  }
  out << "end_quadrature" << endl;
  return out;
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
#define _RHEOLEF_instanciation(T)               \
template class quadrature_on_geo<Float>;	\
template class quadrature_rep<Float>;		\
template ostream& operator<< (ostream&, const quadrature_on_geo<T>&); 	\
template ostream& operator<< (ostream&, const quadrature_rep<T>&);

_RHEOLEF_instanciation(Float)

} // namespace rheolef
