#ifndef _RHEOLEF_PAIR_SET_H
#define _RHEOLEF_PAIR_SET_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
// same as index_set, but with a pair<size_t,T> instead of a size_t
// wrapper arround map<size_t,T> with goodies such as union as a+b
//
// motivation: this class is usefull for array<pair_set,M>
// to send/receive variable-sized lists via MPI correctly
// array<pair_set,M> is used by the asr class
// send & receive of such lists are used to assembly a global matrix locally
// when computing the csr*csr product
//
// author: Pierre.Saramito@imag.fr
//
// date: 19 mai 2012
//
#include "rheolef/distributed.h"
#include "rheolef/pretty_name.h"
#include "rheolef/container_traits.h"
#ifdef _RHEOLEF_HAVE_MPI
#include "rheolef/mpi_pair_datatype.h"
#endif // _RHEOLEF_HAVE_MPI

#include <boost/serialization/map.hpp>
#include <boost/serialization/base_object.hpp>

namespace rheolef {

/*Class:
NAME:  pair_set - a set of (index,value) pair (@PACKAGE@-@VERSION@)
SYNOPSYS:       
    A class for: l = @{(0,3.3),...(7,8.2)@} i.e. a wrapper for STL @code{map<size_t,T>} with 
    some assignment operators, such as l1 += l2.
    This class is suitable for use with the @code{array<T>} class,
    as @code{array<pair_set>} (@pxref{array class}).
TODO: 
    template <T,A> with A=std::allocator or heap_allocator
AUTHOR: Pierre.Saramito@imag.fr
DATE: date: 19 may 2012
End:
*/
//<verbatim:
template<class T, class A = std::allocator<std::pair<std::size_t,T> > >
class pair_set: public std::map<std::size_t, T, std::less<std::size_t>, A> {
public:

// typedefs:

  typedef std::size_t                    size_type;
  typedef std::pair<size_type,T>         pair_type;
  typedef std::pair<const size_type,T>   const_pair_type;
  typedef pair_type                      value_type;
  typedef A                              allocator_type;
  typedef std::map<size_type, T, std::less<size_type>, allocator_type>
                                         base;
  typedef typename base::iterator        iterator;
  typedef typename base::const_iterator  const_iterator;

// allocators:

  pair_set (const A& alloc = A());
  pair_set (const pair_set<T,A>& x, const A& alloc = A());
  pair_set<T,A>& operator= (const pair_set<T,A>& x);
  void clear ();

// basic algebra: semantic of a sparse vector

  pair_set<T,A>& operator+= (const pair_type&     x); // c := a union {x}
  template<class B>
  pair_set<T,A>& operator+= (const pair_set<T,B>& b); // c := a union b

// boost mpi:

  template <class Archive>
  void serialize (Archive& ar, const unsigned int version);
};
// io:
template <class T, class A>
std::istream& operator>> (std::istream& is,       pair_set<T,A>& a);
template <class T, class A>
std::ostream& operator<< (std::ostream& os, const pair_set<T,A>& b);
//>verbatim:

// operator += for array::assembly
template <class T>
struct pair_set_add_op : std::binary_function<T,T,T> {
    T& operator()(T& x, const T&                           y) const { return x += y; }
    T& operator()(T& x, const typename T::pair_type&       y) const { return x += y; }
    T& operator()(T& x, const typename T::const_pair_type& y) const { return x += y; }
};
// for boost mpi and array<pair_set>:
template <class T, class A>
struct default_set_op<pair_set<T,A> > {
  typedef pair_set_add_op<pair_set<T,A> > type;
};
template <class T, class A>
struct is_container<pair_set<T,A> > : boost::mpl::true_ {
  typedef boost::mpl::true_ type;
};
#ifdef _RHEOLEF_HAVE_MPI
template <class T, class A>
struct is_container_of_mpi_datatype<pair_set<T,A> > : boost::mpi::is_mpi_datatype<T> {
  typedef typename boost::mpi::is_mpi_datatype<T>::type type;
};
#endif // _RHEOLEF_HAVE_MPI
// -------------------------------------------------------------------
// inlined
// -------------------------------------------------------------------
template <class T, class A>
inline
pair_set<T,A>::pair_set (const A& alloc)
 : base (std::less<size_type>(), alloc)
{
}
template <class T, class A>
inline
void
pair_set<T,A>::clear ()
{
  base::clear();
}
template <class T, class A>
inline
pair_set<T,A>&
pair_set<T,A>::operator+= (const pair_type& x)
{
  iterator p = base::find(x.first);
  if (p == base::end()) {
    // insert a new element
    base::insert (x);
  } else {
    // increment an existing element
    (*p).second += x.second;
  }
  return *this;
}
template <class T, class A>
template <class Archive>
void
pair_set<T,A>::serialize (Archive& ar, const unsigned int version)
{
  ar & boost::serialization::base_object<base>(*this);
}
// -------------------------------------------------------------------
// not inlined
// -------------------------------------------------------------------
template <class T, class A>
pair_set<T,A>::pair_set (const pair_set& a, const A& alloc)
 : base(std::less<size_type>(), a.get_allocator())
{
  for (const_iterator iter = a.base::begin(), last = a.base::end(); iter != last; iter++) {
    base::insert (*iter);
  }
}
template <class T, class A>
pair_set<T,A>&
pair_set<T,A>::operator= (const pair_set<T,A>& a)
{
  base::clear();
  for (const_iterator iter = a.base::begin(), last = a.base::end(); iter != last; iter++) {
    base::insert (*iter);
  }
  return *this;
}
template <class T, class A>
template <class B>
pair_set<T,A>&
pair_set<T,A>::operator+= (const pair_set<T,B>& b)
{
  for (typename pair_set<T,B>::const_iterator iter = b.begin(), last = b.end(); iter != last; iter++) {
    operator+= (*iter);
  }
  return *this;
}
template <class T, class A>
std::istream&
operator>> (std::istream& is, pair_set<T,A>& a)
{
  typedef typename pair_set<T,A>::size_type size_type;
  typedef typename pair_set<T,A>::pair_type pair_type;
  size_type n;
  is >> n;
  a.clear();
  for (size_type i = 0; i < n; i++) {
    pair_type xi;
    is >> xi.first >> xi.second;
    a.insert (xi);
  }
  return is;
}
template <class T, class A>
std::ostream&
operator<< (std::ostream& os, const pair_set<T,A>& a)
{
  typedef typename pair_set<T,A>::size_type size_type;
  typedef typename pair_set<T,A>::const_iterator const_iterator;
  os << a.size() << "\t";	
  for (const_iterator iter = a.begin(), last = a.end(); iter != last; iter++) {
    os << " " << (*iter).first << " " << (*iter).second;
  }
  return os;
}

} // namespace rheolef
#endif // _RHEOLEF_PAIR_SET_H
