/*
 * QuteCom, a voice over Internet phone
 * Copyright (C) 2010 Mbdsys
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/**
 * @author Mathieu Stute <mstute@gmail.com>
 */

#include "LinuxVolumeControl.h"
#include "EnumDeviceType.h"

#include "alsa_mixer.h"

#include <util/Logger.h>

#include <string>

std::string LinuxVolumeControl::mixerNameRewrite(const std::string &name) {
	size_t pos;
	if((pos = name.find("hw:")) != std::string::npos) {
		pos += 3; // strlen("hw:")
	} else if((pos = name.find("dmix:")) != std::string::npos) {
		pos += 5; // strlen("dmix:")
	} else {
		// no rewrite needed
		return name;
	}
	// get the first number
	size_t endpos = name.find_first_not_of("0123456789", pos);
	return std::string("hw:") + name.substr(pos, endpos-pos);
}


LinuxVolumeControl::LinuxVolumeControl(const AudioDevice & audioDevice) :
	_audioDevice(audioDevice),
	// Rewrite device names like "hw:0,1", "plughw:0", "dmix:0" to "hw:0" in order to be able to open the mixer
	_strDeviceName(mixerNameRewrite(audioDevice.getData()[1]))
{

	EnumDeviceType::DeviceType deviceType =
		EnumDeviceType::toDeviceType(audioDevice.getData()[2]);

	switch (deviceType) {
	case EnumDeviceType::DeviceTypeMicrophoneIn:
	case EnumDeviceType::DeviceTypeWaveIn:
		_deviceType = CAPTURE_VOLUME;
		// if mixer "Capture" is not present, fall back to "Mic"
		if(getLevel() < 0) {
			_deviceType = MIC_CAPTURE_VOLUME;
		}
		break;
	case EnumDeviceType::DeviceTypeMasterVolume:
	case EnumDeviceType::DeviceTypeWaveOut:
		_deviceType = MASTER_VOLUME;
		break;
	default:
		LOG_FATAL("Unknown audio device type");
	}
}

int LinuxVolumeControl::getLevel() {
	int level;
	if(get_mixer_level(_strDeviceName.c_str(), _deviceType, &level) < 0) {
		return -1;
	}
	return level;
}

bool LinuxVolumeControl::setLevel(unsigned level) {
	return set_mixer_level(_strDeviceName.c_str(), _deviceType, level) >= 0;
}

bool LinuxVolumeControl::isMuted() {
	return false;
}

bool LinuxVolumeControl::setMute(bool mute) {
	return false;
}

bool LinuxVolumeControl::isSettable() const {
	return true;
}
