/***************************************************************************
 *   Copyright (C) 2007 by Anistratov Oleg                                 *
 *   ower86@gmail.com                                                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation;                         *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 ***************************************************************************/
#include "shortcutseditor.h"

#include <QGridLayout>
#include <QMapIterator>

#include "qchatsettings.h"
#include "shortcutgrabber.h"

ShortcutsEditor::ShortcutsEditor(QWidget *parent)
 : QDialog(parent)
{
  QGridLayout*     grid = new QGridLayout(this);
  QStringList      list;

  list.append(tr("Action"));
  list.append(tr("Shortcut"));

  m_treeWgt          = new QTreeWidget(this);
  m_grabShortcutDlg  = new ShortcutGrabber(this);
  m_okBtn            = new QPushButton(tr("Ok"), this);
  m_cancelBtn        = new QPushButton(tr("Cancel"), this);
  m_defaultsBtn      = new QPushButton(tr("Defaults"), this);
  m_clearShortcutBtn = new QPushButton(tr("Clear Shortcut"), this);

  grid->addWidget(m_treeWgt         , 0, 0, 5, 1);
  grid->addWidget(m_okBtn           , 0, 1);
  grid->addWidget(m_cancelBtn       , 1, 1);
  grid->addWidget(m_defaultsBtn     , 2, 1);
  grid->addWidget(m_clearShortcutBtn, 3, 1);

  m_treeWgt->setColumnCount(2);
  m_treeWgt->setHeaderLabels(list);

  m_treeWgt->setAlternatingRowColors(true);

  connect(m_treeWgt, SIGNAL(itemDoubleClicked(QTreeWidgetItem*, int)), this, SLOT(grabShortcutDlg(QTreeWidgetItem*, int)));
  connect(m_treeWgt, SIGNAL(currentItemChanged(QTreeWidgetItem*, QTreeWidgetItem*)),
               this, SLOT  (currentItemChanged(QTreeWidgetItem*, QTreeWidgetItem*)));

  connect(m_okBtn           , SIGNAL(clicked()), this, SLOT(applyChanges()));
  connect(m_cancelBtn       , SIGNAL(clicked()), this, SLOT(reject()));
  connect(m_defaultsBtn     , SIGNAL(clicked()), this, SLOT(defaults()));
  connect(m_clearShortcutBtn, SIGNAL(clicked()), this, SLOT(clearCurrentShortcut()));


  m_shortcuts = QChatSettings::settings()->allShortcuts();

  init();

  m_clearShortcutBtn->setEnabled(false);

  setWindowTitle("Configure Shortcuts");

  m_treeWgt->setAnimated(true);
}

ShortcutsEditor::~ShortcutsEditor()
{
}

void ShortcutsEditor::init()
{
  QMapIterator< QString, QList<QKeySequence> > sh(m_shortcuts);
  QStringList      list;
  QTreeWidgetItem* item;
  QString          group;

  QMap<QString, QTreeWidgetItem*> top_lvl_items;
  QTreeWidgetItem* group_item;

  while(sh.hasNext())
  {
    sh.next();

    list.clear();
    list.append(QChatSettings::settings()->shortcutDisplayName(sh.key()));

    if(sh.value().size() > 0)
      list.append((sh.value()[0]).toString());

    group = QChatSettings::settings()->shortcutGroup(sh.key());

    if(!top_lvl_items.contains(group))
    {
      group_item = new QTreeWidgetItem(m_treeWgt, QStringList(group));
      top_lvl_items.insert(group, group_item);
    }
    else
      group_item = top_lvl_items[group];

    item = new QTreeWidgetItem(group_item, list);
    item->setData(0, Qt::UserRole, QVariant(sh.key()));
  }

  m_treeWgt->sortItems(0, Qt::AscendingOrder);

  m_treeWgt->expandAll();

  m_treeWgt->resizeColumnToContents(0);
  m_treeWgt->resizeColumnToContents(1);
}

void ShortcutsEditor::grabShortcutDlg(QTreeWidgetItem* item, int)
{
  QString act_name = item->data(0, Qt::UserRole).toString();

  if(m_shortcuts.contains(act_name))
  {
    if(m_grabShortcutDlg->execute())
    {
      QKeySequence seq = m_grabShortcutDlg->sequence();

//       if(!m_shortcuts[act_name].contains(seq))
        m_shortcuts[act_name].clear();
        m_shortcuts[act_name].append(seq);

      item->setText(1, seq.toString());
    }
  }
}

void ShortcutsEditor::applyChanges()
{
  QChatSettings::settings()->setAllShortcuts(m_shortcuts);
  accept();
}

void ShortcutsEditor::defaults()
{
  QChatSettings prefs;

  m_treeWgt->clear();

  m_shortcuts = prefs.allShortcuts();

  init();

  m_clearShortcutBtn->setEnabled(false);

  m_treeWgt->show();
}

void ShortcutsEditor::clearCurrentShortcut()
{
  QTreeWidgetItem* current = m_treeWgt->currentItem();

  if(current)
  {
    QString act_name = current->data(0, Qt::UserRole).toString();

    if(m_shortcuts.contains(act_name))
      m_shortcuts[act_name].clear();

    m_clearShortcutBtn->setEnabled(false);

    current->setText(1, "");
  }
}

void ShortcutsEditor::currentItemChanged(QTreeWidgetItem* current, QTreeWidgetItem* /*previous*/)
{
  if(!current)
    return;

  QString act_name = current->data(0, Qt::UserRole).toString();

  m_clearShortcutBtn->setEnabled((m_shortcuts.contains(act_name) && m_shortcuts[act_name].size() > 0));
}
