# This file is part of the Trezor project.
#
# Copyright (C) 2012-2022 SatoshiLabs and contributors
#
# This library is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the License along with this library.
# If not, see <https://www.gnu.org/licenses/lgpl-3.0.html>.

import pytest

from trezorlib import messages, tools

VECTORS = (  # descriptor, checksum
    (
        "pkh([5c9e228d/44h/0h/0h]xpub6BiVtCpG9fQPxnPmHXG8PhtzQdWC2Su4qWu6XW9tpWFYhxydCLJGrWBJZ5H6qTAHdPQ7pQhtpjiYZVZARo14qHiay2fvrX996oEP42u8wZy/0/*)",
        "lnrxp640",
    ),
    (
        "pkh([5c9e228d/44h/0h/0h]xpub6BiVtCpG9fQPxnPmHXG8PhtzQdWC2Su4qWu6XW9tpWFYhxydCLJGrWBJZ5H6qTAHdPQ7pQhtpjiYZVZARo14qHiay2fvrX996oEP42u8wZy/1/*)",
        "w8x8u09h",
    ),
    (
        "sh(wpkh([5c9e228d/49h/0h/0h]xpub6CVKsQYXc9awxgV1tWbG4foDvdcnieK2JkbpPEBKB5WwAPKBZ1mstLbKVB4ov7QzxzjaxNK6EfmNY5Jsk2cG26EVcEkycGW4tchT2dyUhrx/0/*))",
        "kx6mu23v",
    ),
    (
        "sh(wpkh([5c9e228d/49h/0h/0h]xpub6CVKsQYXc9awxgV1tWbG4foDvdcnieK2JkbpPEBKB5WwAPKBZ1mstLbKVB4ov7QzxzjaxNK6EfmNY5Jsk2cG26EVcEkycGW4tchT2dyUhrx/1/*))",
        "r85dy4yn",
    ),
    (
        "wpkh([5c9e228d/84h/0h/0h]xpub6DDUPHpUo4pcy43iJeZjbSVWGav1SMMmuWdMHiGtkK8rhKmfbomtkwW6GKs1GGAKehT6QRocrmda3WWxXawpjmwaUHfFRXuKrXSapdckEYF/0/*)",
        "vyj8qz0q",
    ),
    (
        "wpkh([5c9e228d/84h/0h/0h]xpub6DDUPHpUo4pcy43iJeZjbSVWGav1SMMmuWdMHiGtkK8rhKmfbomtkwW6GKs1GGAKehT6QRocrmda3WWxXawpjmwaUHfFRXuKrXSapdckEYF/1/*)",
        "ashxahlc",
    ),
)


@pytest.mark.parametrize("descriptor, checksum", VECTORS)
def test_descriptor_checksum(descriptor, checksum):
    assert tools.descriptor_checksum(descriptor) == checksum


BASE58_VECTORS = (  # data_hex, encoding_b58
    ("", ""),
    ("61", "2g"),
    ("626262", "a3gV"),
    ("636363", "aPEr"),
    ("73696d706c792061206c6f6e6720737472696e67", "2cFupjhnEsSn59qHXstmK2ffpLv2"),
    (
        "00eb15231dfceb60925886b67d065299925915aeb172c06647",
        "1NS17iag9jJgTHD1VXjvLCEnZuQ3rJDE9L",
    ),
    ("516b6fcd0f", "ABnLTmg"),
    ("bf4f89001e670274dd", "3SEo3LWLoPntC"),
    ("572e4794", "3EFU7m"),
    ("ecac89cad93923c02321", "EJDM8drfXA6uyA"),
    ("10c8511e", "Rt5zm"),
    ("00000000000000000000", "1111111111"),
    ("00" * 32, "11111111111111111111111111111111"),
    (
        "000111d38e5fc9071ffcd20b4a763cc9ae4f252bb4e48fd66a835e252ada93ff480d6dd43dc62a641155a5",
        "123456789ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnopqrstuvwxyz",
    ),
    (
        "000102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f202122232425262728292a2b2c2d2e2f303132333435363738393a3b3c3d3e3f404142434445464748494a4b4c4d4e4f505152535455565758595a5b5c5d5e5f606162636465666768696a6b6c6d6e6f707172737475767778797a7b7c7d7e7f808182838485868788898a8b8c8d8e8f909192939495969798999a9b9c9d9e9fa0a1a2a3a4a5a6a7a8a9aaabacadaeafb0b1b2b3b4b5b6b7b8b9babbbcbdbebfc0c1c2c3c4c5c6c7c8c9cacbcccdcecfd0d1d2d3d4d5d6d7d8d9dadbdcdddedfe0e1e2e3e4e5e6e7e8e9eaebecedeeeff0f1f2f3f4f5f6f7f8f9fafbfcfdfeff",
        "1cWB5HCBdLjAuqGGReWE3R3CguuwSjw6RHn39s2yuDRTS5NsBgNiFpWgAnEx6VQi8csexkgYw3mdYrMHr8x9i7aEwP8kZ7vccXWqKDvGv3u1GxFKPuAkn8JCPPGDMf3vMMnbzm6Nh9zh1gcNsMvH3ZNLmP5fSG6DGbbi2tuwMWPthr4boWwCxf7ewSgNQeacyozhKDDQQ1qL5fQFUW52QKUZDZ5fw3KXNQJMcNTcaB723LchjeKun7MuGW5qyCBZYzA1KjofN1gYBV3NqyhQJ3Ns746GNuf9N2pQPmHz4xpnSrrfCvy6TVVz5d4PdrjeshsWQwpZsZGzvbdAdN8MKV5QsBDY",
    ),
)


@pytest.mark.parametrize("data_hex,encoding_b58", BASE58_VECTORS)
def test_b58encode(data_hex, encoding_b58):
    assert tools.b58encode(bytes.fromhex(data_hex)) == encoding_b58


@pytest.mark.parametrize("data_hex,encoding_b58", BASE58_VECTORS)
def test_b58decode(data_hex, encoding_b58):
    assert tools.b58decode(encoding_b58).hex() == data_hex


def test_return_success_deprecation(recwarn):
    def mkfoo() -> str:
        ret = tools._return_success(messages.Success(message="foo"))
        assert ret is not None  # too bad we can't hook "is None" check
        return ret

    # check that just returning success will not cause a warning
    mkfoo()
    assert len(recwarn) == 0

    with pytest.deprecated_call():
        # equality is deprecated
        assert mkfoo() == "foo"
    with pytest.deprecated_call():
        # comparison is deprecated
        assert mkfoo() < "fooa"
    with pytest.deprecated_call():
        # truthiness is deprecated
        assert mkfoo()
    with pytest.deprecated_call():
        # addition is deprecated (and hopefully all other operators)
        assert mkfoo() + "a" == "fooa"
    with pytest.deprecated_call():
        # indexing is deprecated
        assert mkfoo()[0] == "f"
    with pytest.deprecated_call():
        # methods are deprecated
        assert mkfoo().startswith("f")


def test_deprecation_helper(recwarn):
    def mkfoo() -> messages.Success:
        return tools._deprecation_retval_helper(messages.Success(message="foo"))

    # check that just returning success will not cause a warning
    mkfoo()
    assert len(recwarn) == 0

    with pytest.deprecated_call():
        # attributes are deprecated
        assert mkfoo().message == "foo"

    with pytest.deprecated_call():
        # equality is deprecated (along with other operators hopefully)
        assert mkfoo() != "foo"
