/*	Streams

PIRL CVS ID: Streams.java,v 1.11 2012/04/16 06:18:24 castalia Exp

Copyright (C) 2004-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package	PIRL.Utilities;

import	java.io.InputStream;
import	java.io.FileInputStream;
import	java.io.File;
import	java.io.FileNotFoundException;
import	java.io.IOException;
import	java.net.URL;
import	java.net.MalformedURLException;

import	javax.imageio.stream.ImageInputStream;
import	javax.imageio.stream.FileImageInputStream;

import	PIRL.Image_Tools.HTTP_ImageInputStream;


/**	The <i>Streams</i> class provides a collection of methods for
	handling I/O streams.
<p>
	@author	Bradford Castalia, UA/PIRL
	@version	1.11
*/
public class Streams
{
public static final String
	ID = "PIRL.Utilities.Streams (1.11 2012/04/16 06:18:24)";

//!	No object constructor, just static functions.
private Streams () {}

/**	Obtain an InputStream from a named source.
<p>
	The source String is first treated as a URL specification. If this
	fails the source is treated as a pathname to a locally accessible
	file. If no file is found the source is treated as the name of a
	system resource, which can find a file inside a jar file when the
	application is being run from a jar.
<p>
	@param	source	The name of a stream source.
	@return	An InputStream, or null if no source file can be found.
*/
public static synchronized InputStream Get_Stream
	(
	String	source
	)
{
if (source == null)
	return null;
InputStream
	stream = null;

try
	{
	//	Treat the source as a URL.
	URL
		url = new URL (source);
	try {stream = url.openStream ();}
	catch (IOException exception) {/* Not accessable. */}
	}
catch (MalformedURLException exception) {}

if (stream == null)
	{
	//	Treat the source as a filename.
	File
		file = new File (source);
	try
		{
		FileInputStream
			fstream = new FileInputStream (file);
		stream = (InputStream)fstream;
		}
	catch (FileNotFoundException exception) {}
	}

if (stream == null)
	{
	//	Treat the source as a system resource.
	URL
		url = ClassLoader.getSystemResource (source);
	if (url != null)
		{
		try {stream = url.openStream ();}
		catch (IOException exception) {/* Not accessable. */}
		}
	}
return stream;
}

/**	Obtain an ImageInputStream from a named source.
<p>
	If a URL can be constructed from the source String an attempt is
	made to construct an HTTP_ImageInputStream from the URL. 

	If the source is not a valid URL, the  source is treated as a
	pathname to a locally accessible file and an attempt is made
	to construct a FileImageInputStream from the File.
<p>
	@param	source	The name of a stream source.
	@return	An ImageInputStream, or null if no source is accessible.
*/
public static synchronized ImageInputStream Get_Image_Stream
	(
	String	source
	)
{
if (source == null)
	return null;
ImageInputStream
	stream = null;

try
	{
	//	Treat the source as a URL.
	URL
		url = new URL (source);
	try {stream = new HTTP_ImageInputStream (url);}
	catch (Exception exception) {/* Not accessible */ return null;}
	}
catch (MalformedURLException exception) {}

if (stream == null)
	{
	//	Treat the source as a filename.
	File
		file = new File (source);
	try {stream = new FileImageInputStream (file);}
	catch (Exception exception) {/* Not accessible */}
	}
return stream;
}

}
