/*	Host

PIRL CVS ID: Host.java,v 1.7 2012/04/16 06:18:24 castalia Exp

Copyright (C) 2004-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.Utilities;

import	java.net.InetAddress;
import	java.net.UnknownHostException;
import	java.util.Properties;
import	java.io.IOException;


/**	The <i>Host</i> class provides information about the host on
	which the process is running.
<p>
	@author	Bradford Castalia, UA/PIRL
	@version	1.7
*/
public class Host
{
public static final String
	ID = "PIRL.Utilities.Host (1.7 2012/04/16 06:18:24)";


/**	The cannonical, fully qualified name of the host system.
<p>
	If the hostname can not be obtained from the host system
	the {@link #DEFAULT_HOSTNAME} will be used.
<p>
	@see	#SHORT_HOSTNAME
*/
public static final String
	FULL_HOSTNAME;

/**	The short, single word name of the host system.
<p>
	If the hostname can not be obtained from the host system
	the {@link #DEFAULT_HOSTNAME} will be used.
<p>
	@see	#FULL_HOSTNAME
*/
public static final String
	SHORT_HOSTNAME;

/**	The Internet Protocol address of the host system.
<p>
	If the IP address can not be obtained from the host system
	the {@link #DEFAULT_HOSTNAME} will be used.
*/
public static final String
	IP_ADDRESS;

/**	The name that will be provided for the hostname or IP address if
	they can not be obtained from the host system.
*/
public static final String
	DEFAULT_HOSTNAME			= "unknown";

static
	{
	String
		full_hostname = null,
		short_hostname = null,
		ip_address = null;
	try
		{
		InetAddress
			address = InetAddress.getLocalHost ();
		full_hostname  = address.getCanonicalHostName ();
		short_hostname = address.getHostName ();
		ip_address     = address.getHostAddress ();
		}
	catch (UnknownHostException exception)
		{
		if (full_hostname == null)
			full_hostname = DEFAULT_HOSTNAME;
		if (short_hostname == null)
			short_hostname = DEFAULT_HOSTNAME;
		if (ip_address == null)
			ip_address = DEFAULT_HOSTNAME;
		}
	FULL_HOSTNAME  = full_hostname;
	SHORT_HOSTNAME = short_hostname;
	IP_ADDRESS     = ip_address;
	}

//!	No object constructor, just static functions.
private Host () {}

/**	Get the name of the host system.
<p>
	@return	The hostname String of the local system. If the hostname
		can not be determined, the {@link #DEFAULT_HOSTNAME} is returned.
	@see	#SHORT_HOSTNAME
*/
public static String Hostname ()
{return SHORT_HOSTNAME;}

/**	Get the name of a host system.
<p>
	@param	address	The textual representation of a host system IP
		address or its hostname. If null, the local host system is
		assumed.
	@return	The hostname String for the address. This will be null if
		hostname lookup failed. <b>N.B.</b>: Unlike for the {@link
		#Hostname() local system}, the {@link #DEFAULT_HOSTNAME} will not
		be returned if hostname lookup fails; instead of an ambiguous
		name for possibly multiple unknown hostnames, the calling
		application is expected to have rules to deal with an unknown
		hostname in this case.
	@throws	SecurityException	If hostname lookup is not allowed.
*/
public static String Hostname
	(
	String	address
	)
{
try {return InetAddress.getByName (address).getHostName ();}
catch (UnknownHostException exception) {}
return null;
}

/**	Get the fully qualified name of the host system.
<p>
	@return	The fully qualified hostname String of the local system. If
		the hostname can not be determined, the {@link #DEFAULT_HOSTNAME}
		is returned.
	@see	#FULL_HOSTNAME
*/
public static String Full_Hostname ()
{return FULL_HOSTNAME;}

/**	Get the fully qualified name for a hostname.
<p>
	@param	hostname	The name host system or the textual
		representation of its IP address. If null, the local host system
		is assumed.
	@return	The fully qualified hostname String for the hostname.
		This will be null if hostname lookup failed. <b>N.B.</b>: Unlike
		for the {@link #Full_Hostname() local system}, the {@link
		#DEFAULT_HOSTNAME} will not be returned if hostname lookup fails;
		instead of an ambiguous name for possibly multiple unknown
		hostnames, the calling application is expected to have rules to
		deal with an unknown hostname in this case.
	@throws	SecurityException	If hostname lookup is not allowed.
*/
public static String Full_Hostname
	(
	String	hostname
	)
{
try {return InetAddress.getByName (hostname).getCanonicalHostName ();}
catch (UnknownHostException exception) {}
return null;
}

/**	Get the IP address of the host system.
<p>
	@return	The String representation of the IP address of the local
		system. If the IP address can not be determined, the {@link
		#DEFAULT_HOSTNAME} is returned.
	@see	#IP_ADDRESS
*/
public static String IP_Address ()
{return IP_ADDRESS;}

/**	Get the IP address for a hostname.
<p>
	@param	hostname	The name host system or the textual
		representation of its IP address. If null, the local host system
		is assumed.
	@return	The String representation of the IP address for the hostname.
		This will be null if no IP address for the hostname could be
		found. <b>N.B.</b>: Unlike for the {@link #IP_Address() local
		system}, the {@link #DEFAULT_HOSTNAME} will not be returned if
		hostname lookup fails; instead of an ambiguous name for possibly
		multiple unknown hostnames, the calling application is expected
		to have rules to deal with an unknown hostname in this case.
	@throws	SecurityException	If hostname lookup is not allowed.
*/
public static String IP_Address
	(
	String	hostname
	)
{
try {return InetAddress.getByName (hostname).getHostAddress ();}
catch (UnknownHostException exception) {}
return null;
}

/**	Get the name of the host operating system.

	@return The name of the host OS from the os.name system property.
*/
public static String OS_Name ()
{return System.getProperty ("os.name");}


/**	Get all the environment variables from the host.

	@return	A Properties object contain all of the environment
		variables.
*/
public static Properties Environment ()
{
String
	get_env = "env",	//	Unix.
	OS = OS_Name ().toLowerCase ();

//	Check for MS/Windows special cases.
if (OS.indexOf ("windows 9") != -1)
	get_env = "command.com /c set";
else if (OS.indexOf ("nt") != -1 ||
		 OS.indexOf ("windows 20") != -1 ||
		 OS.indexOf ("windows xp") != -1)
	get_env = "cmd.exe /c set";
  
Properties
	environment = new Properties ();
try {environment.load (Runtime.getRuntime ().exec (get_env).getInputStream ());}
catch (IOException exception)
	{/* Shouldn't happen */}
return environment;
}

}
