/*	SQL

PIRL CVS ID: SQL.java,v 1.7 2012/04/16 06:08:57 castalia Exp

Copyright (C) 2003-2007  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package	PIRL.Database;

import	PIRL.Viewers.*;

import	java.awt.*;
import	java.awt.event.*;

import	javax.swing.*;
import	javax.swing.text.*;
import	javax.swing.event.*;
import	javax.swing.undo.*;

import	java.io.File;
import	java.io.FileReader;
import	java.io.BufferedReader;
import	java.io.FileWriter;
import	java.io.BufferedWriter;
import	java.io.FileNotFoundException;
import	java.io.IOException;

import	java.util.Vector;
import	java.util.Hashtable;
import	java.util.Iterator;

import	java.net.URL;


/**	A modal dialog containing a text editor pane in which to
	specify an SQL command.
<P>
	The </CODE>{@link #Get_Command(String, String, Component)
	Get_Command}</CODE> method will construct the dialog GUI, if it has
	not already been done, clears the text editor and then enables it
	for user interaction. When the user indicates that they are done
	the current contents of the text editor are returned; null will be
	returned if the Cancel button is used.
<P>
	@author	Bradford Castalia, UA/PIRL
	@version 1.7
*/
public class SQL
	extends JDialog
{
private static final String
	ID = "PIRL.Database.SQL (1.7 2012/04/16 06:08:57)";

private static final Color		KEYWORD_FOREGROUND_COLOR = new Color
									((float)0.00, (float)0.42, (float)0.06),
								KEYWORD_BACKGROUND_COLOR = new Color
									((float)0.91, (float)1.00, (float)0.90),
								OPERATION_FOREGROUND_COLOR = new Color
									((float)0.00, (float)0.00, (float)1.00),
								OPERATION_BACKGROUND_COLOR = new Color
									((float)0.92, (float)0.99, (float)1.00),
								MODIFIED_COLOR = new Color
									((float)1.00, (float)1.00, (float)0.00);
private static Color			NORMAL_COLOR;

private static final int		DEFAULT_VIEW_WIDTH		= 500,
								DEFAULT_VIEW_HEIGHT		= 350;

private static ImageIcon		OPEN_ICON = null;
private static final String		OPEN_ICON_NAME
									= "Open_File16.gif";
private static ImageIcon		SAVE_ICON;
private static final String		SAVE_ICON_NAME
									= "Save16.gif";
private static ImageIcon		SAVE_AS_ICON;
private static final String		SAVE_AS_ICON_NAME
									= "SaveAs16.gif";
private static ImageIcon		CUT_ICON;
private static final String		CUT_ICON_NAME
									= "Cut16.gif";
private static ImageIcon		COPY_ICON;
private static final String		COPY_ICON_NAME
									= "Copy16.gif";
private static ImageIcon		PASTE_ICON;
private static final String		PASTE_ICON_NAME
									= "Paste16.gif";
private static ImageIcon		UNDO_ICON;
private static final String		UNDO_ICON_NAME
									= "Undo16.gif";
private static ImageIcon		REDO_ICON;
private static final String		REDO_ICON_NAME
									= "Redo16.gif";
private static ImageIcon		SQL_FILE_ICON;
private static final String		SQL_FILE_ICON_NAME
									= "Bar_Arrow_Right_24.gif";

//	This SQL dialog.
private static SQL				_SQL_Dialog_ = null;

//	Reusable file chooser dialog.
private static JFileChooser		_File_Chooser_ = null;

//	The currently chosen file.
private File					Current_File = null;
private JLabel					Filename_Label;

//	SQL command text:

private static String			_Command_Line_;

private JTextPane				SQL_Pane;
private boolean					SQL_Modified = false;

//	Listeners for the SQL pane and its document.
private Caret_Listener			SQL_Caret_Listener
									= new Caret_Listener ();
private Document_Listener		SQL_Document_Listener
									= new Document_Listener ();
private Undoable_Edit_Listener	SQL_Edit_Listener
									= new Undoable_Edit_Listener ();

//	Editing:

private static Hashtable		Text_Actions = null;
private static final String		EDIT_CUT		= "Cut",
								EDIT_COPY		= "Copy",
								EDIT_PASTE		= "Paste",
								EDIT_SELECT_ALL	= "Select All",
								EDIT_UNDO		= "Undo",
								EDIT_REDO		= "Redo";

private JPopupMenu				Popup_Menu;

private UndoManager				Undo_Manager	= new UndoManager ();
private Undo_Action				Undo			= new Undo_Action ();
private Redo_Action				Redo			= new Redo_Action ();

//	Styles for the SQL text:

private static final int		BASIC_STYLE		= 0,
								COMMENT_STYLE	= 1,
								QUOTED_STYLE	= 2,
								OPERATION_STYLE	= 3,
								KEYWORD_STYLE	= 4;
private static final String		STYLE_NAMES[] =
								{
								"BASIC",
								"COMMENT",
								"QUOTED",
								"OPERATION",
								"KEYWORD"
								};
private static final SimpleAttributeSet
								SQL_Styles[]	= new SimpleAttributeSet[5];
static
	{
	SQL_Styles[BASIC_STYLE]		= new SimpleAttributeSet ();

	SQL_Styles[COMMENT_STYLE]	= new SimpleAttributeSet ();
	StyleConstants.setItalic		(SQL_Styles[COMMENT_STYLE], true);

	SQL_Styles[QUOTED_STYLE]	= new SimpleAttributeSet ();
	StyleConstants.setBold			(SQL_Styles[QUOTED_STYLE], true);

	SQL_Styles[OPERATION_STYLE]	= new SimpleAttributeSet ();
	StyleConstants.setForeground	(SQL_Styles[OPERATION_STYLE],
										OPERATION_FOREGROUND_COLOR);
	StyleConstants.setBackground	(SQL_Styles[OPERATION_STYLE],
										OPERATION_BACKGROUND_COLOR);
	StyleConstants.setBold			(SQL_Styles[OPERATION_STYLE], true);

	SQL_Styles[KEYWORD_STYLE]	= new SimpleAttributeSet ();
	StyleConstants.setForeground	(SQL_Styles[KEYWORD_STYLE],
										KEYWORD_FOREGROUND_COLOR);
	StyleConstants.setBackground	(SQL_Styles[KEYWORD_STYLE],
										KEYWORD_BACKGROUND_COLOR);
	StyleConstants.setBold			(SQL_Styles[KEYWORD_STYLE], true);
	}

private static final Vector		SQL_OPERATIONS = new Vector ();
static
								{
								SQL_OPERATIONS.add ("<");
								SQL_OPERATIONS.add (">");
								SQL_OPERATIONS.add ("<=");
								SQL_OPERATIONS.add (">=");
								SQL_OPERATIONS.add ("=");
								SQL_OPERATIONS.add ("!=");
								SQL_OPERATIONS.add ("AND");
								SQL_OPERATIONS.add ("OR");
								SQL_OPERATIONS.add ("NOT");
								SQL_OPERATIONS.add ("LIKE");
								}

private static final Vector		SQL_KEYWORDS = new Vector ();
static
								{
								SQL_KEYWORDS.add ("SELECT");
								SQL_KEYWORDS.add ("FROM");
								SQL_KEYWORDS.add ("WHERE");
								}

//	Document stylizer thread.
//private Style_SQL				Stylizer = new Style_SQL ();

//	System new-line sequence.
private static final String		NL = Database.NL;


//  DEBUG control.
private static final int
	DEBUG_OFF		= 0,
	DEBUG_UI_SETUP	= 1 << 0,
	DEBUG_FILE		= 1 << 1,
	DEBUG_SQL		= 1 << 2,
	DEBUG_GET		= 1 << 3,
	DEBUG_OPS		= 1 << 4,
	DEBUG_EDITS		= 1 << 5,
	DEBUG_LISTENERS	= 1 << 6,
	DEBUG_STYLE		= 1 << 7,
	DEBUG_ALL		= -1,

	DEBUG			= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
private SQL
	(
	String		title,
	Frame		owner
	)
{
super (owner, ((title == null) ? "SQL" : title), true);
if ((DEBUG & DEBUG_UI_SETUP) != 0)
	System.out.println(">>> SQL: " + title);

//	Dialog main panel.

Load_Icons (this);

JPanel panel		= new JPanel (new GridBagLayout ());
GridBagConstraints location = new GridBagConstraints ();

//	Filename label.

Filename_Label		= new JLabel ("", SwingConstants.LEFT);
Filename_Label.setOpaque (true);
NORMAL_COLOR = Filename_Label.getBackground ();
location.insets		= new Insets (0, 5, 0, 5);
location.gridwidth	= GridBagConstraints.REMAINDER;
location.anchor		= GridBagConstraints.WEST;
panel.add (Filename_Label, location);

//	SQL command line panel.

SQL_Pane			= Create_SQL_Pane ();
JScrollPane scroll_pane = new JScrollPane (SQL_Pane);
scroll_pane.setPreferredSize (new Dimension
	(DEFAULT_VIEW_WIDTH, DEFAULT_VIEW_HEIGHT));
scroll_pane.setMinimumSize (new Dimension
	(DEFAULT_VIEW_WIDTH, DEFAULT_VIEW_HEIGHT));
location.insets		= new Insets (0, 5, 0, 5);
location.gridwidth	= GridBagConstraints.REMAINDER;
location.fill		= GridBagConstraints.BOTH;
location.weightx	= 1.0;
location.weighty	= 1.0;
panel.add (scroll_pane, location);

//	Buttons.

JButton button		= new JButton ("Clear");
button.setMnemonic ('L');
button.setToolTipText ("Clear command");
button.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event) {Clear ();}});
location.insets		= new Insets (5, 5, 5, 0);
location.gridwidth	= 1;
location.fill		= GridBagConstraints.NONE;
location.weightx	= 0.0;
location.weighty	= 0.0;
panel.add (button, location);

location.fill		= GridBagConstraints.HORIZONTAL;
location.weightx	= 1.0;
panel.add (Box.createHorizontalGlue (), location);

button				= new JButton ("Cancel");
button.setMnemonic ('C');
button.setToolTipText ("Cancel operation");
button.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event) {Cancel ();}});
location.insets		= new Insets (5, 0, 5, 0);
location.anchor 	= GridBagConstraints.EAST;
location.fill		= GridBagConstraints.NONE;
location.weightx	= 0.0;
panel.add (button, location);

button				= new JButton ("Execute");
button.setMnemonic ('X');
button.setToolTipText ("Execute command");
button.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event) {Accept ();}});
location.insets		= new Insets (5, 0, 5, 5);
location.gridwidth	= GridBagConstraints.REMAINDER;
panel.add (button, location);

//	Menu bars (set after Create_SQL_Pane for text edit actions).

JMenuBar menu_bar	= Create_Menus ();
setJMenuBar (menu_bar);

//	Attach the Popup_Menu.
MouseListener popup_listener = new Popup_Listener ();
SQL_Pane.addMouseListener (popup_listener);
menu_bar.addMouseListener (popup_listener);

//	Enable the Caret, Undoable_Edit and Document Listeners.
Enable_Listeners (true);

getContentPane ().add (panel, BorderLayout.CENTER);
pack ();
if ((DEBUG & DEBUG_UI_SETUP) != 0)
	System.out.println ("<<< SQL");
}

/*==============================================================================
	Menus
*/
private JMenuBar Create_Menus ()
{
//	File menu.

JMenu
	file_menu		= new JMenu ("File"),
	file_menu_PU	= new JMenu ("File");
file_menu.setMnemonic ('F');

JMenuItem
	menu_item		= new JMenuItem ("Open...", OPEN_ICON),
	menu_item_PU	= new JMenuItem ("Open...");
menu_item.setMnemonic ('O');
menu_item.setAccelerator (KeyStroke.getKeyStroke ('O', Event.CTRL_MASK));
menu_item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event) {Open_File ();}});
menu_item_PU.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event) {Open_File ();}});
file_menu.add (menu_item);
file_menu_PU.add (menu_item_PU);

menu_item			= new JMenuItem ("Save", SAVE_ICON);
menu_item_PU		= new JMenuItem ("Save");
menu_item.setMnemonic ('S');
menu_item.setAccelerator (KeyStroke.getKeyStroke ('S', Event.CTRL_MASK));
menu_item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event) {Save_File ();}});
menu_item_PU.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event) {Save_File ();}});
file_menu.add (menu_item);
file_menu_PU.add (menu_item_PU);

menu_item			= new JMenuItem ("Save As...", SAVE_AS_ICON);
menu_item_PU		= new JMenuItem ("Save As...");
menu_item.setMnemonic ('A');
menu_item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event) {Save_File_As ();}});
menu_item_PU.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event) {Save_File_As ();}});
file_menu.add (menu_item);
file_menu_PU.add (menu_item_PU);

//	File chooser used with File menus.

_File_Chooser_ = new JFileChooser (System.getProperty ("user.dir"));
_File_Chooser_.setFileSelectionMode (JFileChooser.FILES_ONLY);

//	Edit menu.

JMenu
	edit_menu		= new JMenu ("Edit"),
	edit_menu_PU	= new JMenu ("Edit");
edit_menu.setMnemonic ('E');

menu_item = edit_menu.add (Undo);
edit_menu_PU.add (Undo);
menu_item.setIcon (UNDO_ICON);
menu_item.setMnemonic ('R');
menu_item.setAccelerator
	(KeyStroke.getKeyStroke ('Z', Event.CTRL_MASK));

menu_item = edit_menu.add (Redo);
edit_menu_PU.add (Redo);
menu_item.setIcon (REDO_ICON);
menu_item.setMnemonic ('U');
menu_item.setAccelerator
	(KeyStroke.getKeyStroke ('Z', Event.SHIFT_MASK | Event.CTRL_MASK));

edit_menu.addSeparator();
edit_menu_PU.addSeparator();

Action action = (Action)Text_Actions.get (EDIT_CUT);
menu_item = edit_menu.add (action);
edit_menu_PU.add (action);
menu_item.setIcon (CUT_ICON);
menu_item.setMnemonic ('T');
menu_item.setAccelerator
	(KeyStroke.getKeyStroke ('X', Event.CTRL_MASK));

action = (Action)Text_Actions.get (EDIT_COPY);
menu_item = edit_menu.add (action);
edit_menu_PU.add (action);
menu_item.setIcon (COPY_ICON);
menu_item.setMnemonic ('C');
menu_item.setAccelerator
	(KeyStroke.getKeyStroke ('C', Event.CTRL_MASK));

action = (Action)Text_Actions.get (EDIT_PASTE);
menu_item = edit_menu.add (action);
edit_menu_PU.add (action);
menu_item.setIcon (PASTE_ICON);
menu_item.setMnemonic ('P');
menu_item.setAccelerator
	(KeyStroke.getKeyStroke ('V', Event.CTRL_MASK));

edit_menu.addSeparator();
edit_menu_PU.addSeparator();

action = (Action)Text_Actions.get (EDIT_SELECT_ALL);
menu_item = edit_menu.add (action);
edit_menu_PU.add (action);
menu_item.setMnemonic ('A');
menu_item.setAccelerator
	(KeyStroke.getKeyStroke ('A', Event.CTRL_MASK));

//	Operations menu.

JMenu
	ops_menu		= new JMenu ("Operations"),
	ops_menu_PU		= new JMenu ("Operations");
ops_menu.setMnemonic ('O');

Iterator operation = SQL_OPERATIONS.iterator ();
while (operation.hasNext ())
	{
	action	= new AbstractAction ((String)operation.next ())
		{public void actionPerformed (ActionEvent event) {Insert_Op (event);}};
	ops_menu.add (action);
	ops_menu_PU.add (action);
	}

//	Assemble the menu bar.

JMenuBar menu_bar	= new JMenuBar ();
menu_bar.add (file_menu);
menu_bar.add (edit_menu);
menu_bar.add (ops_menu);

//	Assemble the popup menu.

Popup_Menu			= new JPopupMenu ();
Popup_Menu.add (ops_menu_PU);
Popup_Menu.add (edit_menu_PU);
Popup_Menu.add (file_menu_PU);

return menu_bar;
}

/**	Enables or disables the cut/copy edit menus.
*/
private void Edit_Menus_Enabled
	(
	boolean			enabled
	)
{
((Action)Text_Actions.get (EDIT_CUT)).setEnabled (enabled);
((Action)Text_Actions.get (EDIT_COPY)).setEnabled (enabled);
}

/*------------------------------------------------------------------------------
	Operation Menu Action
*/
/**	Insert an operation marker into the SQL command.
*/
private void Insert_Op
	(
	ActionEvent		event
	)
{
if ((DEBUG & DEBUG_OPS) != 0)
	System.out.println ("Insert_Op: " + event.getActionCommand ());
AttributeSet old_attributes = SQL_Pane.getCharacterAttributes ();
if ((DEBUG & DEBUG_OPS) != 0)
	{
	if (old_attributes.isEqual (SQL_Styles[COMMENT_STYLE]))
		System.out.println ("  into " + STYLE_NAMES[COMMENT_STYLE]);
	else if (old_attributes.isEqual (SQL_Styles[QUOTED_STYLE]))
		System.out.println ("  into " + STYLE_NAMES[QUOTED_STYLE]);
	}
boolean
	listeners_off = ! (
		old_attributes.isEqual (SQL_Styles[COMMENT_STYLE]) ||
		old_attributes.isEqual (SQL_Styles[QUOTED_STYLE]));

//	Remove any selection.
if (SQL_Pane.getSelectionStart () != SQL_Pane.getSelectionEnd ())
	SQL_Pane.replaceSelection (null);

//	Insert the operation marker.
SQL_Pane.replaceSelection (event.getActionCommand ());
}

/*------------------------------------------------------------------------------
	Popup Menu Listener
*/
class Popup_Listener
	extends MouseAdapter
{
public void mousePressed (MouseEvent event)
{maybeShowPopup (event);}

public void mouseReleased (MouseEvent event)
{maybeShowPopup (event);}

private void maybeShowPopup
	(
	MouseEvent	event
	)
{
if (event.isPopupTrigger ())
	Popup_Menu.show (event.getComponent (), event.getX (), event.getY ());
}
}

/*==============================================================================
	SQL Text Pane
*/
private JTextPane Create_SQL_Pane ()
{
JTextPane pane			= new JTextPane ();
pane.setMargin (new Insets (10, 10, 10, 10));

if (Text_Actions == null)
	Text_Actions = Create_Action_Table (pane);
//	Rename the edit actions.
Action action = (Action)Text_Actions.get (DefaultEditorKit.cutAction);
action.putValue (Action.NAME, EDIT_CUT);
action = (Action)Text_Actions.get (DefaultEditorKit.copyAction);
action.putValue (Action.NAME, EDIT_COPY);
action = (Action)Text_Actions.get (DefaultEditorKit.pasteAction);
action.putValue (Action.NAME, EDIT_PASTE);
action = (Action)Text_Actions.get (DefaultEditorKit.selectAllAction);
action.putValue (Action.NAME, EDIT_SELECT_ALL);
Text_Actions = Create_Action_Table (pane);

return pane;
}

/**	Sets the current SQL command String.
<P>
	@param	command	The SQL command String.
*/
private void Set_Command
	(
	String	command
	)
{
//	Disable the Listeners during the change of text.
Enable_Listeners (false);

//	Remove the current content.
Document document = SQL_Pane.getDocument ();
try
	{
	document.remove (0, document.getLength ());
	if (command != null)
		{
		document.insertString (0, command, null);
		style_SQL ();
		Set_SQL_Modified (false);
		}
	}
catch (BadLocationException exception)
	{
	Dialog_Box.Error ("Unexpected BadLocationException in Set_Command" + NL
		+ "at offset " + exception.offsetRequested () + NL
		+ NL
		+ ID + NL
		+ exception.getMessage (), _SQL_Dialog_);
	}
Enable_Listeners (true);
}

/*------------------------------------------------------------------------------
	PUBLIC INTERFACE
*/
/**	Gets an SQL command String.
<P>
	The SQL dialog window is initialized (if it has not been done
	already) and made visible. The resulting command line is returned.
<P>
	@param	title	The title String for the dialog window.
	@param	initial_SQL_command	A String to use as the initial SQL
		command text.
	@param	parent	The parent window that dialog is associated with,
		which may be null.
	@return	The SQL command String. This will be null if the Cancel
		button was used.
*/
public static String Get_Command
	(
	String		title,
	String		initial_SQL_command,
	Component	parent
	)
{
if ((DEBUG & DEBUG_GET) != 0)
	System.out.println
		(">>> SQL.Get_Command: " + initial_SQL_command);
if (_SQL_Dialog_ == null)
	_SQL_Dialog_ = new SQL
		(title, JOptionPane.getFrameForComponent (parent));
_Command_Line_ = null;
_SQL_Dialog_.Reset ();
_SQL_Dialog_.Set_Command (initial_SQL_command);
_SQL_Dialog_.setLocationRelativeTo (parent);
_SQL_Dialog_.setVisible (true);
if ((DEBUG & DEBUG_GET) != 0)
	System.out.println
		("<<< SQL.Get_Command: " + _Command_Line_ + NL + NL);
return _Command_Line_;
}

public static String Get_Command
	(
	String		title,
	String		initial_SQL_command
	)
{return Get_Command (title, initial_SQL_command, null);}

public static String Get_Command
	(
	String		title,
	Component	parent
	)
{return Get_Command (title, null, parent);}

public static String Get_Command
	(
	String		title
	)
{return Get_Command (title, null, null);}

public static String Get_Command ()
{return Get_Command (null, null, null);}

/*------------------------------------------------------------------------------
	Document Listeners

	WARNING: Don't modify the contents of the document within a Listener.
*/

protected class Caret_Listener
	implements CaretListener
{
public void caretUpdate (CaretEvent event)
{
if ((DEBUG & DEBUG_LISTENERS) != 0)
	System.out.println ("Caret_Listener: selection "
		+ (event.getDot () != event.getMark ()));



if (event.getDot () == event.getMark ())
	Edit_Menus_Enabled (false);
else
	Edit_Menus_Enabled (true);
}
}

protected class Document_Listener
	implements DocumentListener
{
//	Sees insertions into the document.
public void insertUpdate (DocumentEvent event)
{
if ((DEBUG & DEBUG_LISTENERS) != 0)
	System.out.println ("Document_Listener: insertUpdate"
		+ "  Type " + event.getType ()
		+ " at " + event.getOffset () + ", length " + event.getLength ());
Set_SQL_Modified (true);
style_SQL ();
/*
EventQueue.invokeLater (new Runnable ()
	{
	public void run ()
	{
	style_SQL ();
	}});
*/
}

//	Sees removals from the document.
public void removeUpdate (DocumentEvent event)
{
if ((DEBUG & DEBUG_LISTENERS) != 0)
	System.out.println ("Document_Listener: removeUpdate"
		+ "  Type " + event.getType ()
		+ " at " + event.getOffset () + ", length " + event.getLength ());
Set_SQL_Modified (true);
style_SQL ();
}

//	Sees changes to the document style.
public void changedUpdate (DocumentEvent event)
{
if ((DEBUG & DEBUG_LISTENERS) != 0)
	System.out.println ("Document_Listener: changedUpdate"
		+ "  Type " + event.getType ()
		+ " at " + event.getOffset () + ", length " + event.getLength ());
}
}  

/*==============================================================================
	SQL Style
*/
/**	Apply styling to the SQL command.
<P>
	Style attributes are applied to the SQL command based on the
	text content.
<P>
	<I>It could be argued that a custom EditorKit that overrides the
	<CODE>read</CODE> method should be used here. This should be done
	if things become too complicated, but this simple parser is fine
	for now. This method would become the basis for a read method or a
	parser class used by a read method.</I>
*/
public void Style_SQL ()
{
if ((DEBUG & DEBUG_SQL) != 0)
	System.out.println (">>> Style_SQL");
StyledDocument
	document = SQL_Pane.getStyledDocument ();
int
	end = document.getLength ();
if (end == 0)
	{
	if ((DEBUG & DEBUG_SQL) != 0)
		System.out.println ("<<< Style_SQL: no text");
	return;
	}

//	Prevent external document modification during the application of styles.
synchronized (document)
{
StringBuffer
	text = null;
try {text = new StringBuffer (document.getText (0, end).toUpperCase ());}
catch (BadLocationException exception) {}	// Can't happen.

//	Disable the Listeners during the document changes.
Enable_Listeners (false);

//	Reset all styles.
document.setCharacterAttributes (0, end, SQL_Styles[BASIC_STYLE], true);

int
	style = BASIC_STYLE,
	index,
	start = -1;
String
	marker = null;
for (index = 0;
	 index < end;
	 index++)
	{
	if (start < 0)
		{
		//	Check for the start of a special section.

		//	Quoted sequences.
		if (text.charAt (index) == '"' ||
			text.charAt (index) == '\'')
			{
			style = QUOTED_STYLE;
			start = index;
			marker = text.substring (index, index + 1);
			}
		//	C-style comments.
		else if (text.charAt (index) == '/' &&
			(index + 1) < end &&
			text.charAt (index + 1) == '*')
			{
			style = COMMENT_STYLE;
			start = index;
			marker = "*/";
			index++;
			}
		//	#-style comments.
		else if (text.charAt (index) == '#')
			{
			style = COMMENT_STYLE;
			start = index;
			marker = NL;
			index += NL.length () - 1;
			}
		}
	else
		{
		//	Find the section closing marker.
		if ((index = text.indexOf (marker, index)) < 0)
			//	The section continues to the end of the text.
			index = end;
		else
			{
			if (style == QUOTED_STYLE &&
				text.charAt (index - 1) == '\\')
				//	Escaped quote.
				continue;
			index += marker.length ();
			}
		if ((DEBUG & DEBUG_SQL) != 0)
			System.out.println
				(start + "-" + (index - 1) + " " + STYLE_NAMES[style]);
		//	Style the section.
		document.setCharacterAttributes (start, index - start,
			SQL_Styles[style], true);
		//	Mask out the section.
		while (start < index)
			text.setCharAt (start++, (char)0);
		//	Start looking for special sections again.
		start = -1;
		}
	}

Iterator
	names = SQL_OPERATIONS.iterator ();
style = OPERATION_STYLE;
int
	length;
boolean
	is_symbol;
while (names.hasNext ())
	{
	marker = (String)names.next ();
	length = marker.length ();
	is_symbol = ! Character.isLetter (marker.charAt (0));
	start = 0;
	for (start = text.indexOf (marker, start);
		 start >= 0;
		 start = text.indexOf (marker, start))
		{
		index = start + length;
		if (is_symbol ||
			//	A word must be free-standing.
			(start == 0 || text.charAt (start - 1) <= ' ') &&
			(index == end || text.charAt (index) <= ' '))
			{
			if ((DEBUG & DEBUG_SQL) != 0)
				System.out.println
					(start + "-" + (index - 1) + " "
					+ STYLE_NAMES[style] + ": " + marker);
			document.setCharacterAttributes (start, length,
				SQL_Styles[style], true);
			//	Mask it.
			while (start < index)
				text.setCharAt (start++, (char)0);
			}
		else
			start = index;
		}
	}

names = SQL_KEYWORDS.iterator ();
style = KEYWORD_STYLE;
while (names.hasNext ())
	{
	marker = (String)names.next ();
	length = marker.length ();
	start = 0;
	for (start = text.indexOf (marker, start);
		 start >= 0;
		 start = text.indexOf (marker, start))
		{
		index = start + length;
		//	The word must be free-standing.
		if ((start == 0 || text.charAt (start - 1) <= ' ') &&
			(index == end || text.charAt (index) <= ' '))
			{
			if ((DEBUG & DEBUG_SQL) != 0)
				System.out.println
					(start + "-" + (index - 1) + " "
					+ STYLE_NAMES[style] + ": " + marker);
			document.setCharacterAttributes (start, length,
				SQL_Styles[style], true);
			}
		start = index;
		}
	}

Enable_Listeners (true);
if ((DEBUG & DEBUG_SQL) != 0)
	System.out.println ("<<< Style_SQL");
}	//	End of synchrozied section.
}

/**	Apply styles to the SQL command document.
<P>
	The call to Style_SQL method is wrapped in a Runnable for later
	invocation on the event queue because the operation may be
	initiated from within a document listener which has the document
	locked to prevent modification while in the listener.
*/
private void style_SQL ()
{
if ((DEBUG & DEBUG_SQL) != 0)
	System.out.println ("*** style_SQL: Style_SQL being queued.");
EventQueue.invokeLater (new Runnable ()
	{
	public void run ()
	{
	Style_SQL ();
	}});
}

public static String Clean
	(
	String	SQL_string
	)
{
if ((DEBUG & DEBUG_STYLE) != 0)
	System.out.println (">>> SQL.Clean:" + NL
		+ SQL_string + NL);
StringBuffer
	clean_SQL = new StringBuffer (SQL_string);
String
	marker = null;
int
	start = 0,
	advance,
	index = 0,
	end = SQL_string.length (),
	NL_length = NL.length ();
for (index = 0;
	 index < end;
	 index++)
	{
	if (marker == null)
		{
		//	HT
		if (clean_SQL.charAt (index) == '\t')
			{
			clean_SQL.setCharAt (index, ' ');
			continue;
			}
		//	NEW_LINE
		for (advance = 0;
			 advance < NL_length &&
			 	clean_SQL.charAt (index + advance) == NL.charAt (advance);
			 advance++);
		if (advance == NL_length)
			{
			clean_SQL.replace (index, index + NL_length, " ");
			end -= NL_length + 1;
			index += NL_length - 1;
			continue;
			}
		//	C-style comments.
		if (clean_SQL.charAt (index) == '/' &&
			(index + 1) < end &&
			clean_SQL.charAt (index + 1) == '*')
			{
			start = index;
			marker = "*/";
			index++;
			}
		//	#-style comments.
		else if (clean_SQL.charAt (index) == '#')
			{
			start = index;
			marker = NL;
			index += NL_length - 1;
			}
		}
	else
		{
		//	Find the section closing marker.
		if ((index = clean_SQL.indexOf (marker, index)) < 0)
			//	The section continues to the end of the text.
			index = end;
		else
			index += marker.length ();
		clean_SQL.delete (start, index);
		marker = null;
		end -= index - start;
		}
	}
if ((DEBUG & DEBUG_STYLE) != 0)
	System.out.println ("<<< SQL.Clean:" + NL
		+ clean_SQL + NL);
return clean_SQL.toString ();
}

/*==============================================================================
	Undo/Redo
*/
class Undo_Action
	extends AbstractAction
{
public Undo_Action ()
{
super (EDIT_UNDO);
setEnabled (false);
}

public void actionPerformed (ActionEvent e)
{
try {Undo_Manager.undo ();}
catch (CannotUndoException exception)
	{Dialog_Box.Warning ("Can't undo: " + exception, _SQL_Dialog_);}
Update_State ();
Redo.Update_State ();
}

protected void Update_State ()
{
if (Undo_Manager.canUndo ())
	{
	setEnabled (true);
	putValue (Action.NAME, Undo_Manager.getUndoPresentationName ());
	Set_SQL_Modified (true);
	}
else
	{
	setEnabled (false);
	putValue (Action.NAME, EDIT_UNDO);
	Set_SQL_Modified (false);
	}
}      
}    

class Redo_Action
	extends AbstractAction
{
public Redo_Action ()
{
super (EDIT_REDO);
setEnabled (false);
}

public void actionPerformed
	(
	ActionEvent		event
	)
{
try {Undo_Manager.redo ();}
catch (CannotRedoException exception)
	{Dialog_Box.Warning ("Can't redo: " + exception, _SQL_Dialog_);}
Update_State ();
Undo.Update_State ();
}

protected void Update_State ()
{
if (Undo_Manager.canRedo ())
	{
	setEnabled (true);
	putValue (Action.NAME, Undo_Manager.getRedoPresentationName ());
	}
else
	{
	setEnabled (false);
	putValue (Action.NAME, EDIT_REDO);
	}
}
}

/*..............................................................................
	Edit Listener
*/
protected class Undoable_Edit_Listener
	implements UndoableEditListener
{
public void undoableEditHappened
	(
	UndoableEditEvent	event
	)
{
if ((DEBUG & (DEBUG_EDITS | DEBUG_LISTENERS)) != 0)
	System.out.println ("--- Undoable_Edit_Listener::undoableEditHappened -" + NL
		+ event.getEdit ().getPresentationName ()
		+ " is" + (event.getEdit ().isSignificant () ? " " : " not ") + "significant.");
//	Remember the edit and update the menus.
Undo_Manager.addEdit (event.getEdit ());
Undo.Update_State ();
Redo.Update_State ();
}
}

/*==============================================================================
	File
*/
private void Open_File ()
{
if (Save_Check () &&
	_File_Chooser_.showOpenDialog (this)
		== JFileChooser.APPROVE_OPTION)
	{
	File file = _File_Chooser_.getSelectedFile ();
	if ((DEBUG & DEBUG_FILE) != 0)
		System.out.println ("Open_File: " + file.getPath ());
	try
		{
		BufferedReader
			reader = new BufferedReader (new FileReader (file));
		String
			new_text = "",
			line;
		while ((line = reader.readLine ()) != null)
			new_text += line + NL;
		reader.close ();
		Set_File (file);
		Set_Command (new_text);
		style_SQL ();
		Set_SQL_Modified (false);
		Undo_Manager.discardAllEdits ();
		}
	catch (FileNotFoundException exception)
		{Dialog_Box.Error
			("Couldn't find the file: " + file.getPath () + NL
			+ NL
			+ ID + NL
			+ exception.getMessage (), this);}
	catch (IOException exception)
		{Dialog_Box.Error
			("Unable to read the file: " + file.getPath () + NL
			+ NL
			+ ID + NL
			+ exception.getMessage (), this);}
	catch (Exception exception)
		{Dialog_Box.Error
			("System error!" + NL
			+ NL
			+ ID + NL
			+ exception.getMessage (), this);}
	}
}

private void Save_File ()
{
if (SQL_Modified)
	{
	if (Current_File == null)
		Save_File_As ();
	else if (Write_File (Current_File))
		Set_SQL_Modified (false);
	}
}

private boolean Save_File_As ()
{
if (_File_Chooser_.showSaveDialog (this)
	== JFileChooser.APPROVE_OPTION)
	{
	File file = _File_Chooser_.getSelectedFile ();
	if ((DEBUG & DEBUG_FILE) != 0)
		System.out.println ("Save_File_As: " + file.getPath ());
	if (Write_File (file))
		{
		Set_File (file);
		Set_SQL_Modified (false);
		return true;
		}
	}
return false;
}

/**	Writes the current contents of the SQL command pane to the
	specified file.
<P>
	@param	file	File to be written.
	@return	true if the file was sucessfully written; false otherwise.
*/
private boolean Write_File
	(
	File	file
	)
{
try
	{
	BufferedWriter writer = new BufferedWriter (new FileWriter (file));
	String text = SQL_Pane.getText ();
	if ((DEBUG & DEBUG_FILE) != 0)
		System.out.println ("saving:" + NL
			+ text);
	writer.write (text, 0, text.length ());
	writer.close ();
	return true;
	}
catch (IOException exception)
	{
	Dialog_Box.Error ("Unable to write the file: " + file.getPath () + NL
		+ NL
		+ ID + NL
		+ exception.getMessage (), this);
	}
catch (Exception exception)
	{Dialog_Box.Error
		("System error!" + NL
		+ NL
		+ ID + NL
		+ exception.getMessage (), this);}
return false;
}

/**	Checks if the SQL text has been changed and, if so, whether
	the user wants to save it. Then saves the text in a file if
	the user says Yes.
<P>
	@return	false if the user specified Cancel on the query to
		save modified SQL text, or the save failed; true otherwise.
*/
private boolean Save_Check ()
{
if (SQL_Modified)
	{
	int check = Dialog_Box.Check (((Current_File == null) ?
		"Save the modified SQL" :
		("Save the modified file:" + NL
			+ Current_File.getPath ())),
		this);
	if (check == 0)
		//	Cancel the new file.
		return false;
	if (check > 0)
		{
		if (Current_File == null && ! Save_File_As ())
			return false;
		else if (! Write_File (Current_File))
			return false;
		}
	}
return true;
}

/**	Sets the Current_File and displays the name.
*/
private void Set_File
	(
	File		file
	)
{
String
	filename;
if (file == null ||
	file.getName ().equals (""))
	{
	Current_File = null;
	filename = "(no file)";
	Filename_Label.setText ("(no file)");
	}
else
	{
	Current_File = file;
	filename = Current_File.getAbsolutePath ();
	}
if (SQL_Modified)
	filename += " (modified)";
Filename_Label.setText (filename);
}

/*==============================================================================
	Clear, Cancel, Accept Actions
*/
private void Clear ()
{
if (Save_Check ())
	Reset ();
}

private void Cancel ()
{
if (Save_Check ())
	setVisible (false);
}

private void Accept ()
{
if (Save_Check ())
	{
	_Command_Line_ = SQL_Pane.getText ();
	setVisible (false);
	}
}

/*==============================================================================
	Helpers
*/
/**	Resets the Dialog to its initial empty state.
*/
private void Reset ()
{
Set_File (null);					//	Clear the current File.
Set_Command (null);					//	Remove the document content.
Undo_Manager.discardAllEdits ();	//	Drop any undo edits.
Edit_Menus_Enabled (false);			//	Disable the edit actions.
}

/**	Sets the SQL_Modified state and updates the display to indicate it.
*/
private void Set_SQL_Modified
	(
	boolean			modified
	)
{
if (SQL_Modified != modified)
	{
	//	The modified state has changed.
	if ((DEBUG & DEBUG_SQL) != 0)
		System.out.println ("Set_SQL_Modified: " + modified);
	if (SQL_Modified = modified)
		{
		Filename_Label.setBackground (MODIFIED_COLOR);
		Edit_Menus_Enabled (true);
		}
	else
		{
		Filename_Label.setBackground (NORMAL_COLOR);
		Edit_Menus_Enabled (false);
		}
	Set_File (Current_File);	// Update the filename display.
	}
}

private Hashtable Create_Action_Table
	(
	JTextComponent	text_component
	)
{
Hashtable actions_table = new Hashtable ();
Action[] actions = text_component.getActions ();
for (int i = 0; i < actions.length; i++)
	{
	Action action = actions[i];
	actions_table.put (action.getValue (Action.NAME), action);
	}
return actions_table;
}

private void Enable_Listeners
	(
	boolean			enable
	)
{
if ((DEBUG & DEBUG_LISTENERS) != 0)
	System.out.println ("--- Enable_Listeners: " + enable);
StyledDocument document = SQL_Pane.getStyledDocument ();
if (enable)
	{
	document.addUndoableEditListener (SQL_Edit_Listener);
	document.addDocumentListener (SQL_Document_Listener);
	SQL_Pane.addCaretListener (SQL_Caret_Listener);
	}
else
	{
	document.removeUndoableEditListener (SQL_Edit_Listener);
	document.removeDocumentListener (SQL_Document_Listener);
	SQL_Pane.removeCaretListener (SQL_Caret_Listener);
	}
}

private static void Load_Icons
	(
	Object	object
	)
{
if (OPEN_ICON != null)
	return;
char
	file_separator = System.getProperty ("file.separator").charAt (0);
String
	package_name = object.getClass ().getName ();
package_name = package_name.substring (0, package_name.lastIndexOf ('.'))
		.replace ('.', file_separator)
		+ file_separator + "Icons" + file_separator;
URL
	iconURL;
iconURL = ClassLoader.getSystemResource (package_name + OPEN_ICON_NAME);
if (iconURL != null)
	OPEN_ICON = new ImageIcon (iconURL);
iconURL = ClassLoader.getSystemResource (package_name + SAVE_ICON_NAME);
if (iconURL != null)
	SAVE_ICON = new ImageIcon (iconURL);
iconURL = ClassLoader.getSystemResource (package_name + SAVE_AS_ICON_NAME);
if (iconURL != null)
	SAVE_AS_ICON = new ImageIcon (iconURL);
iconURL = ClassLoader.getSystemResource (package_name + CUT_ICON_NAME);
if (iconURL != null)
	CUT_ICON = new ImageIcon (iconURL);
iconURL = ClassLoader.getSystemResource (package_name + COPY_ICON_NAME);
if (iconURL != null)
	COPY_ICON = new ImageIcon (iconURL);
iconURL = ClassLoader.getSystemResource (package_name + PASTE_ICON_NAME);
if (iconURL != null)
	PASTE_ICON = new ImageIcon (iconURL);
iconURL = ClassLoader.getSystemResource (package_name + UNDO_ICON_NAME);
if (iconURL != null)
	UNDO_ICON = new ImageIcon (iconURL);
iconURL = ClassLoader.getSystemResource (package_name + REDO_ICON_NAME);
if (iconURL != null)
	REDO_ICON = new ImageIcon (iconURL);
}

/*==============================================================================
	Test stub
*/
private static String SQL_SELECT_test =
	"Select"  + NL +
	"# SQL SELECT test" + NL +
	"a_field, b_field from the table" + NL +
	"/*" + NL +
	"    Where conditions:" + NL +
	"*/" + NL +
	"Where a_field < 3" + NL +
	"and b_field = 'test'" + NL +
	"from";

public static void main (String[] args)
{
if (args.length > 0)
	SQL_SELECT_test = args[0];
final JFrame
	frame = new JFrame ("SQL");
frame.addWindowListener (new WindowAdapter ()
	{public void windowClosing (WindowEvent e) {System.exit (0);}});

JButton
	button = new JButton ("SQL...");
button.addActionListener (new ActionListener ()
	{
	public void actionPerformed (ActionEvent e)
		{
		SQL_SELECT_test = SQL.Get_Command
			("SQL test", SQL_SELECT_test, frame);
		System.out.println (SQL_SELECT_test);
		}
	});

JPanel
	contentPane = new JPanel ();
frame.setContentPane (contentPane);
contentPane.setLayout (new BoxLayout (contentPane, BoxLayout.Y_AXIS));
contentPane.setBorder (BorderFactory.createEmptyBorder (20, 20, 20, 20));
contentPane.add (button);
button.setAlignmentX (JComponent.CENTER_ALIGNMENT);
frame.pack ();
frame.setVisible (true);
}


}

