# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
# Copyright (C) 2013 Canonical Ltd.
# Author: Sergio Schvezov <sergio.schvezov@canonical.com>

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 3 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import atexit
import contextlib
import logging
import lzma
import os.path
import shutil
import tarfile
import tempfile


log = logging.getLogger()


def cleanup(directory):
    if os.path.exists(directory) and os.path.isdir(directory):
        print('Removing directory %s' % directory)
        shutil.rmtree(directory)


def create_temp_dir():
    directory = tempfile.mkdtemp()
    atexit.register(cleanup, directory)
    return directory


def create_path(directory, subdir):
    directory = os.path.join(directory, subdir)
    if not os.path.exists(directory):
        log.debug('Creating directory %s' % directory)
        try:
            os.makedirs(directory)
        except OSError as e:
            if e.errno == 17:
                pass
            else:
                raise e
    return directory


def lzma_extract_file(lzma_tarfile, file_name):
    log.info('Decompressing %s from %s' % (file_name, lzma_tarfile))
    temp_dir = create_temp_dir()
    with contextlib.closing(lzma.LZMAFile(lzma_tarfile)) as xz:
        with tarfile.open(fileobj=xz) as tar:
            tar.extract(member=file_name, path=temp_dir)
    return os.path.join(temp_dir, file_name)
