// Copyright (C) 2024 The Android Open Source Project
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

import {Registry} from '../base/registry';
import {SidebarManager, SidebarMenuItem} from '../public/sidebar';

export type SidebarMenuItemInternal = SidebarMenuItem & {
  id: string; // A unique id generated by this class at registration time.
};

export class SidebarManagerImpl implements SidebarManager {
  readonly enabled: boolean;
  private _visible: boolean;
  private lastId = 0;

  readonly menuItems = new Registry<SidebarMenuItemInternal>((m) => m.id);

  constructor(args: {disabled?: boolean; hidden?: boolean}) {
    this.enabled = !args.disabled;
    this._visible = !args.hidden;
  }

  addMenuItem(item: SidebarMenuItem): Disposable {
    // Assign a unique id to every item. This simplifies the job of the mithril
    // component that renders the sidebar.
    const id = `sidebar_${++this.lastId}`;
    const itemInt: SidebarMenuItemInternal = {...item, id};
    return this.menuItems.register(itemInt);
  }

  public get visible() {
    return this._visible;
  }

  public toggleVisibility() {
    if (!this.enabled) return;
    this._visible = !this._visible;
  }
}
