///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/mainwnd/MainFrame.h>
#include <core/gui/ApplicationManager.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/ViewportManager.h>
#include <core/viewport/ViewportPanel.h>
#include <core/viewport/input/ViewportInputManager.h>
#include <core/viewport/input/ViewportInputHandler.h>
#include <core/viewport/input/NavigationModes.h>
#include <core/viewport/snapping/SnappingManager.h>

namespace Core {

IMPLEMENT_ABSTRACT_PLUGIN_CLASS(ViewportInputHandler, PluginClass)
IMPLEMENT_ABSTRACT_PLUGIN_CLASS(SimpleInputHandler, ViewportInputHandler)

/******************************************************************************
* Updates the cursor in the viewport.
******************************************************************************/
void ViewportInputHandler::updateCursor()
{
	if(VIEWPORT_INPUT_MANAGER.currentHandler() == NULL) return;
	if(VIEWPORT_INPUT_MANAGER.currentHandler() == this || VIEWPORT_INPUT_MANAGER.currentHandler() == temporaryNavMode) {
		if(APPLICATION_MANAGER.guiMode())
			MAIN_FRAME->viewportPanel()->updateViewportCursor();
	}
}

/******************************************************************************
* This is called by the system after the input handler has become the active handler.
******************************************************************************/
void ViewportInputHandler::onActivated()
{
}

/******************************************************************************
* This is called by the system after the input handler is no longer the active handler.
******************************************************************************/
void ViewportInputHandler::onDeactivated()
{
	if(temporaryNavMode != NULL) {
		temporaryNavMode->onDeactivated();
		temporaryNavMode = NULL;
	}
}

/******************************************************************************
* Handles the mouse down event for the given viewport.
******************************************************************************/
void ViewportInputHandler::onMouseDown(Viewport& vp, QMouseEvent* event)
{
	if(event->button() == Qt::RightButton) {
		if(handlerActivationType() != EXCLUSIVE)
			VIEWPORT_INPUT_MANAGER.removeInputHandler(this);
	}
	else if(event->button() == Qt::MidButton && VIEWPORT_INPUT_MANAGER.currentHandler() == this) {
		if(event->modifiers().testFlag(Qt::ShiftModifier) == false) 
			temporaryNavMode = PanMode::instance();
		else
			temporaryNavMode = OrbitMode::instance();
		temporaryNavMode->onActivated();
		temporaryNavMode->onMouseDown(vp, event);
		updateCursor();
	}
}

/******************************************************************************
* Handles the mouse up event for the given viewport.
******************************************************************************/
void ViewportInputHandler::onMouseUp(Viewport& vp, QMouseEvent* event)
{
	if(temporaryNavMode != NULL) {
		temporaryNavMode->onMouseUp(vp, event);
		temporaryNavMode->onDeactivated();
		temporaryNavMode = NULL;
		updateCursor();
	}
}

/******************************************************************************
* Handles the mouse move event for the given viewport.
******************************************************************************/
void ViewportInputHandler::onMouseMove(Viewport& vp, QMouseEvent* event)
{
	if(temporaryNavMode != NULL) {
		temporaryNavMode->onMouseMove(vp, event);
	}
}

/******************************************************************************
* Handles the mouse wheel event for the given viewport.
******************************************************************************/
void ViewportInputHandler::onMouseWheel(Viewport& vp, QWheelEvent* event)
{
	ZoomMode::instance()->Zoom(vp, (FloatType)event->delta());
}

////////////////////////////////////////////////////////////////////////////////////////////////

/******************************************************************************
* Is called when the user aborts the operation by clicking the right
* mouse button or activating another input mode.
******************************************************************************/
void SimpleInputHandler::onAbort()
{
	if(viewport()) {
		viewport()->releaseMouse();
		_viewport = NULL;
	}
}

/******************************************************************************
* Is called when the user finishes the operation. 
******************************************************************************/
void SimpleInputHandler::onFinish()
{
	if(viewport()) {
		viewport()->releaseMouse();
		_viewport = NULL;
	}
}

/******************************************************************************
* Is called when the input is deactivated.
******************************************************************************/
void SimpleInputHandler::onDeactivated()
{
	ViewportInputHandler::onDeactivated();
	if(viewport()) onAbort();

	// Cleanup
	SNAPPING_MANAGER.clearLastSnapPoint();
	if(APPLICATION_MANAGER.guiMode())
		MAIN_FRAME->viewportPanel()->unsetCursor();
}

/******************************************************************************
* Handles the mouse down event for the given viewport.
******************************************************************************/
void SimpleInputHandler::onMouseDown(Viewport& vp, QMouseEvent* event)
{
	if(event->button() == Qt::RightButton) {
		if(viewport()) onAbort();
		else ViewportInputHandler::onMouseDown(vp, event);
	}
	else if(event->button() == Qt::LeftButton) {
		OVITO_ASSERT(viewport() == NULL || viewport() == &vp);
		if(viewport() == NULL) {
			_viewport = &vp;
			vp.grabMouse();
		}
		onMousePressed(event);
	}
	else ViewportInputHandler::onMouseDown(vp, event);
}

/******************************************************************************
* Handles the mouse up event for the given viewport.
******************************************************************************/
void SimpleInputHandler::onMouseUp(Viewport& vp, QMouseEvent* event)
{
	if(viewport() != NULL) onMouseReleased(event);
	else ViewportInputHandler::onMouseUp(vp, event);
}

/******************************************************************************
* Handles the mouse move event for the given viewport.
******************************************************************************/
void SimpleInputHandler::onMouseMove(Viewport& vp, QMouseEvent* event)
{
	if(viewport() != NULL)
		onMouseDrag(event);
	else {
		onMouseFreeMove(vp, event);
		ViewportInputHandler::onMouseMove(vp, event);
	}
}

/******************************************************************************
* Handles the mouse leave event for the given viewport.
******************************************************************************/
void SimpleInputHandler::onMouseLeave(Viewport& vp, QEvent* event)
{
	ViewportInputHandler::onMouseLeave(vp, event);
	SNAPPING_MANAGER.clearLastSnapPoint();
}

/******************************************************************************
* Show a preview snapping marker in the viewport.
* This method should be called from the onMouseFreeMove() implementation.
******************************`************************************************/
void SimpleInputHandler::snapPreview(Viewport& vp, QMouseEvent* event)
{
	Point3 p;
	vp.snapPoint(event->pos(), p);
	
	// Redraw viewports immediately to show the snapping marker in all viewports.
	VIEWPORT_MANAGER.updateViewports();
	VIEWPORT_MANAGER.processViewportUpdates();	
}

};
