///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __ASSIGN_COLOR_MODIFIER_H
#define __ASSIGN_COLOR_MODIFIER_H

#include <core/Core.h>
#include <core/scene/animation/controller/Controller.h>
#include <atomviz/AtomViz.h>
#include "../AtomsObjectModifierBase.h"

namespace AtomViz {

/******************************************************************************
* This modifier assigns a single color to the selected atoms of the input object.
******************************************************************************/
class ATOMVIZ_DLLEXPORT AssignColorModifier : public AtomsObjectModifierBase
{
public:

	/// Default constructor.
	AssignColorModifier(bool isLoading = false);

	/// Asks the modifier for its validity interval at the given time.
	virtual TimeInterval modifierValidity(TimeTicks time);

	/// Returns the color that is assigned to the selected atoms.
	Color color() const { return colorCtrl ? (Color)colorCtrl->getCurrentValue() : Color(0,0,0); }
	/// Sets the color that is assigned to the selected atoms.
	void setColor(const Color& color) { if(colorCtrl) colorCtrl->setCurrentValue(color); }
	/// Returns the controller for the color that is assigned to the selected atoms.
	VectorController* colorController() const { return colorCtrl; }
	/// Sets the controller for the color that is assigned to the selected atoms.
	void setColorController(const VectorController::SmartPtr& ctrl) { colorCtrl = ctrl; }

protected:

	/// Modifies the atoms object. The time interval passed
	/// to the function is reduced to the interval where the modified object is valid/constant.
	virtual EvaluationStatus modifyAtomsObject(TimeTicks time, TimeInterval& validityInterval);

	/// This controller stores the constant color to be assigned to all atoms.
	ReferenceField<VectorController> colorCtrl;

private:

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(AssignColorModifier)
	DECLARE_REFERENCE_FIELD(colorCtrl)
};

/******************************************************************************
* A properties editor for the AssignColorModifier class.
******************************************************************************/
class ATOMVIZ_DLLEXPORT AssignColorModifierEditor : public PropertiesEditor
{
protected:

	/// Creates the user interface controls for the editor.
	virtual void createUI(const RolloutInsertionParameters& rolloutParams);

private:

	Q_OBJECT
	DECLARE_PLUGIN_CLASS(AssignColorModifierEditor)
};

};	// End of namespace AtomViz

#endif // __ASSIGN_COLOR_MODIFIER_H
