/*
 *  Copyright (c) 2007 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Module.h"

#include <sstream>
#include <fstream>

#include "GTLCore/CompilationMessage.h"
#include "GTLCore/CompilationMessages.h"
#include "GTLCore/Function.h"
#include "GTLCore/Parameter.h"
#include "GTLCore/Macros_p.h"
#include "GTLCore/VirtualMachine_p.h"
#include "GTLCore/ModuleData_p.h"
#include "GTLCore/Type.h"
#include <GTLCore/LLVMBackend/ContextManager_p.h>

#include "Compiler.h"
#include "Debug.h"

using namespace OpenCTL;

// LLVM
#include <llvm/Module.h>
#include <llvm/Bitcode/ReaderWriter.h>
#ifdef LLVM_27_OR_28
#include <llvm/System/Path.h>
#include <llvm/System/DynamicLibrary.h>
#else
#include <llvm/Support/Path.h>
#include <llvm/Support/DynamicLibrary.h>
#endif
#include <llvm/Module.h>
#include <llvm/LLVMContext.h>

struct Module::Private {
  Private() : moduleData(0), isStandardLibrary(false) {}
  GTLCore::String name;
  GTLCore::String nameSpace;
  GTLCore::String source;
  bool compiled;
  GTLCore::ModuleData* moduleData;
  GTLCore::CompilationMessages compilationErrors;
  bool isStandardLibrary;
};

Module::Module() : d(new Private)
{
  d->compiled = false;
  d->name = "";
}

Module::~Module()
{
  if(d->moduleData and d->moduleData->llvmLinkedModule())
  {
    GTLCore::VirtualMachine::instance()->unregisterModule( d->moduleData->llvmLinkedModule() );
  }
  delete d->moduleData;
  delete d;
}

GTLCore::String Module::name() const
{
  return d->name;
}

GTLCore::String Module::nameSpace() const
{
  return d->nameSpace;
}

void Module::setSource(const GTLCore::String& name, const GTLCore::String& source)
{
  d->name = name;
  d->source = source;
}

void Module::loadFromFile(const GTLCore::String& fileName)
{
  d->isStandardLibrary = fileName.endWith( "ctlstdlib.ctl" );
  d->source = "";
  d->name = llvm::sys::Path((const std::string&)fileName).getBasename().str();
  std::ifstream in;
  in.open(fileName.c_str() );
  if(not in)
  {
    OCTL_DEBUG( "Impossible to open file " << fileName );
    return;
  }
  std::string str;
  std::getline(in,str);
  while ( in ) {
    d->source += str;
    d->source += "\n";
    std::getline(in,str);
  }
}

std::list<const GTLCore::Function*> Module::functions(const GTLCore::String& name) const
{
  if( d->moduleData )
  {
    const std::list<GTLCore::Function*>* fs = d->moduleData->function( nameSpace(), name);
    if(fs )
    {
      std::list<const GTLCore::Function*> fsr;
      foreach(const GTLCore::Function* f, *fs)
      {
        fsr.push_back(f);
      }
      return fsr;
    }
  }
  return std::list<const GTLCore::Function*>();
}

const GTLCore::Function* Module::function(const GTLCore::String& name, std::vector<GTLCore::Parameter>& arguments) const
{
  return d->moduleData->function(d->nameSpace, name, arguments);
}

void Module::compile()
{
  if(d->source.isEmpty()) return;
  if(d->moduleData and d->moduleData->llvmLinkedModule())
  {
    GTLCore::VirtualMachine::instance()->unregisterModule( d->moduleData->llvmLinkedModule());
  }
  delete d->moduleData;
  llvm::LLVMContext& context = *LLVMBackend::ContextManager::context(); // TODO someday (llvm 2.7 ?) use own context
  llvm::Module* llvmModule = new llvm::Module((const std::string&)d->name, context);
  d->moduleData = new GTLCore::ModuleData(llvmModule);

  Compiler c;
  bool result = c.compile(d->isStandardLibrary, d->source, d->name, d->moduleData, llvmModule );

  if(result)
  {
    d->compiled = true;
    llvm::sys::DynamicLibrary::LoadLibraryPermanently( _OPENCTL_LIB_, 0 ); // This needed because when OpenCTL is used in a plugins, OpenCTL symbols aren't loaded globally and then llvm can't find them
    d->moduleData->doLink();
    GTLCore::VirtualMachine::instance()->registerModule( d->moduleData->llvmLinkedModule() );
  } else {
    delete d->moduleData;
    d->moduleData = 0;
    d->compilationErrors = c.errorMessages();
  }
}

bool Module::isCompiled() const
{
  return d->compiled;
}

GTLCore::CompilationMessages Module::compilationMessages() const
{
  return d->compilationErrors;
}

GTLCore::String Module::asmSourceCode() const
{
  std::ostringstream os;
  os << *d->moduleData->llvmLinkedModule() << std::endl;
  return os.str();
}

std::list<GTLCore::Function*> Module::functions()
{
  return d->moduleData->functions();
}

const GTLCore::ModuleData* Module::data() const
{
  return d->moduleData;
}

const GTLCore::TypesManager* Module::typesManager() const
{
  return d->moduleData->typesManager();
}
