/*
 * Copyright 2013 Canonical Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "cachingprovider.h"

#include <QDebug>
#include <QFileInfo>
#include <QEventLoop>
#include <QImageReader>
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QNetworkAccessManager>
#include <QCryptographicHash>
#include <QDir>
#include <QStandardPaths>

/*
  CachingProvider is a implementation of QQuickImageProvider that caches the images
  it provides so that if the original disappears we can still display them.
  The uri that should be used is: "image://cache/<image_url>" where image_url is the
  url that you would use normally without the cache

  Example:
  \qml
    Image {
      source: "image://cache/http://example.org/example.png"
    }
  \endqml
*/

CachingProvider::CachingProvider()
    : QQuickImageProvider(QQuickImageProvider::Image)
{
}

QImage CachingProvider::downloadImage(const QUrl &url)
{
    QNetworkAccessManager networkMgr;
    int followedRedirects = 0;
    QUrl currentUrl(url);

    while (followedRedirects < REDIRECT_LIMIT) {
        QNetworkReply* reply = networkMgr.get(QNetworkRequest(currentUrl));

        /* The requestImage method is called in a separate thread and needs
         * to return a QImage when finished. Therefore the only way to
         * use QNetworkAccessManager to download the image is to wait for it
         * to emit the finished() signal by spawning a new event loop */
        QEventLoop loop;
        QObject::connect(reply, SIGNAL(finished()), &loop, SLOT(quit()));
        loop.exec();

        QVariant attribute = reply->attribute(QNetworkRequest::RedirectionTargetAttribute);
        QUrl redirect = attribute.toUrl();
        if (redirect.isEmpty()) {
            if (reply->error() != QNetworkReply::NoError) {
                qWarning() << "FAILED TO DOWNLOAD" << reply->errorString();
            }

            /* If the server reported an error (such as the file not existing)
             * or is not reachable, then QImageReader::read will fail and return
             * a QImage where isNull() is true, which when returned from the image
             * provider will cause QML to print an appropriate warning. */
            QImageReader reader(reply);
            return reader.read();
        } else {
            currentUrl = redirect;
            followedRedirects++;
        }
    }

    /* We followed too many redirects, and there's one more. Give up */
    qWarning() << "Failed to download the image due to too many redirects.";
    return QImage();
}

QImage CachingProvider::requestImage(const QString &id, QSize *size, const QSize& requestedSize)
{
    QUrl url(id);
    bool canCache = true;

    QDir dataLocation(QStandardPaths::writableLocation(QStandardPaths::DataLocation));
    if (!dataLocation.exists()) {
        if (!QDir::root().mkpath(dataLocation.absolutePath())) {
            qWarning() << "Failed to access the application directory. Images will not be cached.";
            canCache = false;
        }
    }

    QString hash = QString(QCryptographicHash::hash(url.toString().toUtf8(), QCryptographicHash::Md5).toHex());
    QString extension = QFileInfo(url.path()).suffix();
    QFileInfo cacheFile = dataLocation.filePath(hash + "." + extension);

    QImage image;
    bool needDownload = false;

    if (cacheFile.exists()) {
        if (!image.load(cacheFile.absoluteFilePath())) {
            qWarning() << "Failed to load file from cache at path:" << cacheFile.absoluteFilePath();
            needDownload = true;
        }
    } else needDownload = true;

    if (needDownload) {
        if (url.scheme() != "file") {
            image = downloadImage(url);
        } else {
            image.load(url.path());
        }

        if (canCache && !image.isNull()) {
            if (!image.save(cacheFile.absoluteFilePath())) {
                qWarning() << "Failed to save file to cache with path:" << cacheFile.absoluteFilePath();
            }
        }
    }

    *size = image.size();
    if (!image.isNull() && requestedSize.isValid() && (image.size() != requestedSize)) {
        return image.scaled(requestedSize, Qt::IgnoreAspectRatio, Qt::SmoothTransformation);
    } else {
        return image;
    }
}
