/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var isNumberArray = require( '@stdlib/assert/is-number-array' ).primitives;
var isTypedArrayLike = require( '@stdlib/assert/is-typed-array-like' );
var setReadOnly = require( '@stdlib/utils/define-read-only-property' );
var fCDF = require( './../../base/dists/f/cdf' );
var fQuantile = require( './../../base/dists/f/quantile' );
var variance = require( './../../base/variance' );
var min = require( '@stdlib/math/base/special/min' );
var PINF = require( '@stdlib/constants/float64/pinf' );
var validate = require( './validate.js' );
var print = require( './print.js' ); // eslint-disable-line stdlib/no-redeclare


// MAIN //

/**
* Computes a two-sample F-test for equal variances.
*
* @param {NumericArray} x - first data array
* @param {NumericArray} y - second data array
* @param {Options} [options] - function options
* @param {number} [options.alpha=0.05] - significance level
* @param {string} [options.alternative='two-sided'] - alternative hypothesis (`two-sided`, `less` or `greater`)
* @param {PositiveNumber} [options.ratio=1] - ratio of population variances under H0
* @throws {TypeError} x argument has to be a typed array or array of numbers
* @throws {TypeError} y argument has to be a typed array or array of numbers
* @throws {TypeError} options has to be simple object
* @throws {TypeError} alpha option has to be a number primitive
* @throws {RangeError} alpha option has to be a number in the interval `[0,1]`
* @throws {TypeError} alternative option has to be a string primitive
* @throws {Error} alternative option must be `two-sided`, `less` or `greater`
* @throws {TypeError} ratio option has to be a number primitive
* @returns {Object} test result object
*
* @example
* var x = [ 610, 610, 550, 590, 565, 570 ];
* var y = [ 560, 550, 580, 550, 560, 590, 550, 590 ];
*
* var out = vartest( x, y );
*/
function vartest( x, y, options ) {
	var estimate;
	var alpha;
	var ratio;
	var beta;
	var cint;
	var opts;
	var pval;
	var stat;
	var xvar;
	var yvar;
	var alt;
	var err;
	var out;
	var dfX;
	var dfY;

	if ( !isTypedArrayLike( x ) && !isNumberArray( x ) ) {
		throw new TypeError( 'invalid argument. First argument `x` must be a numeric array. Value: `' + x + '`.' );
	}
	if ( !isTypedArrayLike( y ) && !isNumberArray( y ) ) {
		throw new TypeError( 'invalid argument. Second argument `y` must be a numeric array. Value: `' + y + '`.' );
	}
	opts = {};
	if ( options ) {
		err = validate( opts, options );
		if ( err ) {
			throw err;
		}
	}
	ratio = opts.ratio || 1.0;
	if ( opts.alpha === void 0 ) {
		alpha = 0.05;
	} else {
		alpha = opts.alpha;
	}
	if ( alpha < 0.0 || alpha > 1.0 ) {
		throw new RangeError( 'invalid argument. Option `alpha` must be a number in the range 0 to 1. Value: `' + alpha + '`.' );
	}
	dfX = x.length - 1;
	dfY = y.length - 1;

	xvar = variance( x.length, 1, x, 1 );
	yvar = variance( y.length, 1, y, 1 );

	estimate = xvar / yvar;
	stat = estimate / ratio;
	pval = fCDF( stat, dfX, dfY );

	alt = opts.alternative || 'two-sided';
	switch ( alt ) {
	case 'two-sided':
		pval = 2.0 * min( pval, 1.0 - pval );
		beta = alpha / 2.0;
		cint = [
			estimate / fQuantile( 1.0 - beta, dfX, dfY ),
			estimate / fQuantile( beta, dfX, dfY )
		];
		break;
	case 'greater':
		pval = 1.0 - pval;
		cint = [
			estimate / fQuantile( 1.0 - alpha, dfX, dfY ),
			PINF
		];
		break;
	case 'less':
		cint = [
			0.0,
			estimate / fQuantile( alpha, dfX, dfY )
		];
		break;
	default:
		throw new Error( 'Invalid option. `alternative` must be either `two-sided`, `less` or `greater`. Value: `' + alt + '`' );
	}
	out = {};
	setReadOnly( out, 'rejected', pval <= alpha );
	setReadOnly( out, 'alpha', alpha );
	setReadOnly( out, 'pValue', pval );
	setReadOnly( out, 'statistic', stat );
	setReadOnly( out, 'ci', cint );
	setReadOnly( out, 'alternative', alt );
	setReadOnly( out, 'xvar', xvar );
	setReadOnly( out, 'yvar', yvar );
	setReadOnly( out, 'dfX', dfX );
	setReadOnly( out, 'dfY', dfY );
	setReadOnly( out, 'method', 'F test for comparing two variances' );
	setReadOnly( out, 'nullValue', ratio );
	setReadOnly( out, 'print', print );
	return out;
}


// EXPORTS //

module.exports = vartest;
