/* send_network.c
  
   The "send_network" and "send_tcl" NetExpect commands are implemented here.

   Copyright (C) 2007, 2008, 2009, 2010 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"
#include "nexp_speakers.h"
#include "util-tcl.h"

static int dry_run;
static int count;
static int fixups;
static pcb_t pcb;

static int
process_options(Tcl_Interp *interp, int objc, Tcl_Obj * const *objv)
{
    int i, index;
    char *pdudef = NULL;
    double delay;
    char errbuf[PDU_ERRBUF_SIZE];
    static const char *options[] = {
	"-n", "-o", "-count", "-delay", "-rate", "-fixups", NULL
    };
    enum options {
	OPT_DRYRUN, OPT_SPEAKER, OPT_COUNT, OPT_DELAY, OPT_RATE, OPT_FIXUPS
    };

    /*
     * Parse command-line arguments.
     */
    for (i = 1; i < objc && *Tcl_GetString(objv[i]) == '-'; i++) {
	if (Tcl_GetIndexFromObj(interp, objv[i], options, "option", 0, &index)
	    != TCL_OK)
	    return -1;

	switch ( (enum options) index) {
	case OPT_DRYRUN:
	    dry_run = 1;
	    break;
	case OPT_SPEAKER:
	    if (++i >= objc) {
		Tcl_WrongNumArgs(interp, 1, objv, "-o speaker");
		goto error;
	    }

	    pcb.speaker = lookup_speaker(Tcl_GetString(objv[i]) );
	    if (!pcb.speaker) {
		nexp_error(interp, "No speaker named \"%s\". Use "
			   "\"spawn_network -info\" to find out existing "
			   "speakers.", Tcl_GetString(objv[i]) );
		goto error;
	    }
	    break;
	case OPT_COUNT:
	    if (++i >= objc) {
		Tcl_WrongNumArgs(interp, 1, objv, "-c npackets");
		goto error;
	    }

	    if (Tcl_GetIntFromObj(interp, objv[i], &count) != TCL_OK)
		goto error;
	    break;
	case OPT_DELAY:
	    if (++i >= objc) {
		Tcl_WrongNumArgs(interp, 1, objv, "-d seconds");
		goto error;
	    }

	    if (Tcl_GetDoubleFromObj(interp, objv[i], &delay) != TCL_OK)
		goto error;

	    pcb.delay.tv_sec = delay;
	    pcb.delay.tv_usec = (delay - pcb.delay.tv_sec)*1000000UL;
	    break;
	case OPT_RATE:
	    if (++i >= objc) {
		Tcl_WrongNumArgs(interp, 1, objv, "-r PPS");
		goto error;
	    }

	    /* Convert a packets-per-second rate to a usecs delay */
	    if (Tcl_GetDoubleFromObj(interp, objv[i], &delay) != TCL_OK)
		goto error;

	    if (delay == 0.0) {
		nexp_error(interp, "Rate can't be 0 packets per second.");
		goto error;
	    }

	    delay = 1/delay;

	    pcb.delay.tv_sec = delay;
	    pcb.delay.tv_usec = (delay - pcb.delay.tv_sec)*1000000UL;
	    break;
	case OPT_FIXUPS:
	    fixups++; /* We're doing fixups of this packet after building */
	    break;
	}
    }

    /*
     * We treat whatever is left on the command line, i.e. anything that
     * is not an option (anything that doesn't start with '-'), as a PDU
     * definition.
     */
    pdudef = copy_objv(objc - i, &objv[i]);
    if (!pdudef) {
	fprintf(stderr, "\
usage: send_network [-n] [-fixups] [-o <speaker ID>] [-count <count>]\n\
          [-delay <usecs>] [-rate <pps>] <PDU definition>");
	return -1;
    }

#ifdef DEBUG
    printf("PDU definition = %s\n", pdudef);
#endif

    if ( (pcb.pdu = pb_parsedef(pdudef, errbuf) ) == NULL) {
	nexp_error(interp, "%s", errbuf);
	goto error;
    }

    pcb.def = pdudef;

    return 0;

error:
    if (pdudef)
	free(pdudef);

    return -1;
}

static int
NExp_SendNetworkObjCmd(ClientData clientData _U_, Tcl_Interp *interp,
		       int objc, Tcl_Obj * const *objv)
{
    size_t packet_size;
    uint8_t *packet;
    int i, j;
    Tcl_Obj *obj;
    int packet_count, packets_sent;
    const char *pdu_name;
    const GNode *next_pdu;

    count = 0;
    dry_run = 0;
    fixups = 0;

    memset(&pcb, 0, sizeof(pcb) );

    if (process_options(interp, objc, objv) == -1)
	return TCL_ERROR;

    /*
     * The command line has been processed, now we need to make sure we have
     * the minimum options to be able to run.
     *
     * We only do this if we will actually send something out, i.e. if
     * we are not doing a dry run.
     */
    if (dry_run) {
	pcb_dump(&pcb, vflag);
	printf("\n");
	pb_dumppdu(pcb.pdu);
	goto done;
    }

    /*
     * Make sure the PCB has an assigned speaker: if the user has not
     * explicitely specified a speaker, then we use the default speaker,
     * which is referred to by name via the Tcl variable "speaker_id"
     * (SPEAKER_SPAWN_ID_VARNAME).
     */
    if (!pcb.speaker) {
	pcb.speaker = lookup_speaker(nexp_get_var(interp,
					SPEAKER_SPAWN_ID_VARNAME) );
	if (!pcb.speaker) {
	    nexp_error(interp, "Can't find a suitable speaker! Use "
			       "spawn_network to create one.");
	    return TCL_ERROR;
	}
    }

    packet_count = count ? count : pb_permutation_count(pcb.pdu);
    packet = xmalloc(pb_len(pcb.pdu) );

    for (packets_sent = i = 0; i < packet_count; i++) {
	packet_size = pb_build(pcb.pdu, packet, NULL);

	if (fixups) {
	    /*
	     * User requested that fixups are performed on this
	     * packet. Rely on libdnet for this. Only perform
	     * fixups for IPv4 or IPv6 packets.
	     */

	    pdu_name = pb_getname(pcb.pdu);

	    if (!strcmp(pdu_name, "ether") ) {
		next_pdu = pb_nextpdu(pcb.pdu);

		if (next_pdu) {
		    pdu_name = pb_getname(next_pdu);

		    if (!strcmp(pdu_name, "ip") )
			ip_checksum(packet + sizeof(struct eth_hdr),
				    packet_size - sizeof(struct eth_hdr) );
		    else if (!strcmp(pdu_name, "ip6") )
			ip6_checksum(packet + sizeof(struct eth_hdr),
				     packet_size - sizeof(struct eth_hdr) );
		}
	    } else if (!strcmp(pdu_name, "ip") )
		ip_checksum(packet, packet_size);
	    else if (!strcmp(pdu_name, "ip6") )
		ip6_checksum(packet, packet_size);
	}

	for (j = 0; j < pcb.repeat_count + 1; j++) {
	    /*
	     * Handle inter-packet sending delay.
	     */
	    if (timerisset(&pcb.delay)
		&& timerisset(&pcb.speaker->ts) ) {
		struct timeval now, later;

		timeradd(&pcb.speaker->ts, &pcb.delay, &later);

		gettimeofday(&now, NULL);
		/*
		 * Some versions of Solaris (seen in Solaris 8) don't
		 * like using timercmp() with "<=" or ">=". Rearranging
		 * the operands and just using '>' or '<' works around
		 * this limitation.
		 */
		while (timercmp(&later, &now, >) )
		    gettimeofday(&now, NULL);
	    }

	    nexp_pdu_output(&pcb, packet, packet_size);

	    if (vflag)
		putchar('!');

	    packets_sent++;
	}
    }

    if (vflag)
	putchar('\n');

    free(packet);

    /*
     * The send_network command returns the number of packets that were
     * sent.
     */
    obj = Tcl_NewIntObj(packets_sent);
    Tcl_SetObjResult(interp, obj);

done:
    if (!pcb.keep_after_send)
	pcb_destroy(&pcb);

    return TCL_OK;
}

static int
NExp_SendTclObjCmd(ClientData clientData _U_, Tcl_Interp *interp,
		   int objc, Tcl_Obj * const *objv)
{
    size_t packet_size;
    uint8_t *packet;
    int i, j;
    Tcl_Obj *obj, *list;
    int packet_count;
    const char *pdu_name;
    const GNode *next_pdu;

    count = 0;
    dry_run = 0;
    fixups = 0;

    memset(&pcb, 0, sizeof(pcb) );

    if (process_options(interp, objc, objv) == -1)
	return TCL_ERROR;

    /*
     * The command line has been processed, now we need to make sure we have
     * the minimum options to be able to run.
     *
     * We only do this if we will actually send something out, i.e. if
     * we are not doing a dry run.
     */
    if (dry_run) {
	pcb_dump(&pcb, vflag);
	printf("\n");
	pb_dumppdu(pcb.pdu);
	goto done;
    }

    packet_count = count ? count : pb_permutation_count(pcb.pdu);
    packet = xmalloc(pb_len(pcb.pdu) );

    list = Tcl_NewListObj(0, NULL);

    for (i = 0; i < packet_count; i++) {
	packet_size = pb_build(pcb.pdu, packet, NULL);

	if (fixups) {
	    /*
	     * User requested that fixups are performed on this
	     * packet. Rely on libdnet for this. Only perform
	     * fixups for IPv4 or IPv6 packets.
	     */

	    pdu_name = pb_getname(pcb.pdu);

	    if (!strcmp(pdu_name, "ether") ) {
		next_pdu = pb_nextpdu(pcb.pdu);

		if (next_pdu) {
		    pdu_name = pb_getname(next_pdu);

		    if (!strcmp(pdu_name, "ip") )
			ip_checksum(packet + sizeof(struct eth_hdr),
				    packet_size - sizeof(struct eth_hdr) );
		    else if (!strcmp(pdu_name, "ip6") )
			ip6_checksum(packet + sizeof(struct eth_hdr),
				     packet_size - sizeof(struct eth_hdr) );
		}
	    } else if (!strcmp(pdu_name, "ip") )
		ip_checksum(packet, packet_size);
	    else if (!strcmp(pdu_name, "ip6") )
		ip6_checksum(packet, packet_size);
	}

	for (j = 0; j < pcb.repeat_count + 1; j++) {
	    /*
	     * Handle inter-packet sending delay.
	     */
	    if (timerisset(&pcb.delay)
		&& timerisset(&pcb.speaker->ts) ) {
		struct timeval now, later;

		timeradd(&pcb.speaker->ts, &pcb.delay, &later);

		gettimeofday(&now, NULL);
		/*
		 * Some versions of Solaris (seen in Solaris 8) don't
		 * like using timercmp() with "<=" or ">=". Rearranging
		 * the operands and just using '>' or '<' works around
		 * this limitation.
		 */
		while (timercmp(&later, &now, >) )
		    gettimeofday(&now, NULL);
	    }

	    /*
	     * Create new ByteArray object...
	     *
	     * Important: can't use Tcl_NewStringObj() here since then Tcl will
	     * perform Unicode conversion in some situations. We want to return
	     * raw, binary data, so we must use Tcl_NewByteArrayObj().
	     */
	    obj = Tcl_NewByteArrayObj(packet, packet_size);

	    /*
	     * ... and append it to the list.
	     */
	    Tcl_ListObjAppendElement(interp, list, obj);

	    if (vflag)
		putchar('!');
	}
    }

    if (vflag)
	putchar('\n');

    free(packet);

    Tcl_SetObjResult(interp, list);

done:
    pcb_destroy(&pcb);

    return TCL_OK;
}

static struct nexp_cmd_data cmd_data[] = {
    {"send_network", NExp_SendNetworkObjCmd, NULL, 0, 0},
    {"send_tcl", NExp_SendTclObjCmd, NULL, 0, 0},

    {NULL, NULL, NULL, 0, 0}
};

void
nexp_init_send_cmds(Tcl_Interp *interp)
{
    nexp_create_commands(interp, cmd_data);
}
