/* pdu-ipgeneric.c

   A PDU builder for IPv4 packets that uses the generic builder as much
   as possible. This is provided as an example/proof of concept. For
   another approach to writing PDU builders see pdu-ip.c.

   Copyright (C) 2007, 2008, 2009, 2010 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "pbuild-priv.h"
#include "pdu-ip.h"

static field_t record_route_opt_fields[] = {
    {
	.name = "length",
	.type = PDU_FTYPE_UINT8
    },
    {
	.name = "ptr",
	.type = PDU_FTYPE_UINT8
    },
    {
	.name = "router",
	.type = PDU_FTYPE_IP
    },
    {
	.name = NULL
    }
};

static field_t fields[] = {
    {
	.name = "version",
	.type = PDU_FTYPE_BITFIELD,
	.offset = 0, /* in bits */
	.size = 4, /* in bits */
	.defval = (defval_t []) { {.type = PDU_DEF_NUM, ._number = 4} }
    },
    {
	.name = "ihl",
	.type = PDU_FTYPE_BITFIELD,
	.offset = 4, /* in bits */
	.size = 4 /* in bits */
    },
    {
	.name = "tos",
	.type = PDU_FTYPE_UINT8,
	.offset = offsetof(struct ip_hdr, ip_tos)
    },
    {
	.name = "length",
	.type = PDU_FTYPE_UINT16,
	.offset = offsetof(struct ip_hdr, ip_len)
    },
    {
	.name = "id",
	.type = PDU_FTYPE_UINT16,
	.offset = offsetof(struct ip_hdr, ip_id),
	.defval = (defval_t []) { {.type = PDU_DEF_NUM, ._number = 1} }
    },
    {
	.name = "rf",
	.type = PDU_FTYPE_BIT,
	.offset = 48
    },
    {
	.name = "df",
	.type = PDU_FTYPE_BIT,
	.offset = 49
    },
    {
	.name = "mf",
	.type = PDU_FTYPE_BIT,
	.offset = 50
    },
    {
	.name = "fragoff",
	.type = PDU_FTYPE_BITFIELD,
	.offset = 51,
	.size = 13 
    },
    {
	.name = "ttl",
	.type = PDU_FTYPE_UINT8,
	.offset = offsetof(struct ip_hdr, ip_ttl),
	.defval = (defval_t []) {
	    {.type = PDU_DEF_NUM, ._number = IP_TTL_DEFAULT}
	}
    },
    {
	.name = "proto",
	.type = PDU_FTYPE_UINT8,
	.offset = offsetof(struct ip_hdr, ip_p)
    },
    {
	.name = "cksum",
	.type = PDU_FTYPE_UINT16,
	.offset = offsetof(struct ip_hdr, ip_sum)
    },
    {
	.name = "src",
	.type = PDU_FTYPE_IP,
	.offset = offsetof(struct ip_hdr, ip_src)
    },
    {
	.name = "dst",
	.type = PDU_FTYPE_IP,
	.offset = offsetof(struct ip_hdr, ip_dst)
    },
    {
	.name = "options",
	.type = PDU_FTYPE_BRACED_ARGS,
	.data = (field_t []) {
	    {
		.name = "nop",
		.type = PDU_FTYPE_BRACED_ARGS,
		.data = (field_t []) {
		    {
			.name = "type",
			.type = PDU_FTYPE_UINT8,
			.defval = (defval_t []) {
			    {.type = PDU_DEF_NUM, ._number = IP_OPT_EOL}
			}
		    },
		    {
			.name = NULL
		    }
		},
		.defval = NULL
	    },
	    {
		.name = "eol",
		.type = PDU_FTYPE_BRACED_ARGS,
		.data = (field_t []) {
		    {
			.name = "type",
			.type = PDU_FTYPE_UINT8,
			.defval = (defval_t []) {
			    {.type = PDU_DEF_NUM, ._number = IP_OPT_NOP}
			}
		    },
		    {
			.name = NULL
		    }
		},
		.defval = NULL
	    },
	    {
		.name = "raw",
		.type = PDU_FTYPE_DATA
	    },
	    {
		.name = "lsrr",
		.type = PDU_FTYPE_BRACED_ARGS,
		.data = record_route_opt_fields
	    },
	    {
		.name = "ssrr",
		.type = PDU_FTYPE_BRACED_ARGS,
		.data = record_route_opt_fields
	    },
	    {
		.name = "rr",
		.type = PDU_FTYPE_BRACED_ARGS,
		.data = record_route_opt_fields,
		.defval = NULL
	    },
	    {
		.name = NULL
	    }
	},
	.defval = NULL
    },
    {
	.name = NULL
    }
};

static void
postbuild(const GNode *pdu, void *dest, void *prev_pdu_hdr _U_)
{
    struct ip_hdr *ip;
    uint16_t u16;
    void *field;
    size_t opts_len, payload_len;

    ip = dest;

    opts_len = ( (struct node_data *) pdu->data)->_data_pdu.opts_len;
    payload_len = ( (struct node_data *) pdu->data)->_data_pdu.payload_len;

    /************** Handle IP total length **************/
    u16 = (field = _pb_pdata(pdu, "length") ) ? htons(num_next(field) )
			    : htons(sizeof(struct ip_hdr) + opts_len + payload_len);
    SSVAL(ip, offsetof(struct ip_hdr, ip_len), u16);

    /************** Handle IP checksum **************/
    if ( (field = _pb_pdata(pdu, "cksum") ) )
	u16 = htons(num_next(field) );
    else {
	/* Needs to be 0 before calculating the cksum per RFC 791 */
	SSVAL(ip, offsetof(struct ip_hdr, ip_sum), 0);

	u16 = _pb_cksum(dest, sizeof(struct ip_hdr) + opts_len);
    }
    SSVAL(ip, offsetof(struct ip_hdr, ip_sum), u16);

    /************** Handle IP Internet Header Length **************/
    u16 = (field = _pb_pdata(pdu, "ihl") ) ? num_next(field)
			    : (sizeof(struct ip_hdr) + opts_len)/4;
    ip->ip_hl |= u16;
}

/********************************************************************
 *			       Options                              *
 ********************************************************************/

static void
ipv4opt_raw(const GNode *option, void *dest, void *curr_pdu_hdr _U_,
	    void *prev_pdu _U_)
{
    struct payload *raw_option;

    raw_option = ( (struct node_data *) option->data)->_data_parm.data;

    memcpy(dest, raw_option->data, raw_option->len);
}

static void
ipv4opt_rr(const GNode *option _U_, void *dest, void *curr_pdu_hdr _U_,
	   void *prev_pdu _U_)
{
    GNode *child;
    struct node_data *node_data;
    uint8_t *len_ptr;
    const char *opt_name;
    numspec_t *field;

    opt_name = ( (struct node_data *) option->data)->name;

    if (!strcmp(opt_name, "rr") )
	*(uint8_t *) dest++ = IP_OPT_RR;
    else if (!strcmp(opt_name, "lsrr") )
	*(uint8_t *) dest++ = IP_OPT_LSRR;
    else
	*(uint8_t *) dest++ = IP_OPT_SSRR;

    len_ptr = (uint8_t *) dest++; /* Skip putting in a length for now */

    *(uint8_t *) dest++ = num_next(_pb_pdata(option, "ptr") );

    for (child = g_node_first_child(option);
	 child;
	 child = g_node_next_sibling(child) ) {
	node_data = child->data;

	/* We only care about "router"; everything else ("length",
	 * "type") is handled above.
	 */
	if (!strcmp(node_data->name, "router") ) {
	    SIVAL(dest, 0, ip_next(node_data->_data_parm.data) );
	    dest += sizeof(ip_addr_t);
	}
    }

    if ( (field = _pb_pdata(option, "length") ) )
	/* Put in fake length */
	*(uint8_t *) len_ptr = num_next(field);
    else 
	/* Calculate real length */
	*(uint8_t *) len_ptr = (uint8_t *) dest - len_ptr + 1;
}

/*
 * Returns size of IPv4 raw, rr, lsrr, or ssrr options. All other IPv4
 * options have fixed sizes and therefore don't need a .get_len method
 * in their option definition structures.
 */
static size_t
option_len(const GNode *option)
{
    struct payload *payload;
    const char *opt_name;
    size_t opt_size;
    struct node_data *node_data;
    GNode *child;

    opt_name = ( (struct node_data *) option->data)->name;

    if (!strcmp(opt_name, "raw") ) {
	payload = ( (struct node_data *) option->data)->_data_parm.data;
	opt_size = payload->len;
    } else {
	/* RR, LSRR, or SSRR */

	opt_size = 1 /* option type */ + 1 /* option length */
		   + 1 /* ptr */;

	for (child = g_node_first_child(option);
	     child;
	     child = g_node_next_sibling(child) ) {
	    node_data = child->data;

	    /* We only care about "router"; everything else ("length",
	     * "type", "ptr") is handled above.
	     */
	    if (!strcmp(node_data->name, "router") )
		opt_size += sizeof(ip_addr_t);
	}
    }

    return opt_size;
}


static const pdu_option_t pdu_option_ipv4raw = {
    .name = "raw",
    .description = "Raw option",
    .build = &ipv4opt_raw,
    .get_len = &option_len
};

static const pdu_option_t pdu_option_ipv4eol = {
    .name = "eol",
    .description = "End of option list",
    .len = sizeof(uint8_t)
};

static const pdu_option_t pdu_option_ipv4nop = {
    .name = "nop",
    .description = "No operation (pad)",
    .len = sizeof(uint8_t)
};

static const pdu_option_t pdu_option_ipv4rr = {
    .name = "rr",
    .description = "IPv4 record route",
    .build = &ipv4opt_rr,
    .get_len = &option_len
};

static const pdu_option_t pdu_option_ipv4lsrr = {
    .name = "lrrr",
    .description = "IPv4 loose route, record route",
    .build = &ipv4opt_rr,
    .get_len = &option_len
};

static const pdu_option_t pdu_option_ipv4ssrr = {
    .name = "srrr",
    .description = "IPv4 strict route, record route",
    .build = &ipv4opt_rr,
    .get_len = &option_len
};

static const pdu_t pdu_ipv4 = {
    .name = "ip-generic",
    .description = "IPv4 packet",
    .documented_in = "RFC 791",
    .len = sizeof(struct ip_hdr),
    .fields = fields,
    .options_class = "ip-generic",
    .postbuild = &postbuild
};

void
_pb_register_ipv4_generic(void)
{
    _pb_register_protocol(&pdu_ipv4);

    _pb_register_protocol_option("ip-generic", &pdu_option_ipv4raw);
    _pb_register_protocol_option("ip-generic", &pdu_option_ipv4eol);
    _pb_register_protocol_option("ip-generic", &pdu_option_ipv4nop);
    _pb_register_protocol_option("ip-generic", &pdu_option_ipv4rr);
    _pb_register_protocol_option("ip-generic", &pdu_option_ipv4lsrr);
    _pb_register_protocol_option("ip-generic", &pdu_option_ipv4ssrr);
}
