#!/usr/bin/perl
# SPDX-License-Identifier: BSD-2-Clause
# Copyright 1996-2016 The NASM Authors - All Rights Reserved

#
# version.pl
#
# Parse the NASM version file and produce appropriate macros
#
# The NASM version number is assumed to consist of:
#
# <major>.<minor>[.<subminor>][pl<patchlevel>]]<tail>
#
# ... where <tail> is not necessarily numeric, but if it is of the form
# -<digits> it is assumed to be a snapshot release.
#
# This defines the following macros:
#
# version.h:
# NASM_MAJOR_VER
# NASM_MINOR_VER
# NASM_SUBMINOR_VER	-- this is zero if no subminor
# NASM_PATCHLEVEL_VER	-- this is zero is no patchlevel
# NASM_SNAPSHOT		-- if snapshot
# NASM_VERSION_ID       -- version number encoded
# NASM_VER		-- whole version number as a string
#
# version.mac:
# __?NASM_MAJOR?__
# __?NASM_MINOR?__
# __?NASM_SUBMINOR?__
# __?NASM_PATCHLEVEL?__
# __?NASM_SNAPSHOT?__
# __?NASM_VERSION_ID?__
# __?NASM_VER?__
#

($what) = @ARGV;

$line = <STDIN>;
$line =~ s/\s+//g;

undef $maj;
undef $min;
undef $smin;
undef $plvl;
undef $tail;
$is_rc = 0;

if ( $line =~ /^([0-9]+)\.([0-9]+)(.*)$/ ) {
    $maj  = $1;
    $min  = $2;
    $tail = $3;
    if ( $tail =~ /^\.([0-9]+)(.*)$/ ) {
	$smin = $1;
	$tail = $2;
    }
    if ( $tail =~ /^(pl|\.)([0-9]+)(.*)$/ ) {
	$plvl = $2;
	$tail = $3;
    } elsif ( $tail =~ /^rc([0-9]+)(.*)$/ ) {
	$is_rc = 1;
	$plvl = $1;
	$tail = $2;
    }
} else {
    die "$0: Invalid input format\n";
}

if ($tail =~ /^\-([0-9]+)$/) {
    $snapshot = $1;
} else {
    undef $snapshot;
}

$nmaj = $maj+0;   $nmin = $min+0;
$nsmin = $smin+0; $nplvl = $plvl+0;

if ($is_rc) {
    $nplvl += 90;
    if ($nsmin > 0) {
	$nsmin--;
    } else {
	$nsmin = 99;
	if ($nmin > 0) {
	    $nmin--;
	} else {
	    $nmin = 99;
	    $nmaj--;
	}
    }
}

$nasm_id = ($nmaj << 24)+($nmin << 16)+($nsmin << 8)+$nplvl;

$mangled_ver = sprintf("%d.%02d", $nmaj, $nmin);
if ($nsmin || $nplvl || defined($snapshot)) {
    $mangled_ver .= sprintf(".%02d", $nsmin);
    if ($nplvl || defined($snapshot)) {
	$mangled_ver .= '.'.$nplvl;
    }
}
($mtail = $tail) =~ tr/-/./;
$mangled_ver .= $mtail;

if ( $what eq 'h' ) {
    print  "#ifndef NASM_VERSION_H\n";
    print  "#define NASM_VERSION_H\n";
    printf "#define NASM_MAJOR_VER      %d\n", $nmaj;
    printf "#define NASM_MINOR_VER      %d\n", $nmin;
    printf "#define NASM_SUBMINOR_VER   %d\n", $nsmin;
    printf "#define NASM_PATCHLEVEL_VER %d\n", $nplvl;
    if (defined($snapshot)) {
	printf "#define NASM_SNAPSHOT       %d\n", $snapshot;
    }
    printf "#define NASM_VERSION_ID     0x%08x\n", $nasm_id;
    printf "#define NASM_VER            \"%s\"\n", $line;
    print  "#endif /* NASM_VERSION_H */\n";
} elsif ( $what eq 'mac' ) {
    print  "STD: version\n";
    printf "%%define __?NASM_MAJOR?__ %d\n", $nmaj;
    printf "%%define __?NASM_MINOR?__ %d\n", $nmin;
    printf "%%define __?NASM_SUBMINOR?__ %d\n", $nsmin;
    printf "%%define __?NASM_PATCHLEVEL?__ %d\n", $nplvl;
    if (defined($snapshot)) {
	printf "%%define __?NASM_SNAPSHOT?__ %d\n", $snapshot;
    }
    printf "%%define __?NASM_VERSION_ID?__ 0%08Xh\n", $nasm_id;
    printf "%%define __?NASM_VER?__ \"%s\"\n", $line;
} elsif ( $what eq 'sed' ) {
    printf "s/\@\@NASM_MAJOR\@\@/%d/g\n", $nmaj;
    printf "s/\@\@NASM_MINOR\@\@/%d/g\n", $nmin;
    printf "s/\@\@NASM_SUBMINOR\@\@/%d/g\n", $nsmin;
    printf "s/\@\@NASM_PATCHLEVEL\@\@/%d/g\n", $nplvl;
    printf "s/\@\@NASM_SNAPSHOT\@\@/%d/g\n", $snapshot;	# Possibly empty
    printf "s/\@\@NASM_VERSION_ID\@\@/%d/g\n", $nasm_id;
    printf "s/\@\@NASM_VERSION_XID\@\@/%08x/g\n", $nasm_id;
    printf "s/\@\@NASM_VER\@\@/%s/g\n", $line;
    printf "s/\@\@NASM_MANGLED_VER\@\@/%s/g\n", $mangled_ver;
} elsif ( $what eq 'make' ) {
    printf "NASM_VER=%s\n", $line;
    printf "NASM_MAJOR_VER=%d\n", $nmaj;
    printf "NASM_MINOR_VER=%d\n", $nmin;
    printf "NASM_SUBMINOR_VER=%d\n", $nsmin;
    printf "NASM_PATCHLEVEL_VER=%d\n", $nplvl;
    printf "NASM_VERSION_ID=%d\n", $nasm_id;
    printf "NASM_VERSION_XID=%08x\n", $nasm_id;
    if (defined($snapshot)) {
	printf "NASM_SNAPSHOT=%d\n", $snapshot;
    }
} elsif ( $what eq 'nsis' ) {
    printf "!define VERSION \"%s\"\n", $line;
    printf "!define MAJOR_VER %d\n", $nmin;
    printf "!define MINOR_VER %d\n", $nmin;
    printf "!define SUBMINOR_VER %d\n", $nsmin;
    printf "!define PATCHLEVEL_VER %d\n", $nplvl;
    if (defined($snapshot)) {
	printf "!define SNAPSHOT_VER=%d\n", $snapshot;
    }
} elsif ( $what eq 'id' ) {
    print $nasm_id, "\n";	 # Print ID in decimal
} elsif ( $what eq 'xid' ) {
    printf "0x%08x\n", $nasm_id; # Print ID in hexadecimal
} elsif ( $what eq 'docsrc' ) {
    printf "\\M{version}{%s}\n", $line;
    printf "\\M{subtitle}{version %s}\n", $line;
} else {
    die "$0: Unknown output: $what\n";
}

exit 0;
