# libraryeditor.py
#
#   Copyright (C) 2004 Daniel Burrows <dburrows@debian.org>
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# Code to edit the list of directories that is contained within a
# library.

import config
import listenable

import gobject
import gtk
import operator
import os
import sets

from warnings import warn

def valid_library(lib):
    """Tests if lib is a valid library: a string or a list of strings."""
    if isinstance(lib, basestring):
        return True

    if isinstance(lib, list) and reduce(operator.__and__,
                                        map(lambda x:isinstance(x, basestring),
                                            lib),
                                        True):
        return True

    return False


# Defined libraries:
config.add_option('General', 'DefinedLibraries',
                  {},
                  lambda x:reduce(operator.__and__,
                                  map(lambda (key,val):isinstance(key, basestring) and valid_library(val),
                                      x.items()),
                                  True))

# The library to load on startup.  If None, load no library on startup.
config.add_option('General', 'DefaultLibrary',
                  None,
                  lambda x:x == None or isinstance(x, basestring))

library_edits=listenable.Listenable()
"""This signal is emitted when a library edit is committed.  Its
single argument is a list of tuples (old_name, new_name) where each
old_name is the name of a library, and the new_name is the library's
new name or None if it has been deleted."""

class LibraryEditor:
    """Wraps the dialog that edits the library definitions.

    Internal use, don't touch."""
    def __init__(self, glade_location):
        """Initializes the dialog from Glade and sets up bookkeeping
        information."""
        xml=gtk.glade.XML(glade_location, root='library_dialog')

        xml.signal_autoconnect({'new_library' : self.new_library,
                                'delete_library' : self.delete_library,
                                'rename_library' : self.rename_library,
                                'add_directory' : self.handle_add_directory,
                                'remove_directory' : self.remove_directory})

        # Widget extraction
        self.new_button=xml.get_widget('new')
        self.delete_button=xml.get_widget('delete')
        self.rename_button=xml.get_widget('rename')
        self.add_button=xml.get_widget('add')
        self.remove_button=xml.get_widget('remove')

        self.libraries_list=xml.get_widget('libraries_list')
        self.directories_list=xml.get_widget('directories_list')
        self.directories_label=xml.get_widget('directories_label')

        self.gui=xml.get_widget('library_dialog')

        self.add_button.set_sensitive(0)
        self.remove_button.set_sensitive(0)
        self.delete_button.set_sensitive(0)
        self.rename_button.set_sensitive(0)

        # Get the shared list of libraries.
        self.libraries={}

        for key,val in config.get_option('General', 'DefinedLibraries').iteritems():
            self.libraries[key]=sets.Set(val)
        # Used to "remember" what the currently selected library is,
        # to avoid unnecessary adjustments of other widgets.
        self.selected_library=None
        self.default_library=config.get_option('General', 'DefaultLibrary')
        self.filesel=None

        # Used to track the history of each library that originally existed.
        self.library_new_names={}
        self.library_orig_names={}

        for key in self.libraries.keys():
            self.library_new_names[key]=key
            self.library_orig_names[key]=key

        self.libraries_model=gtk.ListStore(gobject.TYPE_STRING, gobject.TYPE_BOOLEAN)
        self.directories_model=gtk.ListStore(gobject.TYPE_STRING)

        renderer=gtk.CellRendererText()
        renderer.set_property('editable', 1)
        renderer.connect('edited', self.handle_library_name_edited)
        col=gtk.TreeViewColumn('Library',
                               renderer,
                               text=0)
        self.libraries_list.append_column(col)
        renderer=gtk.CellRendererToggle()
        renderer.set_radio(1)
        renderer.set_property('activatable', 1)
        renderer.connect('toggled', self.handle_default_library_toggled)
        col=gtk.TreeViewColumn('Default',
                               renderer,
                               active=1)
        self.libraries_list.append_column(col)

        col=gtk.TreeViewColumn('Directories',
                               gtk.CellRendererText(),
                               text=0)
        self.directories_list.append_column(col)

        self.libraries_list.set_model(self.libraries_model)
        self.directories_list.set_model(self.directories_model)

        def sort_by_column_data(model, iter1, iter2):
            return cmp(model[iter1][0], model[iter2][0])

        self.libraries_model.set_default_sort_func(sort_by_column_data)
        self.libraries_model.set_sort_column_id(-1, gtk.SORT_ASCENDING)

        self.directories_model.set_default_sort_func(sort_by_column_data)
        self.directories_model.set_sort_column_id(-1, gtk.SORT_ASCENDING)

        libraries_selection=self.libraries_list.get_selection()
        libraries_selection.connect('changed', lambda *args:self.library_changed(libraries_selection))

        directories_selection=self.directories_list.get_selection()
        directories_selection.connect('changed', lambda *args:self.directory_changed(directories_selection))

        self.update_library_list()

        self.gui.connect('response', self.handle_response)

    def handle_response(self, dialog, response_id):
        """Deal with our own response, by pumping the changed value
        back into the configuration system."""

        assert(dialog == self.gui)

        if response_id == gtk.RESPONSE_OK:
            # Trigger library edit events:
            library_edits.call_listeners(self.library_new_names.items())

            # Copy and convert back to lists.
            new_libraries={}
            for key,val in self.libraries.iteritems():
                new_libraries[key]=list(val)

            config.set_option('General', 'DefinedLibraries', new_libraries)
            config.set_option('General', 'DefaultLibrary', self.default_library)

        dialog.destroy()
                         

    # Call this when the list of libraries needs to be regenerated
    # from scratch:
    def update_library_list(self):
        selected=self.selected_library

        model=gtk.ListStore(gobject.TYPE_STRING, gobject.TYPE_BOOLEAN)
        names=self.libraries.keys()
        names.sort()
        selected_iter=None

        for name in names:
            isdefault=(name == self.default_library)
            my_iter=model.append([name, isdefault])
            if name == selected:
                selected_iter = my_iter

        self.libraries_list.set_model(model)
        self.libraries_model=model

        if selected_iter <> None:
            self.libraries_list.get_selection().select_iter(selected_iter)

    # Similarly
    def update_directory_list(self):
        """Synchronizes the displayed list of directories to be
        consistent with the directories associated with
        self.selected_library."""

        selected=self.selected_library

        self.directories_model.clear()
        if selected == None:
            self.directories_label.set_markup('<b>No library selected</b>')
            self.add_button.set_sensitive(0)
        else:
            self.directories_label.set_markup('<b>Contents of %s</b>'%selected)
            for d in self.libraries[selected]:
                self.directories_model.append([d])
            self.add_button.set_sensitive(1)

    def library_changed(self, selection):
        """Handles the selection of a row in the 'library' list."""

        if self.filesel <> None:
            self.filesel.destroy()

        if selection.count_selected_rows() == 0:
            self.selected_library=None
            self.directories_model.clear()
            self.delete_button.set_sensitive(0)
            self.rename_button.set_sensitive(0)
            self.update_directory_list()
        else:
            model,paths=selection.get_selected_rows()
            assert(len(paths)==1)

            new_selection=model[paths[0]][0]

            if new_selection <> self.selected_library:
                self.selected_library=new_selection
                self.update_directory_list()
            self.delete_button.set_sensitive(1)
            self.rename_button.set_sensitive(1)

    def directory_changed(self, selection):
        """Handles the selection of a row in the 'directory' list."""

        if selection.count_selected_rows() == 0:
            self.remove_button.set_sensitive(0)
        else:
            self.remove_button.set_sensitive(1)

    def new_library(self, widget=None, name=None, start_editing=True, *args):
        """Handles clicks on the 'New Library' button by generating a
        new library.  Returns a TreeIter corresponding to the location
        of the new library in the model tree."""

        if name == None:
            name='New Library'
        x=2

        while self.libraries.has_key(name):
            name='New Library %d'%x
            x+=1

        # The first library you create is the default library (by default).
        isdefault=(len(self.libraries)==0)

        self.libraries[name]=sets.Set()
        if isdefault:
            self.default_library=name
        self.library_orig_names[name]=None

        iter=self.libraries_model.append([name, isdefault])

        self.libraries_list.get_selection().select_iter(iter)
        path=self.libraries_model.get_path(iter)
        self.libraries_list.set_cursor(path,
                                       self.libraries_list.get_column(0),
                                       start_editing)

        return path,iter

    def delete_library(self, *args):
        selection=self.libraries_list.get_selection()

        assert(self.selected_library <> None)
        assert(selection.count_selected_rows() == 1)

        model,paths=selection.get_selected_rows()

        curr_iter=model[paths[0]]

        assert(self.selected_library == curr_iter[0])

        if self.selected_library == self.default_library:
            self.default_library = None
        if self.library_new_names.has_key(self.selected_library):
            self.library_new_names[self.selected_library]=None
        del self.library_orig_names[self.selected_library]
        del self.libraries[self.selected_library]
        del curr_iter
        del model[paths[0]]

    def rename_library(self, *args):
        model,paths=self.libraries_list.get_selection().get_selected_rows()

        assert(len(paths)==1)

        self.libraries_list.grab_focus()
        self.libraries_list.set_cursor(paths[0],
                                       self.libraries_list.get_column(0),
                                       True)

    def handle_library_name_edited(self, cell, path, new_text):
        row=self.libraries_model[path]

        old_text=row[0]

        assert(self.libraries.has_key(old_text))

        if new_text == old_text:
           # Do nothing to avoid unnecessary recomputation.
            return

        if self.libraries.has_key(new_text):
            m=gtk.MessageDialog(type=gtk.MESSAGE_ERROR,
                                buttons=gtk.BUTTONS_OK,
                                message_format='The library "%s" already exists.'%new_text)
            m.connect('response', lambda *args:m.destroy())
            m.show()
            return

        self.libraries[new_text]=self.libraries[old_text]
        self.library_orig_names[new_text]=self.library_orig_names[old_text]
        orig_name=self.library_orig_names[new_text]
        if orig_name <> None:
            self.library_new_names[orig_name]=new_text

        if old_text == self.selected_library:
            self.selected_library=new_text
        if old_text == self.default_library:
            self.default_library=new_text

        del self.libraries[old_text]
        del self.library_orig_names[old_text]

        self.libraries_model[path]=(new_text,row[1])

        self.libraries_model.sort_column_changed()

    def handle_default_library_toggled(self, cell, path):
        row=self.libraries_model[path]

        old_value=row[1]

        if old_value:
            self.default_library=None
            self.libraries_model[path]=(row[0],0)
        else:
            if self.default_library <> None:
                # Just wipe out all default settings.  Inefficient but
                # safe, and should be OK if only a few libraries
                # exist.
                for tmprow in self.libraries_model:
                    tmprow[1]=0

            # Commit the new default.
            self.libraries_model[path]=(row[0], 1)
            self.default_library=row[0]

    def __do_add_directories(self, fns):
        assert(reduce(operator.__and__,
                      map(lambda x:os.path.isdir(x), fns),
                      True))

        dirs=self.libraries[self.selected_library]
        for d in fns:
            if d not in dirs:
                dirs.add(d)
                self.directories_model.append([d])

        self.directories_model.sort_column_changed()

    def add_directories(self, iter, dirs):
        """Add some directories to a library.  iter is the TreeIter of
        the library to be modified, in the library list TreeView."""

        self.libraries_list.get_selection().select_iter(iter)
        self.__do_add_directories(dirs)

    def handle_add_directory(self, *args):
        """Prompt the user and a new directory to a library."""

        assert(self.selected_library <> None)

        if self.filesel <> None:
            self.filesel.show()
            return

        def on_response(dialog, response_id):
            if response_id <> gtk.RESPONSE_OK:
                # Do nothing.
                self.filesel.destroy()
                return

            fns=self.filesel.get_filenames()
            self.filesel.destroy()

            self.__do_add_directories(fns)

        def on_destroy(*args):
            self.filesel=None

        self.filesel=gtk.FileChooserDialog(
            title='Choose a directory to add to %s'%self.selected_library,
            action=gtk.FILE_CHOOSER_ACTION_SELECT_FOLDER,
            buttons=(gtk.STOCK_OK, gtk.RESPONSE_OK,
                     gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL))
        self.filesel.connect('response', on_response)
        self.filesel.connect('destroy', on_destroy)
        self.filesel.set_select_multiple(True)
        self.filesel.show()

    def remove_directory(self, *args):
        selected=self.selected_library
        assert(selected)

        lib=self.libraries[selected]

        selection=self.directories_list.get_selection()

        model,paths=selection.get_selected_rows()

        for p in paths:
            d=model[p][0]

            if d not in lib:
                warn('Warning: trying to remove %s from library %s, but it\'s not there!'%(d, selected))

            self.libraries[selected].remove(d)

        # Future-proofing for the possibility of enabling multi-selections.
        #
        # We need to get all iterators first, because iterators don't
        # go bad when you delete them.
        iters=map(lambda x:self.directories_model.get_iter(x), paths)

        for i in iters:
            self.directories_model.remove(i)

        # Don't need to signal a sort update, since removing rows
        # never affects the sorted order.

active_library_editor=None
"""Internal use, don't touch."""

def show_library_editor(glade_location, name=None, dirs=None):
    """Create and display the dialog that edits library definitions.
    If a dialog is already open, just bring it to the front.
    glade_location is the location from which the glade file should be
    loaded.

    If 'name' is not None, a new library named 'name' (possibly
    mangled to make it unique) will be created.  If, in addition,
    'dirs' is not None, the new library will initially contain the
    directories specified in 'dirs'.

    This function prevents a situation where two editors are running
    at once and Bad Stuff happens."""

    global active_library_editor

    def zap_library_editor(*args):
        global active_library_editor
        active_library_editor=None

    if active_library_editor <> None:
        active_library_editor.gui.present()
    else:
        active_library_editor=LibraryEditor(glade_location)
        active_library_editor.gui.connect('destroy', zap_library_editor)

    if name <> None:
        p,i=active_library_editor.new_library(name=name, start_editing=False)

        if dirs <> None:
            active_library_editor.add_directories(i, dirs)
