 function PARtoNRRD_Philips_RelX(Astruct)

% ====================================
% HEADER INFORMATION
% ====================================
% Author:   Jonathan Farrell
%           F.M. Kirby Research Center for Functional Brain Imaging
%           Kennedy Krieger Institute
%           Johns Hopkins University School of Medicine
%           Baltimore, MD 21205
%           jonfarrell@jhu.edu
%
% Name:     PARtoNRRD_Philips_RelX.m
% Creation Date:  November 30, 2005
% Major Modification:  October 25, 2006

% ====================================
% UPDATE HISTORY
% =====================================
% November 2, 2006 | Also added the check for the Philips software release.  
%                   i,e. Astruct.release must be one of the known versions.  
%                   I also corrected the line where I check for 100,100,100
%                   in the.grad file I now check all three values
%
% November 3, 2006 | Added code to automatically detect the data order as
%                    slice, volume or other.  
%
% ====================================
% PUPROSE
% ====================================

% Create the NRRD headers for Philips DTI data.  This is done by 1) 
% extracting parametes from the PAR file and 2) computing the DTI gradient 
% table (.grad file) with DTI_gradient_table_creator.m (for Kirby center 
% scanners) or DTI_gradient_table_creator_Philips_RelX.m (for other 
% Philips scanners) 

% ====================================
% USAGE
% ====================================

% The <filename>.nhdr file can be used to upload the filename.rec data 
% files into Slicer.  Your .par file and .rec file must have the same base
% filename (i.e. bob.rec, bob.par).  The rec file will be 

% ===========================================
% USER SUPPLIED INFORMATION AND EXAMPLE INPUT
% ===========================================

%% Make NRRD files for all par files in a directory (uncomment below)
%jonpath = 'C:\Program Files\MATLAB71\work\test';
%A = dir(fullfile(jonpath,'*.par'));
%for ii=1:size(A,1)
%	parlist{ii} = char(fullfile(jonpath,A(ii).name));
%end

% The following paramters are neccesary and sufficent to determine the
% gradient table on the Philips MR system. Some of them can be populated 
% from the par file. However, the par file does not contain all the 
% information you need.  Also, the information the par file contains 
% depends on the par file version (depends on the Philips software used
% to create the par file).  You can find out the version of the par file
% by checking the top right hand corner.  This program will check if your
% inputs are complete and compatible.For a more detailed description of 
% the input parameters and their possible values, please read the 
% documentation in the DTI_gradient_table_creator_Philips_RelX.m code

%% if par file version is 3
%Astruct.type = 'dti';
%Astruct.didREG = 'n';
%Astruct.writeGRAD = 'y';
%Astruct.grad_choice = 'Jones30';
%Astruct.foldover = 'AP';
%Astruct.fat_shift = 'P';
%Astruct.patient_position = 'hf';
%Astruct.patient_orientation = 'sp';
%Astruct.release = 'Rel_10.x'; 
%Astruct.bvalue = '700';
%Astruct.par_file = 'jones30_v3.par';


% if par file version is 4, then some paramters will be parsed from the 
% par file. Astruct.patient_position, Astruct.patient_orientation and 
% Astruct.foldover are not needed as inputs. (uncomment the code below and
% it should create the .grad and .nhdr files for the
% Example-yes-ovp-high_v4.par file available on the website

%Astruct.type = 'dti';
%Astruct.didREG = 'n';
%Astruct.writeGRAD = 'y';
%Astruct.grad_choice = 'yes-ovp-high';
%Astruct.fat_shift = 'P';
%Astruct.release = 'Rel_10.x'; 
%Astruct.bvalue = '700';
%Astruct.par_file = 'Example-yes-ovp-high_v4.par';

% For User Defined Gradient Tables, Interleaved Philips PAR v4

%Astruct.par_file = 'C:\Program Files\MATLAB71\work\examples\interleaved-Philips\DTI_test_DTIwhole_4_1.PAR';
%Astruct.type = 'dti'
%Astruct.didREG = 'n';
%Astruct.writeGRAD = 'y';
%Astruct.grad_choice = 'user-defined';
%Astruct.fat_shift = 'P';
%Astruct.supplied_grad_file = 'C:\Program Files\MATLAB71\work\examples\interleaved-Philips\46me_for_SlicerandMagnet.txt';
%Astruct.bvalue = 1000;
%Astruct.release = 'Rel_1.5';

% check the input for completeness and consistency
% Any paramters in the par file will override the user supplied entries
DTIparams = int_DTIparamcheck_Philips_RelX(Astruct);

% ====================================
% CODE
% ====================================

% Get some information from the PAR file
par = int_loadPAR(Astruct.par_file);

[j1,j2,par_ext] = fileparts(Astruct.par_file);
clear j1;
clear j2;
if strcmp(par_ext,'.par')
    rec_ext = '.rec';
elseif strcmp(par_ext,'.PAR')
    rec_ext = '.REC';
end

% assign some variables
nrows  = par.scn.recon_res(1);
ncols  = par.scn.recon_res(2);
nslices = par.max.num_slices;
nvolumes = par.NumberOfVolumes;  
dims = [nrows,ncols,nslices,nvolumes];

switch(par.img(1).orient.orient)
    case 'TRA'
        if(par.scn.fov(1)~=par.scn.fov(3))
            warning('AXIAL (TRA): par.scn.fov(1)~=par.scn.fov(3). Setting to max');
            par.scn.fov = double(par.scn.fov);
            par.scn.fov([1 3])=max(par.scn.fov([1 3]));

        end
        if(par.scn.slicethk ~= (par.scn.fov(2)/par.max.num_slices))
            warning('Slice Thickness does not match fov/num_slices.ADJUSTING!!!');
            par.scn.slicethk = (par.scn.fov(2)/par.max.num_slices);
        end
    case 'COR'
        if(par.scn.fov(2)~=par.scn.fov(3))
            warning('CORONAL (COR): par.scn.fov(2)~=par.scn.fov(3). Setting to max');
            par.scn.fov = double(par.scn.fov);
            par.scn.fov([2 3])=max(par.scn.fov([2 3]));
        end
        if(par.scn.slicethk ~= (par.scn.fov(1)/par.max.num_slices))
            warning('Slice Thickness does not match fov/num_slices.ADJUSTING!!!');
            par.scn.slicethk = (par.scn.fov(1)/par.max.num_slices);
        end
    case 'SAG'
        if(par.scn.fov(2)~=par.scn.fov(1))
            warning('SAGITTAL (SAG): par.scn.fov(1)~=par.scn.fov(2). Settingto max');
            par.scn.fov = double(par.scn.fov);
            par.scn.fov([1 2])=max(par.scn.fov([1 2]));
        end
        if(par.scn.slicethk ~= (par.scn.fov(3)/par.max.num_slices))
            warning('Slice Thickness does not match fov/num_slices.ADJUSTING!!!');
            par.scn.slicethk = (par.scn.fov(3)/par.max.num_slices);
        end
end

if(par.scn.slicegap~=0)
    warning('Non-zero slice gap: adjusting slice thickness');
    % Read the header variables
    par.scn.slicethk = par.scn.slicethk+par.scn.slicegap;
    par.scn.slicegap=0;
end

% the FOV should have the in plane dimension first
switch(par.img(1).orient.orient)
    case 'TRA'
        FOV(1) = par.scn.fov(1);
        FOV(2) = par.scn.fov(3);
        FOV(3) = par.scn.fov(2);
    case 'COR'
        FOV(1) = par.scn.fov(2);
        FOV(2) = par.scn.fov(3);
        FOV(3) = par.scn.fov(1);
    case 'SAG'
        FOV(1) = par.scn.fov(1);
        FOV(2) = par.scn.fov(2);
        FOV(3) = par.scn.fov(3);
end

if strcmpi(Astruct.type,'dti')	
    
    [PATHSTR,content] = fileparts(Astruct.par_file);  
    
    % figure out the gradient table
    DTI_gradient_table_creator_Philips_RelX(DTIparams);
    grad_file = fullfile(PATHSTR,[content '.grad']);

    %Default key / value pairs
    %NNRD_version = 'NRRD0005'; (included in tag below)
    endian = 'little';

    %Key / value pairs from the Philips PAR file
    type = 'int16';
    dimension = '4';
    space = 'left-posterior-superior'; % chosen so space directions are all positive
    encoding = 'raw';
    space_units = '"mm" "mm" "mm"';
    space_origin = ['(' num2str((FOV(1)-FOV(1)/dims(1))/2) ', ' num2str((FOV(2)-FOV(2)/dims(2))/2) ', ' num2str((FOV(3)-FOV(3)/dims(3))/2) ')'];
    if isnumeric(Astruct.bvalue)
        DWMRI_b_value = num2str(Astruct.bvalue);
    elseif ischar(Astruct.bvalue)
        DWMRI_b_value = Astruct.bvalue;
    end
    
    % The NRRD header needs to accomodate both the volume ordered and slice
    % ordered PAR/REC files
    if strcmpi(par.inputVolumeSliceOrder, 'volume')
        sizes = [num2str(dims(1)) ' ' num2str(dims(2)) ' ' num2str(dims(3)) ' ' num2str(dims(4))];
        thicknesses = ['NaN NaN ' num2str(FOV(3)/dims(3)) ' NaN'];
        space_directions = ['(' num2str(FOV(1)/dims(1)) ',0,0)' ' (0,' num2str(FOV(2)/dims(2)) ',0)' ' (0,0,' num2str(FOV(3)/dims(3)) ')' ' none'];
        centerings = 'cell cell cell none';
        kinds = 'space space space list';
    elseif strcmpi(par.inputVolumeSliceOrder, 'slice')
        sizes = [num2str(dims(1)) ' ' num2str(dims(2)) ' ' num2str(dims(4)) ' ' num2str(dims(3))];
        thicknesses = ['NaN NaN NaN ' num2str(FOV(3)/dims(3))];
        space_directions = ['(' num2str(FOV(1)/dims(1)) ',0,0)' ' (0,' num2str(FOV(2)/dims(2)) ',0)' ' none' ' (0,0,' num2str(FOV(3)/dims(3)) ')'];
        centerings = 'cell cell none cell';
        kinds = 'space space list space';
    end
    
    % The grad file is set up for how DTIstudio or dtiproc interpret the data
    % you can cut and paste the .grad file into DTIstudio and don't need to flip any eigenvectors
    measurement_frame = '(1,0,0) (0,1,0) (0,0,1)';
    modality = 'DWMRI';

    DWMRI_version = '2';

    grad_file_contents = load(grad_file);
    for ii = 1:size(grad_file_contents)
        if length(num2str(grad_file_contents(ii,1))) == 1
            temp = ['000' num2str(grad_file_contents(ii,1))];
        elseif length(num2str(grad_file_contents(ii,1))) == 2
            temp = ['00' num2str(grad_file_contents(ii,1))];
        elseif length(num2str(grad_file_contents(ii,1))) == 3
            temp = ['0' num2str(grad_file_contents(ii,1))];
        elseif length(num2str(grad_file_contents(ii,1))) == 4
            temp = num2str(grad_file_contents(ii,1));
        end
        DWMRI_gradient{ii,1} = ['DWMRI_gradient_' temp];
        % Want to keep 4 precision after the decimal format
        DWMRI_gradient{ii,2} = sprintf('%3.4f',grad_file_contents(ii,2));
        DWMRI_gradient{ii,3} = sprintf('%3.4f',grad_file_contents(ii,3));
        DWMRI_gradient{ii,4} = sprintf('%3.4f',grad_file_contents(ii,4));
    end
  
    % Prepare a tag that references this program
    tag{1} = 'NRRD0005';
    tag{2} = '# NRRD header (.nhdr) created by PARtoNRRD_Philips_RelX.m';
    tag{3} = '# Author:  Jonathan Farrell, JonFarrell@jhu.edu';
    tag{4} = '# F.M. Kirby Research Center for Functional Brain Imaging';
    tag{5} = '# Kennedy Krieger Institute, Johns Hopkins Univesity School of Medicine';
    tag{6} = '# For info see:  http://godzilla.kennedykrieger.org/~jfarrell/software_web.html';
    tag{7} = '# Complete NRRD file format specification at:';
    tag{8} = '# http://teem.sourceforge.net/nrrd/format.html';
    
    % Prepare key / value pairs for output to text file
    %header{1} = NNRD_version; (above in tag)
    header{1} = ['content:' ' ' content rec_ext];
    header{2} = ['type:' ' ' type];
    header{3} = ['dimension:' ' ' dimension];
    header{4} = ['space:' ' ' space];
    header{5} = ['sizes:' ' ' sizes];
    header{6} = ['thicknesses:' ' ' thicknesses];
    header{7} = ['space directions:' ' ' space_directions];
    header{8} = ['centerings:' ' ' centerings];
    header{9} = ['kinds:' ' ' kinds];
    header{10} = ['endian:' ' ' endian];
    header{11} = ['encoding:' ' ' encoding];
    header{12} = ['space units:' ' ' space_units];
    header{13} = ['space origin:' ' ' space_origin];
    header{14} = ['data file:' ' ' content rec_ext];
    header{15} = ['measurement frame:' ' ' measurement_frame];
    header{16} = ['modality:=' modality];
    header{17} = ['DWMRI_b-value:=' DWMRI_b_value];
    header{18} = ['DWMRI_version:=' DWMRI_version];  

    % Append the diffusion encoding directions to the Output_field
    for ii = 1:size(DWMRI_gradient,1)
        if ~((grad_file_contents(ii,2) == 100) & (grad_file_contents(ii,3) == 100) & (grad_file_contents(ii,4) == 100))
                Output_fields{ii,1} = [DWMRI_gradient{ii,1} ':='];
                Output_fields{ii,2} = [DWMRI_gradient{ii,2} ' ' DWMRI_gradient{ii,3} ' ' DWMRI_gradient{ii,4}];
        elseif ((grad_file_contents(ii,2) == 100) & (grad_file_contents(ii,3) == 100) & (grad_file_contents(ii,4) == 100))
            Output_fields{ii,1} = [DWMRI_gradient{ii,1} ':='];
            Output_fields{ii,2} = 'n/a';
        end
    end

    % write to a file
    fid = fopen(fullfile(PATHSTR,[content '.nhdr']),'w','l');
    fprintf(fid,'%s\n',tag{:});
    fclose(fid);
    
    fid = fopen(fullfile(PATHSTR,[content '.nhdr']),'a','l');
    fprintf(fid,'%s\n',header{:});
    fclose(fid);
    
    T_Output_fields = Output_fields';
    fid = fopen(fullfile(PATHSTR,[content '.nhdr']),'a','l');
    fprintf(fid,'%s %s\n' ,T_Output_fields{:});
    fclose(fid);

    disp('.grad file created')
    disp('.nhdr file created')

elseif strcmpi(Astruct.type,'anatomical')
    error('not implemented yet');
end     

% ===================================
% INTERNAL FUNCTIONS (do not change)
% ===================================

function par = int_loadPAR(filename)
% function par = loadPAR(filename)
%
% Load a PAR header file (simplified Phillips format).
%
% Author:   Bennett Landman
% Created:  Jan 25 2005
% History:  Feb 01 2005 - Updated par heirarchy
%           Jun 16 2005 - Forced to retain file format
%           Jan 24 2006 - added version detection
%           Apr 27 2006 - bug fix TRA/SAG/COR

if(~exist(filename))
    filename = [filename 'v2'];
end
if(~exist(filename))
    error(['loadPAR: ' filename ' does not exist.']);
end

[par,vars,slicevars] = int_defaultPARVars;
fp = fopen(filename, 'rt');
imgTagFormat =[];
par.fmt = {}; % fileformat
par.scnfmt={}; % slice scan format
doneslices=0;
while(~feof(fp))
    line = fgetl(fp);
    if(length(line)<1)
        par.fmt{end+1} = {'comment',line};
        continue;
    end
    switch(line(1))
        case '#'        % comment
            par.fmt{end+1} = {'comment',line};
            if(strcmp('# === IMAGE INFORMATION DEFINITION =============================================',strtrim(line)))
                [imgTagFormat,par]= parseImgTagFormat(fp,slicevars,par);
            end
            blankImg = [];
            for j=1:length(imgTagFormat)
                blankImg.(imgTagFormat(j).mem).(imgTagFormat(j).var) = [];
            end

            continue;
        case '.'        % scan file variable
            [tag,key] = strtok(line,':');
            idx = strmatch(tag,vars(:,1));
            if(~isempty(idx))
                if(vars{idx,4})
                    par.(vars{idx,2}).(vars{idx,3}) = sscanf(strtrim(key(2:end)),'%f')';
                else
                    par.(vars{idx,2}).(vars{idx,3}) = strtrim(key(2:end));
                end
                par.fmt{end+1} = {'var',vars{idx,2},vars{idx,3}};
            else
                warning(['loadPAR: Unknown parameter declaration: ' line]);
                par.fmt{end+1} = {'comment',line};
            end
        otherwise       % parse as image slice information


            par = parseScanImgLine(par,line,imgTagFormat,blankImg);
            blankImg=par.img(1);
            %if(~strcmp(par.fmt{end}{1},'SLICES'))
            if(~doneslices)
                doneslices=1;
                par.fmt{end+1}={'SLICES'};
            end

            %end
    end

end
fclose(fp);
q = [par.img.info];
par.NumberOfVolumes = length(unique([q(:).dynamic_scan_num]));
NoV = length(q)/length(unique([q(:).slice_num]));
if(NoV~=par.NumberOfVolumes)
    warning('loadPAR.m: Dynamic Scan Number does not match number of slices. Assuming slices are ordered.');


    cnt = zeros([1 max([q(:).slice_num])]);

    for j=1:length(par.img)
        cnt(par.img(j).info.slice_num) = cnt(par.img(j).info.slice_num)+1;
        par.img(j).info.dynamic_scan_num=cnt(par.img(j).info.slice_num);
    end
end
q = [par.img.info];
par.NumberOfVolumes = length(unique([q(:).dynamic_scan_num]));
switch(par.img(1).orient.orient)
    case 'TRA'
        if(par.scn.fov(1)~=par.scn.fov(3))
            warning('AXIAL (TRA): par.scn.fov(1)~=par.scn.fov(3). Setting to max');
            par.scn.fov = double(par.scn.fov);
            par.scn.fov([1 3])=max(par.scn.fov([1 3]));

        end
        if(par.scn.slicethk ~= (par.scn.fov(2)/par.max.num_slices))
            warning('Slice Thickness does not match fov/num_slices. ADJUSTING!!!');
            par.scn.slicethk = (par.scn.fov(2)/par.max.num_slices);
        end
    case 'COR'
        if(par.scn.fov(2)~=par.scn.fov(3))
            warning('CORONAL (COR): par.scn.fov(2)~=par.scn.fov(3). Setting to max');
            par.scn.fov = double(par.scn.fov);
            par.scn.fov([2 3])=max(par.scn.fov([2 3]));
        end
        if(par.scn.slicethk ~= (par.scn.fov(1)/par.max.num_slices))
            warning('Slice Thickness does not match fov/num_slices. ADJUSTING!!!');
            par.scn.slicethk = (par.scn.fov(1)/par.max.num_slices);
        end
    case 'SAG'
        if(par.scn.fov(2)~=par.scn.fov(1))
            warning('SAGITTAL (SAG): par.scn.fov(1)~=par.scn.fov(2). Setting to max');
            par.scn.fov = double(par.scn.fov);
            par.scn.fov([1 2])=max(par.scn.fov([1 2]));
        end
        if(par.scn.slicethk ~= (par.scn.fov(3)/par.max.num_slices))
            warning('Slice Thickness does not match fov/num_slices. ADJUSTING!!!');
            par.scn.slicethk = (par.scn.fov(3)/par.max.num_slices);
        end
end

if(par.scn.slicegap~=0)
    warning('Non-zero slice gap: adjusting slice thickness');
    % Read the header variables
    par.scn.slicethk = par.scn.slicethk+par.scn.slicegap;
    par.scn.slicegap=0;
end


% Determine File Volume/Slice Order (inputVolumeSliceOrder): 
% a) volume - all slices are listed (in order) for each volume before the next volume
% b) slices - the same slice is listed for all volumes before the next
% slice of the first volume (volumes are ordered)
% c) other - some other ordering (any ordering of volumes/slices is
% supported in the PAR file format)

% Procedure: Build a matrix with each row having: [VOLUME SLICE IDX]
SRC = zeros([length(par.img) 3]);
N=length(par.img);
for j=1:N
    SRT(j,1:3) = [par.img(j).info.dynamic_scan_num par.img(j).info.slice_num j];
end
SRT = sortrows(SRT);
if(sum(SRT(:,3)==(1:N)')==N)
    % This is in volume order
    par.inputVolumeSliceOrder = 'volume';    
else
    SRT = sortrows(SRT(:,[2 1 3]));
    if(sum(SRT(:,3)==(1:N)')==N)
    % This is in slice order
        par.inputVolumeSliceOrder = 'slice';        
    else
        par.inputVolumeSliceOrder = 'unknown';
        warning('loadPAR: Slice ordering is not a predefined type.');
        disp('This toolbox is compatalbe with arbitrary ordering of slices in PAR/REC files.');
        disp('However, other toolboxes or REC readers may assume a specific ordering.');
    end

    
end





% Read
% #
% # === PIXEL VALUES =============================================================
% #  PV = pixel value in REC file, FP = floating point value, DV = displayed value on console
% #  RS = rescale slope,           RI = rescale intercept,    SS = scale slope
% #  DV = PV * RS + RI             FP = DV / (RS * SS)
% #
function par = parseScanImgLine(par,line,imgTagFormat,blankImg)
h = sscanf(line, '%f');
if(length(h)~=sum([imgTagFormat(:).len]))
    error('loadPAR: Slice tag format does not match the number of enteries');
end
img = blankImg;
k=0;
for j=1:length(imgTagFormat)
    img.(imgTagFormat(j).mem).(imgTagFormat(j).var) = h(k+(1:imgTagFormat(j).len));
    k = k+imgTagFormat(j).len;
end
img.orient.orient = 'UNK';
if(img.orient.slice_orientation==1)       img.orient.orient = 'TRA';     end
if(img.orient.slice_orientation==2)       img.orient.orient = 'SAG';     end
if(img.orient.slice_orientation==3)       img.orient.orient = 'COR';     end
if(~isfield(par.scn,'pix_bits'))
    par.scn.pix_bits = img.info.pix_bits;
else
    if(isfield(img.info,'pix_bits'))
        if(par.scn.pix_bits ~= img.info.pix_bits)
            warning('loadPAR - REC file contains image slices with variable bits per pixel.');
        end
    end
end
if(~isfield(par.scn,'recon_res'))
    par.scn.recon_res = img.info.recon_res(:)';
else
    if(isfield(img.info,'recon_res'))
        if(par.scn.recon_res(:) ~= img.info.recon_res(:))
            warning('loadPAR - REC file contains image slices with variable reconstruction sizes.');
        end
    end
end

if(~isfield(par.scn,'slicethk'))
    par.scn.slicethk = img.info.slicethk;
else
    if(isfield(img.info,'slicethk'))
        if(par.scn.slicethk ~= img.info.slicethk)
            warning('loadPAR - REC file contains image slices with variable slice thickness.');
        end
    end
end


if(~isfield(par.scn,'slicegap'))
    par.scn.slicegap = img.info.slicegap;
else
    if(isfield(img.info,'slicegap'))
        if(par.scn.slicegap ~= img.info.slicegap)
            warning('loadPAR - REC file contains image slices with variable slice thickness.');
        end
    end
end

if(isempty(par.img))
    par.img = img;
else
    par.img(end+1) = img;
end

Vs = strfind( par.fmt{8}{2},'V');
try
    par.version = str2num(par.fmt{8}{2}((Vs+1):end));
catch
    par.version = -1;
end

function [fmt,par] = parseImgTagFormat(fp,vars,par)
fmt = [];
%fgetl(fp); % eat the comment line
line = strtrim(fgetl(fp));
par.fmt{end+1} = {'comment',line};
ok_comment = '#  The rest of this file contains ONE line per image, this line contains the following information:';
while(~strcmp(line,'# === IMAGE INFORMATION =========================================================='))
    if(length(line)>1)
        idx = strmatch(line,vars(:,1));
        if(isempty(idx))
            if(~strcmp(line,ok_comment))
                warning(['loadPAR: Cannot interpret slice variable: ' line]);
            end
        else
            fmt(end+1).len = vars{idx,2};
            fmt(end).mem=vars{idx,3};
            fmt(end).var=vars{idx,4};
            par.scnfmt{end+1} = {line,vars{idx,3},vars{idx,4}};
        end
    end
    line = strtrim(fgetl(fp));
    par.fmt{end+1} = {'comment',line};
end




function [emptypar, PARvars,PARSliceVars] = int_defaultPARVars()
emptypar = struct('info',[],'scn',[],'max',[],'orient',[],'special',[],'cardiac',[],'diffusion',[],'img',[]);
PARvars = {...
    ... % Variables from Prerelease 1. Feb 1, 2005
    '.    Patient name                       ','info','patient_name',0;
    '.    Examination name                   ','scn','exam_name',0;
    '.    Protocol name                      ','scn','protocol_name',0;
    '.    Examination date/time              ','info','exam_datetime',0;
    '.    Acquisition nr                     ','scn', 'acquisitin_num',1;
    '.    Reconstruction nr                  ','scn','recon_num',1;
    '.    Scan Duration [sec]                ','scn','scan_dur',1;
    '.    Max. number of cardiac phases      ','max','card_phs',1;
    '.    Max. number of echoes              ','max','num_echo',1;
    '.    Max. number of slices/locations    ','max','num_slices',1;
    '.    Max. number of dynamics            ','max','num_dynamics',1;
    '.    Max. number of mixes               ','max','num_mixes',1;
    '.    Image pixel size [8 or 16 bits]    ','scn','pix_bits',1;
    '.    Technique                          ','scn','technique',0;
    '.    Scan mode                          ','scn','scan_mode',0;
    '.    Scan resolution  (x, y)            ','scn','scan_res',1;
    '.    Scan percentage                    ','scn','scan_pct',1;
    '.    Recon resolution (x, y)            ','scn','recon_res',1;
    '.    Number of averages                 ','scn','NEX',1;
    '.    Repetition time [msec]             ','scn','rep_time',1;
    '.    FOV (ap,fh,rl) [mm]                ','scn','fov',1;
    '.    Slice thickness [mm]               ','scn','slicethk',1;
    '.    Slice gap [mm]                     ','scn','slicegap',1;
    '.    Water Fat shift [pixels]           ','scn','water_fat_shift',1;
    '.    Angulation midslice(ap,fh,rl)[degr]','orient','ang_midslice',1;
    '.    Off Centre midslice(ap,fh,rl) [mm] ','orient','off_ctr_midslice',1;
    '.    Flow compensation <0=no 1=yes> ?   ','special','flow_comp',1;
    '.    Presaturation     <0=no 1=yes> ?   ','special','presatuaration',1;
    '.    Cardiac frequency                  ','cardiac','cardiac_freq',1;
    '.    Min. RR interval                   ','cardiac','min_rr_int',1;
    '.    Max. RR interval                   ','cardiac','max_rr_int',1;
    '.    Phase encoding velocity [cm/sec]   ','cardiac','phase_enc_vel',1;
    '.    MTC               <0=no 1=yes> ?   ','special','mtc',1;
    '.    SPIR              <0=no 1=yes> ?   ','special','spir',1;
    '.    EPI factor        <0,1=no EPI>     ','special','epi_factor',1;
    '.    TURBO factor      <0=no turbo>     ','special','turbo_factor',1;
    '.    Dynamic scan      <0=no 1=yes> ?   ','special','dynamic_scan',1;
    '.    Diffusion         <0=no 1=yes> ?   ','diffusion','diffusion',1;
    '.    Diffusion echo time [msec]         ','diffusion','diffusion_echo',1;
    '.    Inversion delay [msec]             ','special','inversion_delay',1;
    ... % Variables for May 31, 2005
    '.    Series Type                        ','scn','series_type',0;
    '.    Patient position                   ','orient','patient_pos',0;
    '.    Preparation direction              ','orient','prep_dir',0;
    '.    Repetition time [ms]               ','scn','rep_time',1;
    '.    Diffusion echo time [ms]           ','special','diffusion_echo_time',1;
    };

PARSliceVars  = {...
    '#  slice number                             (integer)',1,'info','slice_num';
    '#  echo number                              (integer)',1,'info','echo_num';
    '#  dynamic scan number                      (integer)',1,'info','dynamic_scan_num';
    '#  cardiac phase number                     (integer)',1,'info','cardiac_phase_num';
    '#  image_type_mr                            (integer)',1,'info','image_type_mr';
    '#  scanning sequence                        (integer)',1,'info','scan_seq';
    '#  index in REC file (in images)            (integer)',1,'info','idx_rec';
    '#  image pixel size (in bits)               (integer)',1,'info','pix_bits';
    '#  scan percentage                          (integer)',1,'info','scanpct';
    '#  recon resolution (x y)                   (2*integer)',2,'info','recon_res';
    '#  rescale intercept                        (float)',1,'vis','rescale_intercept';
    '#  rescale slope                            (float)',1,'vis','rescale_slope';
    '#  scale slope                              (float)',1,'vis','scale_slope';
    '#  window center                            (integer)',1,'notused','window_center';
    '#  window width                             (integer)',1,'notused','window_width';
    '#  image angulation (ap,fh,rl in degrees )  (3*float)',3,'orient','img_angulation';
    '#  image offcentre (ap,fh,rl in mm )        (3*float)',3,'orient','img_offcentre';
    '#  slice thickness (in mm )                 (float)',1,'info','slicethk';
    '#  slice gap (in mm )                       (float)',1,'info','slicegap';
    '#  image_display_orientation                (integer)',1,'orient','display_orientation';
    '#  slice orientation ( TRA/SAG/COR )        (integer)',1,'orient','slice_orientation';
    '#  fmri_status_indication                   (integer)',1,'special','fmri_status_indication';
    '#  image_type_ed_es  (end diast/end syst)   (integer)',1,'special','image_type_ed_es';
    '#  pixel spacing (x,y) (in mm)              (2*float)',2,'special','pix_spacing';
    '#  echo_time                                (float)',1,'special','echo_time';
    '#  dyn_scan_begin_time                      (float)',1,'special','dyn_scan_begin_time';
    '#  trigger_time                             (float)',1,'special','trigger_time';
    '#  diffusion_b_factor                       (float)',1,'special','diffusion_b_factor';
    '#  number of averages                       (integer)',1,'special','NEX';
    '#  image_flip_angle (in degrees)            (float)',1,'special','image_flip_angle';
    '#  cardiac frequency   (bpm)                (integer)',1,'special','cardiac_freq'
    '#  minimum RR-interval (in ms)              (integer)',1,'special','minRR';
    '#  maximum RR-interval (in ms)              (integer)',1,'special','maxRR';
    '#  TURBO factor  <0=no turbo>               (integer)',1,'special','turbo';
    '#  Inversion delay (in ms)                  (float)',1,'special','inversion_delay';
    };

% ====================================
% ====================================
% START OF INPUT VALIDATION
% ====================================
% ====================================
function DTIparams = int_DTIparamcheck_Philips_RelX(Astruct)

DTIparams = Astruct;
    
q= {'par_file','didREG','writeGRAD','grad_choice','fat_shift','release'};
for j=1:length(q)
    checklist(j) = isfield(Astruct,q{j});
end
if find(checklist == 0)
    disp('You failed to provide the REQUIRED inputs')
    error('One or more of par_file, didREG, writeGRAD, grad_choice,fat_shift, or release are not defined')
end

if exist(Astruct.par_file) ~= 2
    disp('You need to enter a valid par file')
    error(['Sorry...The par file : ' Astruct.par_file ' you specified does not exist'])
elseif exist(Astruct.par_file) == 2
    DTIparms.par_file = Astruct.par_file;
end

[jonpath,j2,par_ext] = fileparts(Astruct.par_file);
clear j2;
% Get some information from the PAR file
par = int_loadPAR(Astruct.par_file);
    
% assign some variables
nrows  = par.scn.recon_res(1);
ncols  = par.scn.recon_res(2);
nslices = par.max.num_slices;
nvolumes = par.NumberOfVolumes;
dims = [nrows,ncols,nslices,nvolumes];
par_version = par.version;

% ======================================
% Special V3 Par file version validation step
% =======================================
if par_version > 3
    
else % check for required inputs
    checklist = isfield(Astruct,{'foldover','patient_orientation','patient_position'});
    if find(checklist == 0)
        disp('Since your par file is older than V4, (it is most likely a V3), you must enter')
        disp('inputs for the foldover, patient_orientation and patient_position in Astruct')
        error('Foldover, patient_orientation and patient_position are not defined')
    end
end

% =====================================
% Continue with input validation steps
% =====================================
if (strcmpi(Astruct.didREG,'y') | strcmpi(Astruct.didREG,'n'))
    DTIparams.didREG = Astruct.didREG;
else
    disp('Did you perform coregistration of your DTI data?')
    error('Please enter the didREG option as y or n')
end

if (strcmpi(Astruct.writeGRAD,'y') | strcmpi(Astruct.writeGRAD,'n'))
    DTIparams.writeGRAD = Astruct.writeGRAD;
else
    disp('Do you want to write out the output of this program as a .grad file?')
    disp('The .grad file will contain the gradient table and is in a format ')
    disp('that is used in DTIstudio and dtiproc based analysis')
    error('Please enter the didREG option as y or n')
end 

if (strcmpi(Astruct.grad_choice,'yes-ovp-high') | strcmpi(Astruct.grad_choice,'yes-ovp-medium') | strcmpi(Astruct.grad_choice,'yes-ovp-low')...
    |strcmpi(Astruct.grad_choice,'no-ovp-high') | strcmpi(Astruct.grad_choice,'no-ovp-medium') | strcmpi(Astruct.grad_choice,'no-ovp-low')...
    | strcmpi(Astruct.grad_choice,'user-defined'))
    grad_choice = Astruct.grad_choice;
else
    error('Please enter: yes-ovp-high,yes-ovp-medium,yes-ovp-low,no-ovp-high,no-ovp-medium,no-ovp-low,or user-defined for first input ')
end  

if strcmpi(Astruct.grad_choice,'user-defined')
    if isfield(Astruct,'supplied_grad_file')
        DTIparams.supplied_grad_file = Astruct.supplied_grad_file;
        if exist(Astruct.supplied_grad_file) ~= 2
            disp('You need to enter a valid gradient table file')
            error(['Sorry...The gradient table file : ' Astruct.supplied_grad_file ' you specified does not exist'])
        end
    else
        error('Use of the user-defined option requires that you provide a file with the directions you entered at the scanner')
    end
else
    DTIparams.supplied_grad_file = [];
end

if (strcmpi(Astruct.fat_shift,'A') | strcmpi(Astruct.fat_shift,'P') | strcmpi(Astruct.fat_shift,'R')...
    |strcmpi(Astruct.fat_shift,'L') | strcmpi(Astruct.fat_shift,'F') | strcmpi(Astruct.fat_shift,'H'))
    DTIparams.fat_shift = Astruct.fat_shift;
else
    error('Please enter the fat shift direction as A, P, R, L, F,or H ')
end

if (strcmpi(Astruct.release,'Rel_10.x') | strcmpi(Astruct.release,'Rel_11.x') | strcmpi(Astruct.release,'Rel_1.2')...
    | strcmpi(Astruct.release,'Rel_1.5') | strcmpi(Astruct.release,'Rel_1.7'))
    DTIparams.release = Astruct.release;
else
    error('Please enter the Philips software release for your scanner, Choose from :  Rel_10.x, Rel_11.x, Rel_1.2, Rel_1.5 or Rel_1.7 ')
end


% If not a field , default to the par path
if (~isfield(Astruct,'xfmpath'))
    DTIparams.xfmpath = jonpath;
% If empty, defualt to the par path
elseif isempty(Astruct.xfmpath)
    DTIparams.xfmpath = jonpath;
elseif (isfield(Astruct,'xfmpath') & ~isempty(Astruct.xfmpath))
    if(exist(Astruct.xfmpath) == 7)
        DTIparams.xfmpath = Astruct.xfmpath;
    else
        error(['the xfmpath you provided: ' Astruct.xfmpath ' is not a valid directory'])
    end
end

% ============================================
% Gather inputs from V4 and more recent par files
% ============================================
if par_version > 3
    
    % get the patient position
    position_flag = par.orient.patient_pos;
    if findstr(position_flag,'Head')
        DTIparams.patient_position = 'hf';
    elseif findstr(position_flag,'Feat')
        DTIparams.patient_position = 'ff';
    end
    
    % get the patient_orientation
    if findstr(position_flag,'Supine')
        DTIparams.patient_orientation = 'sp';
    elseif findstr(position_flag,'Prone')
        DTIparams.patient_orientation = 'pr';
    elseif findstr(position_flag,'Right')
        DTIparams.patient_orientation = 'rd';
    elseif findstr(position_flag,'Left') 
        DTIparams.patient_orientation = 'ld';
    end
    
    % get the slice orientation
    DTIparams.slice_orientation = par.img(1).orient.orient; 
        
    % get the foldover direction
    if findstr(par.orient.prep_dir,'Anterior-Posterior')
        Astruct.foldover = 'AP';
        DTIparams.foldover = 'AP';
    elseif findstr(par.orient.prep_dir,'Right-Left')
        Astruct.foldover = 'RL';
        DTIparams.foldover = 'RL';
    elseif findstr(par.orient.prep_dir,'Feet-Head')
        Astruct.foldover = 'FH';
        DTIparams.foldover = 'FH';
    end     
else % If older than a V4 (i.e., V3) 
    % get the foldover direction
    if (strcmpi(Astruct.foldover,'AP') | strcmpi(Astruct.foldover,'RL') | strcmpi(Astruct.foldover,'FH'))
        DTIparams.foldover = Astruct.foldover;
    else
        error('Please enter the foldover direction as AP, RL, or FH')
    end

    % get the patient orientation
    if (strcmpi(Astruct.patient_orientation,'sp') | strcmpi(Astruct.patient_orientation,'pr') | strcmpi(Astruct.patient_orientation,'rd')...
        |strcmpi(Astruct.patient_orientation,'ld'))
        DTIparams.patient_orientation = Astruct.patient_orientation;    
    else
        error('Please enter the patient orientation as sp,pr,rd or ld')
    end

    % get the patient position
    if (strcmpi(Astruct.patient_position,'hf') | strcmpi(Astruct.patient_position,'ff'))
        DTIparams.patient_position = Astruct.patient_position;
    else
        error('Please enter the patient position as hf or ff')
    end  
    
    % get the slice orientation for V3 par files
    DTIparams.slice_orientation = par.img(1).orient.orient;    
end

% ======================================
% Check that foldover and fat_shift agree
% =========================================
if (strcmpi(Astruct.foldover,'AP'))
    if ~(strcmpi(Astruct.fat_shift,'A') | strcmpi(Astruct.fat_shift,'P'))
        error(['The fat shift ' Astruct.fat_shift ' and foldover ' Astruct.foldover ' you provided are not compatible'])
    end
elseif (strcmpi(Astruct.foldover,'RL'))
    if ~(strcmpi(Astruct.fat_shift,'R') | strcmpi(Astruct.fat_shift,'L'))
        error(['The fat shift ' Astruct.fat_shift ' and foldover ' Astruct.foldover ' you provided are not compatible'])
    end
elseif (strcmpi(Astruct.foldover,'FH'))
    if ~(strcmpi(Astruct.fat_shift,'F') | strcmpi(Astruct.fat_shift,'H'))
        error(['The fat shift ' Astruct.fat_shift ' and foldover ' Astruct.foldover ' you provided are not compatible'])
    end
end

DTIparams.angulation = par.orient.ang_midslice;
DTIparams.scan_date = strtok(par.info.exam_datetime,' /');

if strcmpi(Astruct.grad_choice,'yes-ovp-low')
    if nvolumes == 8 % assumes 6 directions + 1 b0 + 1 mean DWI
    else
        error(['Your number of volumes, ' num2str(nvolumes) ' and grad_choice of ' Astruct.grad_choice ' are in conflict'])       
    end
elseif strcmpi(Astruct.grad_choice,'yes-ovp-medium')
    if nvolumes == 17 % assumes 15 directions + 1 b0 + 1 mean DWI
    else
       error(['Your number of volumes, ' num2str(nvolumes) ' and grad_choice of ' Astruct.grad_choice ' are in conflict'])
    end
elseif strcmpi(Astruct.grad_choice,'yes-ovp-high')
    if nvolumes == 34 % assumes 32 directions + 1 b0 + 1 mean DWI
    else
        error(['Your number of volumes, ' num2str(nvolumes) ' and grad_choice of ' Astruct.grad_choice ' are in conflict'])
    end   
elseif strcmpi(Astruct.grad_choice,'no-ovp-low')
    if nvolumes == 8 % assumes 6 directions + 1 b0 + 1 mean DWI
    else
        error(['Your number of volumes, ' num2str(nvolumes) ' and grad_choice of ' Astruct.grad_choice ' are in conflict'])
    end
elseif strcmpi(Astruct.grad_choice,'no-ovp-medium')
    if nvolumes == 17 % assumes 15 directions + 1 b0 + 1 mean DWI
    else
        error(['Your number of volumes, ' num2str(nvolumes) ' and grad_choice of ' Astruct.grad_choice ' are in conflict'])
    end
elseif strcmpi(Astruct.grad_choice,'no-ovp-high')
    if nvolumes == 34 % assumes 32 directions + 1 b0 + 1 mean DWI        
    else
        error(['Your number of volumes, ' num2str(nvolumes) ' and grad_choice of ' Astruct.grad_choice ' are in conflict'])
    end
end



% =========================================
% ==========================================
% END INPUT PARAMETER VALIDATION
% =========================================
% =========================================
