#
# $Id$ 
#

"""This module will read satellit data files in OpenMTP format (eg. Meteosat-7 prolog file). Format described in:
'The Meteosat Archive; Format Guide No. 1; Basic Imagery: OpenMTP Format'; EUM FG 1; Rev 2.1; April 2000
"""

import sys
from datetime import timedelta
from StringIO import StringIO
import numpy as np

from mipp import CalibrationError
from mipp import strptime
from mipp.xrit import _xrit
from mipp.xrit import Metadata
from mipp.xrit import bin_reader as rbin

__all__ = ['read_metadata']

ASCII_HEADER_LEN = 1345
BINARY_HEADER_LEN = 144515
BINARY_HEADER_LEN_VISCOMP = 192999

def _read_ascii_header(fp):    
    fp = StringIO(fp.read(ASCII_HEADER_LEN)) # Don't mix iteration and read method.
    hdr = dict()
    for line in fp:
        k = line[:14].strip()
        v = line[15:].strip()
        hdr[k] = v
    return hdr

def _read_binary_header(fp, product_type):
    hdr = dict()
    hdr['fname'] =  fp.read(8)
    hdr['year'] = rbin.read_int4(fp.read(4))
    hdr['jday'] = rbin.read_int4(fp.read(4))
    hdr['slot'] = rbin.read_int4(fp.read(4))
    hdr['dtype'] = rbin.read_int4(fp.read(4))
    hdr['date'] = rbin.read_int4(fp.read(4))
    hdr['time'] = rbin.read_int4(fp.read(4))
    hdr['pltfrm'] = fp.read(2)
    fp.read(2) # spares
    hdr['proc'] = rbin.read_int4(fp.read(4))
    hdr['chan'] = rbin.read_int4(fp.read(4))
    calco_str = fp.read(5)
    if calco_str == '\0\0\0\0\0':
        hdr['calco'] =  None
    else:
        hdr['calco'] =  float(calco_str) / 100000.0
    space_str = fp.read(3)
    if space_str == '\0\0\0':
        hdr['space'] = 0.0
    else:
        hdr['space'] = float(space_str) / 10.0
    hdr['caltim'] = fp.read(5)
    fp.read(3) # spares
    hdr['rec2siz'] = rbin.read_int4(fp.read(4))
    hdr['lrecsiz'] = rbin.read_int4(fp.read(4))
    hdr['loffset'] = rbin.read_int4(fp.read(4))
    hdr['rtmet'] = fp.read(15)
    hdr['dmmod'] = rbin.read_int4(fp.read(4))
    hdr['rsmod'] = rbin.read_int4(fp.read(4))
    hdr['ssp'] = rbin.read_float4(fp.read(4))
    fp.read(12)
    fp.read(4)
    fp.read(8)
    hdr['line1'] = rbin.read_int4(fp.read(4))
    hdr['pixel1'] = rbin.read_int4(fp.read(4))
    hdr['nlines'] = rbin.read_int4(fp.read(4))
    hdr['npixels'] = rbin.read_int4(fp.read(4))
    fp.read(16)
    #hdr['mlt1'] = fp.read(2500)
    #hdr['mlt2'] = fp.read(2500)
    hdr['imgqua'] = rbin.read_int4(fp.read(4))
    fp.read(16)
    fp.read(2636) # only present for unrectified images
    hdr['ndgrp'] = rbin.read_int4(fp.read(4))
    hdr['dmstrt'] = rbin.read_int4(fp.read(4))
    hdr['dmend'] = rbin.read_int4(fp.read(4))
    hdr['dmstep'] = rbin.read_int4(fp.read(4))
    fp.read(8*105*105)
    hdr['ncor'] = rbin.read_int4(fp.read(4))
    hdr['chid1'] = rbin.read_int4(fp.read(4))
    fp.read(16*3030)
    if product_type == 'PVISBAN':
        hdr['chid2'] = rbin.read_int4(fp.read(4))
        fp.read(16*3030)
    return hdr

temp2rad = {}

# channel IR1
temp2rad[4] = np.array([[170.0, 0.667], [171.0, 0.697], [172.0, 0.727], [173.0,
0.758], [174.0, 0.789], [175.0, 0.822], [176.0, 0.856], [177.0, 0.891], [178.0,
0.927], [179.0, 0.964], [180.0, 1.002], [181.0, 1.040], [182.0, 1.080], [183.0,
1.122], [184.0, 1.164], [185.0, 1.207], [186.0, 1.251], [187.0, 1.297], [188.0,
1.344], [189.0, 1.392], [190.0, 1.441], [191.0, 1.491], [192.0, 1.542], [193.0,
1.595], [194.0, 1.649], [195.0, 1.704], [196.0, 1.761], [197.0, 1.818], [198.0,
1.877], [199.0, 1.938], [200.0, 1.999], [201.0, 2.062], [202.0, 2.127], [203.0,
2.192], [204.0, 2.259], [205.0, 2.328], [206.0, 2.397], [207.0, 2.468], [208.0,
2.541], [209.0, 2.615], [210.0, 2.690], [211.0, 2.767], [212.0, 2.846], [213.0,
2.925], [214.0, 3.007], [215.0, 3.089], [216.0, 3.174], [217.0, 3.259], [218.0,
3.347], [219.0, 3.435], [220.0, 3.526], [221.0, 3.617], [222.0, 3.711], [223.0,
3.806], [224.0, 3.902], [225.0, 4.000], [226.0, 4.100], [227.0, 4.201], [228.0,
4.304], [229.0, 4.408], [230.0, 4.514], [231.0, 4.622], [232.0, 4.731], [233.0,
4.842], [234.0, 4.955], [235.0, 5.069], [236.0, 5.185], [237.0, 5.302], [238.0,
5.422], [239.0, 5.542], [240.0, 5.665], [241.0, 5.789], [242.0, 5.915], [243.0,
6.043], [244.0, 6.172], [245.0, 6.303], [246.0, 6.436], [247.0, 6.570], [248.0,
6.706], [249.0, 6.844], [250.0, 6.983], [251.0, 7.125], [252.0, 7.268], [253.0,
7.412], [254.0, 7.559], [255.0, 7.707], [256.0, 7.857], [257.0, 8.009], [258.0,
8.162], [259.0, 8.317], [260.0, 8.474], [261.0, 8.633], [262.0, 8.793], [263.0,
8.955], [264.0, 9.119], [265.0, 9.285], [266.0, 9.453], [267.0, 9.622], [268.0,
9.793], [269.0, 9.966], [270.0, 10.141], [271.0, 10.317], [272.0, 10.495],
[273.0, 10.675], [274.0, 10.857], [275.0, 11.040], [276.0, 11.225], [277.0,
11.412], [278.0, 11.601], [279.0, 11.792], [280.0, 11.984], [281.0, 12.178],
[282.0, 12.374], [283.0, 12.572], [284.0, 12.772], [285.0, 12.973], [286.0,
13.176], [287.0, 13.381], [288.0, 13.587], [289.0, 13.796], [290.0, 14.006],
[291.0, 14.218], [292.0, 14.432], [293.0, 14.647], [294.0, 14.864], [295.0,
15.083], [296.0, 15.304], [297.0, 15.527], [298.0, 15.751], [299.0, 15.977],
[300.0, 16.205], [301.0, 16.435], [302.0, 16.666], [303.0, 16.899], [304.0,
17.134], [305.0, 17.371], [306.0, 17.609], [307.0, 17.849], [308.0, 18.091],
[309.0, 18.335]])

# channel IR2

temp2rad[5] = np.array([ [170.0, 0.664], [171.0, 0.693], [172.0, 0.723],
[173.0, 0.753], [174.0, 0.785], [175.0, 0.818], [176.0, 0.851], [177.0, 0.886],
[178.0, 0.922], [179.0, 0.958], [180.0, 0.996], [181.0, 1.035], [182.0, 1.075],
[183.0, 1.115], [184.0, 1.157], [185.0, 1.200], [186.0, 1.245], [187.0, 1.290],
[188.0, 1.336], [189.0, 1.384], [190.0, 1.433], [191.0, 1.483], [192.0, 1.534],
[193.0, 1.586], [194.0, 1.640], [195.0, 1.695], [196.0, 1.751], [197.0, 1.808],
[198.0, 1.867], [199.0, 1.927], [200.0, 1.988], [201.0, 2.051], [202.0, 2.115],
[203.0, 2.180], [204.0, 2.247], [205.0, 2.315], [206.0, 2.384], [207.0, 2.455],
[208.0, 2.527], [209.0, 2.601], [210.0, 2.676], [211.0, 2.752], [212.0, 2.830],
[213.0, 2.909], [214.0, 2.990], [215.0, 3.072], [216.0, 3.156], [217.0, 3.241],
[218.0, 3.328], [219.0, 3.416], [220.0, 3.506], [221.0, 3.598], [222.0, 3.690],
[223.0, 3.785], [224.0, 3.881], [225.0, 3.978], [226.0, 4.077], [227.0, 4.178],
[228.0, 4.280], [229.0, 4.384], [230.0, 4.490], [231.0, 4.597], [232.0, 4.705],
[233.0, 4.816], [234.0, 4.928], [235.0, 5.041], [236.0, 5.156], [237.0, 5.273],
[238.0, 5.392], [239.0, 5.512], [240.0, 5.634], [241.0, 5.757], [242.0, 5.882],
[243.0, 6.009], [244.0, 6.138], [245.0, 6.268], [246.0, 6.400], [247.0, 6.534],
[248.0, 6.669], [249.0, 6.806], [250.0, 6.945], [251.0, 7.085], [252.0, 7.227],
[253.0, 7.371], [254.0, 7.517], [255.0, 7.664], [256.0, 7.813], [257.0, 7.964],
[258.0, 8.117], [259.0, 8.271], [260.0, 8.427], [261.0, 8.585], [262.0, 8.745],
[263.0, 8.906], [264.0, 9.069], [265.0, 9.234], [266.0, 9.400], [267.0, 9.569],
[268.0, 9.739], [269.0, 9.911], [270.0, 10.084], [271.0, 10.260], [272.0,
10.437], [273.0, 10.616], [274.0, 10.796], [275.0, 10.979], [276.0, 11.163],
[277.0, 11.349], [278.0, 11.537], [279.0, 11.726], [280.0, 11.918], [281.0,
12.111], [282.0, 12.306], [283.0, 12.502], [284.0, 12.701], [285.0, 12.901],
[286.0, 13.103], [287.0, 13.306], [288.0, 13.512], [289.0, 13.719], [290.0,
13.928], [291.0, 14.139], [292.0, 14.351], [293.0, 14.566], [294.0, 14.782],
[295.0, 14.999], [296.0, 15.219], [297.0, 15.440], [298.0, 15.663], [299.0,
15.888], [300.0, 16.115], [301.0, 16.343], [302.0, 16.573], [303.0, 16.805],
[304.0, 17.039], [305.0, 17.274], [306.0, 17.511], [307.0, 17.750], [308.0,
17.991], [309.0, 18.233], [310.0, 18.477], [311.0, 18.723], [312.0, 18.970],
[313.0, 19.219], [314.0, 19.470], [315.0, 19.723], [316.0, 19.977], [317.0,
20.233], [318.0, 20.491], [319.0, 20.751], [320.0, 21.012], [321.0, 21.275],
[322.0, 21.539], [323.0, 21.806], [324.0, 22.074], [325.0, 22.343], [326.0,
22.615], [327.0, 22.888], [328.0, 23.163], [329.0, 23.439], [330.0, 23.717],
[331.0, 23.997], [332.0, 24.279], [333.0, 24.562], [334.0, 24.846], [335.0,
25.133], [336.0, 25.421], [337.0, 25.711], [338.0, 26.002], [339.0, 26.295],
[340.0, 26.590], [341.0, 26.886], [342.0, 27.184], [343.0, 27.484], [344.0,
27.785], [345.0, 28.088], [346.0, 28.392], [347.0, 28.698], [348.0, 29.006],
[349.0, 29.315], [350.0, 29.626], [351.0, 29.938], [352.0, 30.252], [353.0,
30.568], [354.0, 30.885], [355.0, 31.204], [356.0, 31.524], [357.0, 31.846],
[358.0, 32.170], [359.0, 32.495], [360.0, 32.822], [361.0, 33.150], [362.0,
33.480], [363.0, 33.811], [364.0, 34.144], [365.0, 34.478], [366.0, 34.814],
[367.0, 35.151], [368.0, 35.490], [369.0, 35.831]])


# channel WV1

temp2rad[6] = np.array([ [170.0, 0.021], [171.0, 0.023], [172.0, 0.024],
[173.0, 0.026], [174.0, 0.028], [175.0, 0.030], [176.0, 0.033], [177.0, 0.035],
[178.0, 0.038], [179.0, 0.040], [180.0, 0.043], [181.0, 0.046], [182.0, 0.049],
[183.0, 0.053], [184.0, 0.056], [185.0, 0.060], [186.0, 0.064], [187.0, 0.068],
[188.0, 0.073], [189.0, 0.077], [190.0, 0.082], [191.0, 0.087], [192.0, 0.093],
[193.0, 0.099], [194.0, 0.105], [195.0, 0.111], [196.0, 0.118], [197.0, 0.125],
[198.0, 0.132], [199.0, 0.139], [200.0, 0.147], [201.0, 0.156], [202.0, 0.164],
[203.0, 0.174], [204.0, 0.183], [205.0, 0.193], [206.0, 0.204], [207.0, 0.214],
[208.0, 0.226], [209.0, 0.238], [210.0, 0.250], [211.0, 0.263], [212.0, 0.276],
[213.0, 0.290], [214.0, 0.305], [215.0, 0.320], [216.0, 0.335], [217.0, 0.352],
[218.0, 0.369], [219.0, 0.386], [220.0, 0.405], [221.0, 0.423], [222.0, 0.443],
[223.0, 0.464], [224.0, 0.485], [225.0, 0.507], [226.0, 0.529], [227.0, 0.553],
[228.0, 0.577], [229.0, 0.602], [230.0, 0.628], [231.0, 0.655], [232.0, 0.683],
[233.0, 0.712], [234.0, 0.741], [235.0, 0.772], [236.0, 0.804], [237.0, 0.837],
[238.0, 0.870], [239.0, 0.905], [240.0, 0.941], [241.0, 0.978], [242.0, 1.016],
[243.0, 1.056], [244.0, 1.096], [245.0, 1.138], [246.0, 1.181], [247.0, 1.225],
[248.0, 1.271], [249.0, 1.317], [250.0, 1.366], [251.0, 1.415], [252.0, 1.466],
[253.0, 1.518], [254.0, 1.572], [255.0, 1.627], [256.0, 1.684], [257.0, 1.742],
[258.0, 1.802], [259.0, 1.864], [260.0, 1.927], [261.0, 1.991], [262.0, 2.058],
[263.0, 2.126], [264.0, 2.195], [265.0, 2.267], [266.0, 2.340], [267.0, 2.415],
[268.0, 2.492], [269.0, 2.570], [270.0, 2.651], [271.0, 2.733], [272.0, 2.818],
[273.0, 2.904], [274.0, 2.993], [275.0, 3.083], [276.0, 3.175], [277.0, 3.270],
[278.0, 3.367], [279.0, 3.465], [280.0, 3.566], [281.0, 3.670], [282.0, 3.775],
[283.0, 3.883], [284.0, 3.993], [285.0, 4.105], [286.0, 4.220], [287.0, 4.337],
[288.0, 4.456], [289.0, 4.578], [290.0, 4.703], [291.0, 4.829], [292.0, 4.959],
[293.0, 5.091], [294.0, 5.226], [295.0, 5.363], [296.0, 5.503], [297.0, 5.645],
[298.0, 5.790], [299.0, 5.938], [300.0, 6.089], [301.0, 6.243], [302.0, 6.399],
[303.0, 6.559], [304.0, 6.721], [305.0, 6.886], [306.0, 7.054], [307.0, 7.225],
[308.0, 7.399], [309.0, 7.576], [310.0, 7.756], [311.0, 7.940], [312.0, 8.126],
[313.0, 8.316], [314.0, 8.509], [315.0, 8.705], [316.0, 8.904], [317.0, 9.107],
[318.0, 9.312], [319.0, 9.522], [320.0, 9.734], [321.0, 9.950], [322.0,
10.170], [323.0, 10.393], [324.0, 10.619], [325.0, 10.849], [326.0, 11.082],
[327.0, 11.319], [328.0, 11.560], [329.0, 11.804], [330.0, 12.052], [331.0,
12.304], [332.0, 12.559], [333.0, 12.818], [334.0, 13.081], [335.0, 13.347],
[336.0, 13.618], [337.0, 13.892], [338.0, 14.170], [339.0, 14.452], [340.0,
14.738], [341.0, 15.028], [342.0, 15.322], [343.0, 15.620], [344.0, 15.922],
[345.0, 16.228], [346.0, 16.539], [347.0, 16.853], [348.0, 17.171], [349.0,
17.494], [350.0, 17.821], [351.0, 18.152], [352.0, 18.487], [353.0, 18.827],
[354.0, 19.171], [355.0, 19.519], [356.0, 19.871], [357.0, 20.228], [358.0,
20.590], [359.0, 20.955], [360.0, 21.326], [361.0, 21.700], [362.0, 22.080],
[363.0, 22.463], [364.0, 22.851], [365.0, 23.244], [366.0, 23.642], [367.0,
24.044], [368.0, 24.450], [369.0, 24.862]])

# channel WV2

temp2rad[7] = np.array([ [170.0, 0.020], [171.0, 0.022], [172.0, 0.024],
[173.0, 0.026], [174.0, 0.028], [175.0, 0.030], [176.0, 0.032], [177.0, 0.034],
[178.0, 0.037], [179.0, 0.039], [180.0, 0.042], [181.0, 0.045], [182.0, 0.048],
[183.0, 0.051], [184.0, 0.055], [185.0, 0.059], [186.0, 0.063], [187.0, 0.067],
[188.0, 0.071], [189.0, 0.076], [190.0, 0.080], [191.0, 0.085], [192.0, 0.091],
[193.0, 0.096], [194.0, 0.102], [195.0, 0.108], [196.0, 0.115], [197.0, 0.122],
[198.0, 0.129], [199.0, 0.136], [200.0, 0.144], [201.0, 0.152], [202.0, 0.161],
[203.0, 0.170], [204.0, 0.179], [205.0, 0.189], [206.0, 0.199], [207.0, 0.210],
[208.0, 0.221], [209.0, 0.232], [210.0, 0.244], [211.0, 0.257], [212.0, 0.270],
[213.0, 0.284], [214.0, 0.298], [215.0, 0.313], [216.0, 0.328], [217.0, 0.344],
[218.0, 0.361], [219.0, 0.378], [220.0, 0.396], [221.0, 0.414], [222.0, 0.434],
[223.0, 0.454], [224.0, 0.474], [225.0, 0.496], [226.0, 0.518], [227.0, 0.541],
[228.0, 0.565], [229.0, 0.590], [230.0, 0.615], [231.0, 0.642], [232.0, 0.669],
[233.0, 0.697], [234.0, 0.726], [235.0, 0.756], [236.0, 0.788], [237.0, 0.820],
[238.0, 0.853], [239.0, 0.887], [240.0, 0.922], [241.0, 0.959], [242.0, 0.996],
[243.0, 1.035], [244.0, 1.074], [245.0, 1.115], [246.0, 1.158], [247.0, 1.201],
[248.0, 1.246], [249.0, 1.292], [250.0, 1.339], [251.0, 1.388], [252.0, 1.438],
[253.0, 1.489], [254.0, 1.542], [255.0, 1.596], [256.0, 1.652], [257.0, 1.709],
[258.0, 1.768], [259.0, 1.828], [260.0, 1.890], [261.0, 1.953], [262.0, 2.019],
[263.0, 2.085], [264.0, 2.154], [265.0, 2.224], [266.0, 2.296], [267.0, 2.370],
[268.0, 2.445], [269.0, 2.522], [270.0, 2.602], [271.0, 2.683], [272.0, 2.766],
[273.0, 2.851], [274.0, 2.937], [275.0, 3.026], [276.0, 3.117], [277.0, 3.210],
[278.0, 3.305], [279.0, 3.402], [280.0, 3.502], [281.0, 3.603], [282.0, 3.707],
[283.0, 3.813], [284.0, 3.921], [285.0, 4.031], [286.0, 4.144], [287.0, 4.259],
[288.0, 4.377], [289.0, 4.497], [290.0, 4.619], [291.0, 4.744], [292.0, 4.871],
[293.0, 5.001], [294.0, 5.133], [295.0, 5.268], [296.0, 5.406], [297.0, 5.546],
[298.0, 5.689], [299.0, 5.835], [300.0, 5.983], [301.0, 6.134], [302.0, 6.288],
[303.0, 6.445], [304.0, 6.605], [305.0, 6.767], [306.0, 6.933], [307.0, 7.101],
[308.0, 7.272], [309.0, 7.447], [310.0, 7.624], [311.0, 7.805], [312.0, 7.988],
[313.0, 8.175], [314.0, 8.365], [315.0, 8.558], [316.0, 8.754], [317.0, 8.953],
[318.0, 9.156], [319.0, 9.362], [320.0, 9.571], [321.0, 9.784], [322.0,
10.000], [323.0, 10.219], [324.0, 10.442], [325.0, 10.669], [326.0, 10.899],
[327.0, 11.132], [328.0, 11.369], [329.0, 11.610], [330.0, 11.854], [331.0,
12.101], [332.0, 12.353], [333.0, 12.608], [334.0, 12.867], [335.0, 13.130],
[336.0, 13.396], [337.0, 13.666], [338.0, 13.940], [339.0, 14.218], [340.0,
14.500], [341.0, 14.786], [342.0, 15.075], [343.0, 15.369], [344.0, 15.666],
[345.0, 15.968], [346.0, 16.274], [347.0, 16.583], [348.0, 16.897], [349.0,
17.215], [350.0, 17.537], [351.0, 17.863], [352.0, 18.194], [353.0, 18.528],
[354.0, 18.867], [355.0, 19.211], [356.0, 19.558], [357.0, 19.910], [358.0,
20.266], [359.0, 20.626], [360.0, 20.991], [361.0, 21.361], [362.0, 21.734],
[363.0, 22.113], [364.0, 22.495], [365.0, 22.883], [366.0, 23.274], [367.0,
23.671], [368.0, 24.072], [369.0, 24.477]])

class _Calibrator(object):
    def __init__(self, hdr):
        self.hdr = hdr
        
    def __call__(self, image, calibrate=1):
        """From http://www.eumetsat.int/Home/Main/DataProducts/Calibration/MFGCalibration/index.htm?l=en
        """
        # don't know how to calibrate
        if calibrate == 0:
            return (image, 
                    "counts")
        
        if(self.hdr["space"] is None or
           self.hdr["calco"] is None):
            raise CalibrationError("Not implemented")
        radiances = (image - self.hdr["space"]) * self.hdr["calco"]
        if calibrate == 2:
            return (radiances,
                    "W m-2 sr-1")

        # using piecewise linear interpolation between lookup table values.

        return (np.interp(radiances.ravel(),
                         # known radiances
                         temp2rad[self.hdr["chan"]][:, 1],
                         # known bt's
                         temp2rad[self.hdr["chan"]][:, 0]).reshape(radiances.shape),
                "K")

def read_metadata(prologue, image_files):
    """ Selected items from the Meteosat-7 prolog file.
    """
    im = _xrit.read_imagedata(image_files[0])
    fp = StringIO(prologue.data)
    asc_hdr = _read_ascii_header(fp)
    bin_hdr = _read_binary_header(fp, asc_hdr['ProductType'])
    md = Metadata()
    md.calibrate = _Calibrator(bin_hdr)
    md.product_name = prologue.product_id
    pf = asc_hdr['Platform']
    if pf == 'M7':
        pf = 'MET7'
    md.satname = pf.lower()
    md.channel = prologue.product_name[:4]
    md.product_type = asc_hdr['ProductType']
    md.region_name = 'full disc'
    md.sublon = bin_hdr['ssp']
    md.first_pixel = asc_hdr['FirstPixelOri']
    md.data_type = bin_hdr['dtype']*8
    md.no_data_value = 0
    md.image_size = (int(asc_hdr['NumberOfPixels']), int(asc_hdr['NumberOfLines']))
    md.line_offset = int(asc_hdr['LineOffset'])
    # handle 24 hour clock
    d, t = strptime(asc_hdr['Date'], "%y%m%d"), int(asc_hdr['Time'])
    md.time_stamp = d + timedelta(hours=t//100, minutes=t%100)
    md.production_time = strptime(asc_hdr['ProdDate'] + asc_hdr['ProdTime'], "%y%m%d%H:%M:%S")
    md.calibration_unit = ""

    # Calibration table
    md.calibration_table = dict((('name', ''),
                                 ('unit', ''),
                                 ('table', None)))

    segment_size = im.structure.nl
    md.loff = im.navigation.loff + segment_size * (im.segment.seg_no - 1)
    md.coff = im.navigation.coff

    return md

if __name__ == '__main__':
    p = _xrit.read_prologue(sys.argv[1])
    print read_metadata(p, sys.argv[2:])
