/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#ifndef MEDIASCANNER_DBUSTYPES_H
#define MEDIASCANNER_DBUSTYPES_H

// Boost C++
#include <boost/tuple/tuple.hpp>

// C++ Standard Library
#include <list>
#include <map>
#include <set>
#include <string>
#include <utility>
#include <vector>

// Media Scanner Library
#include "mediascanner/glibutils.h"
#include "mediascanner/mediautils.h"
#include "mediascanner/property.h"
#include "mediascanner/utilities.h"

namespace mediascanner {
namespace dbus {

/**
 * @brief This calls describes the signatures of D-Bus interface members.
 * It is useful to avoid type-cast issues between C++ types such as std::string
 * and C types such as GVariantType.
 */
class Signature {
public:
    /**
     * @brief Implicit constructor that converts a C++ string.
     * @param signature The string representation of the signature.
     */
    Signature(const std::string &signature); // NOLINT:runtime/explicit

    /**
     * @brief Implicit constructor that converts a GVariantType.
     * @param signature The GVariantType representation of the signature.
     */
    Signature(const GVariantType *type); // NOLINT:runtime/explicit

    operator const char*() const;
    operator const GVariantType*() const;
    const std::string& str() const;

    static Signature array(const Signature &element_type);
    static Signature dictionary(const Signature &key_type,
                                const Signature &value_type);
    static Signature tuple(const Signature &element);

    std::string signature_;
};

template<typename T>
struct Type {
    typedef T value_type;

    static const Signature& signature();
    static GVariant* make_variant(T value);
    static T make_value(GVariant *variant);
};

////////////////////////////////////////////////////////////////////////////////
// Boxed variant types
////////////////////////////////////////////////////////////////////////////////

template<typename T>
struct BoxedTypeTrait;

template<typename T>
struct BoxedType {
    typedef T value_type;
    typedef typename BoxedTypeTrait<T>::boxed_type boxed_type;

    static const Signature& signature() {
        return Type<boxed_type>::signature();
    }

    static GVariant* make_variant(value_type value) {
        return Type<boxed_type>::make_variant(make_boxed(value));
    }

    static T make_value(GVariant *variant) {
        if (variant == nullptr)
            return value_type();

        return make_unboxed(Type<boxed_type>::make_value(variant));
    }

    static boxed_type make_boxed(value_type value);
    static value_type make_unboxed(boxed_type value);
};

template<> struct BoxedTypeTrait<Fraction> {
    typedef std::pair<Fraction::int_type, Fraction::int_type> boxed_type;
};

template<> struct BoxedTypeTrait<DateTime> {
    typedef uint64_t boxed_type;
};

template<> class Type<Fraction> : public BoxedType<Fraction> { };
template<> class Type<DateTime> : public BoxedType<DateTime> { };

////////////////////////////////////////////////////////////////////////////////
// Container types
////////////////////////////////////////////////////////////////////////////////

template<typename T1, typename T2>
struct Type< std::pair<T1, T2> > {
    typedef std::pair<T1, T2> value_type;

    static const Signature& signature() {
        static const Signature signature =
                Signature::tuple(Type<T1>::signature().str() +
                                 Type<T2>::signature().str());

        return signature;
    }

    static GVariant* make_variant(const value_type &value) {
        GVariant *elements[2] = {
            Type<T1>::make_variant(value.first),
            Type<T2>::make_variant(value.second)
        };

        return g_variant_new_tuple(elements, G_N_ELEMENTS(elements));
    }

    static value_type make_value(GVariant *variant) {
        if (variant == nullptr)
            return value_type();

        Wrapper<GVariant> v1 = take(g_variant_get_child_value(variant, 0));
        Wrapper<GVariant> v2 = take(g_variant_get_child_value(variant, 1));

        return value_type(Type<T1>::make_value(v1.get()),
                          Type<T2>::make_value(v2.get()));
    }
};

template<>
struct Type<Property::Value> {
    static const Signature& signature();
    static GVariant* make_variant(const Property::Value &value);
    static Property::Value make_value(GVariant *variant);
};

template<>
struct Type<Property::ValueMap> {
    static const Signature& signature();
    static GVariant* make_variant(const Property::ValueMap &value);

    static Property::ValueMap make_value(GVariant *variant) {
        return make_value(variant, nullptr);
    }

    static Property::ValueMap make_value(GVariant *variant,
                                         std::set<std::string> *bad_keys);
};

template<>
struct Type<MediaInfo> {
    static const Signature& signature();
    static GVariant* make_variant(const MediaInfo &value);

    static MediaInfo make_value(GVariant *variant) {
        return make_value(variant, nullptr);
    }

    static MediaInfo make_value(GVariant *variant,
                                std::set<std::string> *bad_keys);
};

namespace internal {

template<typename Container>
struct SequenceType {
    typedef typename Container::value_type value_type;

    static const Signature& signature() {
        static const Signature signature =
                Signature::array(Type<value_type>::signature());

        return signature;
    }

    static GVariant* make_variant(const Container &value) {
        GVariantBuilder builder;

        g_variant_builder_init(&builder, signature());

        for (typename Container::const_iterator
             it = value.begin(), end = value.end(); it != end; ++it) {
            GVariant *const element = Type<value_type>::make_variant(*it);
            g_variant_builder_add_value(&builder, element);
        }

        return g_variant_builder_end(&builder);
    }

    static Container make_value(GVariant *variant) {
        Container container;

        if (variant) {
            for (size_t i = 0, l = g_variant_n_children(variant); i < l; ++i) {
                GVariant *const element = g_variant_get_child_value(variant, i);
                add(Type<value_type>::make_value(element), &container);
            }
        }

        return container;
    }

private:
    static void add(const value_type &element,
                    std::list<value_type> *container) {
        container->push_back(element);
    }

    static void add(const value_type &element,
                    std::set<value_type> *container) {
        container->insert(element);
    }

    static void add(const value_type &element,
                    std::vector<value_type> *container) {
        container->push_back(element);
    }
};

} // namespace internal

template<typename T>
struct Type<std::list<T> > : public internal::SequenceType< std::list<T> > {
};

template<typename T>
struct Type<std::set<T> > : public internal::SequenceType< std::set<T> > {
};

template<typename T>
struct Type<std::vector<T> > : public internal::SequenceType< std::vector<T> > {
};

template<typename K, typename V>
struct Type<std::map<K, V> > {
    static const Signature& signature() {
        static const Signature signature =
                Signature::dictionary(Type<K>::signature(),
                                      Type<V>::signature());

        return signature;
    }

    static std::map<K, V> make_value(GVariant *variant) {
        std::map<K, V> map;

        if (variant) {
            for (size_t i = 0, l = g_variant_n_children(variant); i < l; ++i) {
                GVariant *const element = g_variant_get_child_value(variant, i);
                map.insert(Type<typename std::map<K, V>::value_type>::
                           make_value(element));
            }
        }

        return map;
    }
};

namespace internal {

template<size_t N, typename T>
struct TupleTail {
    typedef typename boost::tuples::element<N - 1, T>::type value_type;
    typedef TupleTail<N - 1, T> predecessor;

    static const Signature& signature() {
        static const Signature signature =
                predecessor::signature().str() +
                Type<value_type>::signature().str();

        return signature;
    }

    static void make_value(GVariant *variant, T *value) {
        predecessor::make_value(variant, value);
        value_type &element_target = boost::tuples::get<N - 1>(*value);

        if (variant == nullptr) {
            element_target = value_type();
            return;
        }

        GVariant *const element = g_variant_get_child_value(variant, N - 1);
        element_target = Type<value_type>::make_value(element);
    }

    static void make_variant(const T &value, GVariantBuilder *builder) {
        predecessor::make_variant(value, builder);
        const value_type &element_value = boost::tuples::get<N - 1>(value);
        GVariant *const element = Type<value_type>::make_variant(element_value);
        g_variant_builder_add_value(builder, element);
    }
};

template<typename T>
struct TupleTail<0, T> {
    static const Signature &signature() {
        static const Signature signature = std::string();
        return signature;
    }

    static void make_value(GVariant *, T *) {
    }

    static void make_variant(const T &, GVariantBuilder *) {
    }
};

} // namespace internal

template<typename T1 = boost::tuples::null_type,
         typename T2 = boost::tuples::null_type,
         typename T3 = boost::tuples::null_type,
         typename T4 = boost::tuples::null_type,
         typename T5 = boost::tuples::null_type,
         typename T6 = boost::tuples::null_type,
         typename T7 = boost::tuples::null_type,
         typename T8 = boost::tuples::null_type>
struct TupleType {
    typedef boost::tuples::tuple<T1, T2, T3, T4, T5, T6, T7, T8> tuple_type;
    typedef boost::tuples::length<tuple_type> length;
    typedef internal::TupleTail<length::value, tuple_type> tail;

    static const Signature& signature() {
        static const Signature signature = Signature::tuple(tail::signature());
        return signature;
    }

    static GVariant* make_variant(const tuple_type &value) {
        GVariantBuilder builder;
        g_variant_builder_init(&builder, signature());
        tail::make_variant(value, &builder);
        return g_variant_builder_end(&builder);
    }

    static tuple_type make_value(GVariant *variant) {
        tuple_type value;

        if (variant)
            tail::make_value(variant, &value);

        return value;
    }
};

template<typename T1, typename T2, typename T3, typename T4,
         typename T5, typename T6, typename T7, typename T8>
struct Type< boost::tuples::tuple<T1, T2, T3, T4, T5, T6, T7, T8> >
        : public TupleType<T1, T2, T3, T4, T5, T6, T7, T8> {
};

} // namespace dbus
} // namespace mediascanner

#endif // MEDIASCANNER_DBUSTYPES_H
