#
# Copyright (c) 2010 Canonical
#
# Written by Gustavo Niemeyer <gustavo@niemeyer.net>
#
# This file is part of the Xpresser GUI automation library.
#
# Xpresser is free software; you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3,
# as published by the Free Software Foundation.
#
# Xpresser is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
import os

from xpresser.imagedir import ImageDir, ImageDirError
from xpresser.image import DEFAULT_SIMILARITY

from xpresser.lib.testing import TestCase


class ImageDirTest(TestCase):

    def setUp(self):
        self.dir = ImageDir()

    def test_load_empty_directory(self):
        dirname = self.makeDir()
        self.dir.load(dirname)
        # Nothing happens.

    def test_load_empty_file(self):
        filename = self.makeFile("", basename="xpresser.ini")
        self.dir.load(os.path.dirname(filename))
        # Nothing happens.

    def test_load_image_with_missing_filename_option(self):
        filename = self.makeFile("[image foo]\n", basename="xpresser.ini")
        self.assertRaises(ImageDirError,
                          self.dir.load, os.path.dirname(filename))

    def test_load_image_with_non_existing_file(self):
        filename = self.makeFile("[image foo]\nfilename = nonexistent.png",
                                 basename="xpresser.ini")
        self.assertRaises(ImageDirError,
                          self.dir.load, os.path.dirname(filename))

    def test_load_and_get_image(self):
        filename = self.makeFile("[image foo]\n"
                                 "filename = fake_foo.png\n"
                                 "focus_delta = +1 -2\n"
                                 "similarity = 0.5\n"
                                 "[image bar]\n"
                                 "filename = fake_bar.png\n"
                                 "focus_delta = -30 +40\n"
                                 "[image baz]\n"
                                 "filename = fake_baz.png\n",
                                 basename="xpresser.ini")
        dirname = os.path.dirname(filename)
        self.makeFile("", basename="fake_foo.png", dirname=dirname)
        self.makeFile("", basename="fake_bar.png", dirname=dirname)
        self.makeFile("", basename="fake_baz.png", dirname=dirname)
        self.dir.load(dirname)
        foo = self.dir.get("foo")
        bar = self.dir.get("bar")
        baz = self.dir.get("baz")

        self.assertEquals(foo.name, "foo")
        self.assertEquals(foo.filename, os.path.join(dirname, "fake_foo.png"))
        self.assertEquals(foo.similarity, 0.5)
        self.assertEquals(foo.focus_delta, (1, -2))

        self.assertEquals(bar.name, "bar")
        self.assertEquals(bar.filename, os.path.join(dirname, "fake_bar.png"))
        self.assertEquals(bar.similarity, DEFAULT_SIMILARITY)
        self.assertEquals(bar.focus_delta, (-30, 40))

        self.assertEquals(baz.name, "baz")
        self.assertEquals(baz.filename, os.path.join(dirname, "fake_baz.png"))
        self.assertEquals(baz.similarity, DEFAULT_SIMILARITY)
        self.assertEquals(baz.focus_delta, (0, 0))

    def test_bad_focus_delta(self):
        filename = self.makeFile("[image foo]\n"
                                 "filename = fake_foo.png\n"
                                 "focus_delta = +1,+2\n",
                                 basename="xpresser.ini")
        dirname = os.path.dirname(filename)
        self.makeFile("", basename="fake_foo.png", dirname=dirname)
        self.assertRaises(ImageDirError, self.dir.load, dirname)

    def test_bad_similarity(self):
        filename = self.makeFile("[image foo]\n"
                                 "filename = fake_foo.png\n"
                                 "similarity = bad\n",
                                 basename="xpresser.ini")
        dirname = os.path.dirname(filename)
        self.makeFile("", basename="fake_foo.png", dirname=dirname)
        self.assertRaises(ImageDirError, self.dir.load, dirname)

    def test_load_image_implicitly(self):
        filename = self.makeFile("", basename="image_name.png")
        self.dir.load(os.path.dirname(filename))
        image = self.dir.get("image_name")
        self.assertEquals(image.name, "image_name")
        self.assertEquals(image.filename, filename)

    def test_ini_image_takes_precedence_over_implicitly_loaded(self):
        filename = self.makeFile("[image image_name]\n"
                                 "filename = image_name.png\n"
                                 "focus_delta = +1 +1\n",
                                 basename="xpresser.ini")
        dirname = os.path.dirname(filename)
        image_filename = self.makeFile("", basename="image_name.png",
                                       dirname=dirname)
        self.dir.load(dirname)
        image = self.dir.get("image_name")
        self.assertEquals(image.focus_delta, (1, 1))

    def test_unknown_files_arent_loaded_implicitly(self):
        image_filename1 = self.makeFile("", basename="random_file.txt")
        dirname = os.path.dirname(image_filename1)
        image_filename2 = self.makeFile("", basename="random_file.unknown",
                                        dirname=os.path.dirname(image_filename1))
        self.dir.load(dirname)
        self.assertEquals(self.dir.get("random_file"), None)
