# Copyright 2013-2014 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""Manage a MAAS installation."""

from __future__ import (
    absolute_import,
    print_function,
    unicode_literals,
    )

str = None

__metaclass__ = type
__all__ = [
    'MAASFixture',
    ]


from datetime import timedelta
import httplib
import json
import logging
from random import choice
from string import ascii_letters
from textwrap import dedent

from apiclient.creds import convert_string_to_tuple
from apiclient.maas_client import (
    MAASClient,
    MAASDispatcher,
    MAASOAuth,
    urllib2,
    )
from fixtures import Fixture
from maastest import utils
from maastest.maas_enums import NODEGROUPINTERFACE_MANAGEMENT
import netaddr
from testtools.monkey import MonkeyPatcher


MAAS_ADMIN_USER = 'admin'


def compose_rewrite_for_default_maas_url(config_file, default_maas_url):
    """Return a shell command to rewrite `DEFAULT_MAAS_URL` in config."""
    # Comment out existing DEFAULT_MAAS_URL, and add new one.
    replacement = dedent("""
        # Replaced by maas-test:
        # \\1
        DEFAULT_MAAS_URL = "%s"
        """ % default_maas_url).replace('\n', '\\n')

    return [
        'sed',
        '--in-place=.bak',
        's|^\\(DEFAULT_MAAS_URL *=.*\\)$|%s|' % replacement,
        config_file,
        ]

# Log files that are interesting when debugging MAAS issues.  They will
# be collected just before the fixture is disposed of.
LOG_FILES = [
    # Syslog contains DHCP requests.
    '/var/log/syslog',
    # DHCP lease file.
    '/var/lib/maas/dhcp/dhcpd.leases',
    # MAAS logs.
    '/var/log/maas/maas.log',
    '/var/log/maas/pserv.log',
    '/var/log/maas/celery-region.log',
    '/var/log/maas/celery.log',
    # Apache logs.
    '/var/log/apache2/access.log',
    '/var/log/apache2/error.log',
]


class NoBootImagesError(Exception):
    """No boot images present on the master cluster."""


class MAASFixture(Fixture):
    """A fixture for a MAAS server."""

    def __init__(self, kvmfixture, proxy_url, series, architecture,
                 simplestreams_filter):
        """Initialise a MAAS server installed in a KVM instance.

        :param kvmfixture: The `maastest.KVMFixture` corresponding to the
            KVM instance on which the MAAS server will be installed.
        :param series: The Ubuntu series that this MAAS server should use
            to enlist and commission nodes.
        :param architecture: The architecture that this MAAS server should
            import images for, e.g. 'amd64', 'arm/highbank'.  If the
            subarchitecture isn't specified, 'generic' is assumed.
        """
        self.kvm_fixture = kvmfixture
        self.proxy_url = proxy_url
        self.series = series
        self.simplestreams_filter = simplestreams_filter
        self.architecture = architecture
        self.installed = False
        self._maas_admin = None

    def install_maas(self):
        """Install and configure MAAS in the virtual machine.

        This method is idempotent.  It's OK to run it more than once.
        """
        if self.installed:
            # Already installed.  Nothing to do.
            return
        # Install the English language pack first.  If this is not done
        # before postgres is installed, on some systems it won't be able to
        # set up its main cluster.
        # TODO: Is there no better way to ensure this, e.g. through a
        # dependency?
        self.kvm_fixture.install_packages(['language-pack-en'])

        maas_version = self.get_maas_version()
        logging.info("Installing MAAS (version %s)..." % maas_version)
        # Now we can install maas (which also installs postgres).
        self.kvm_fixture.install_packages(['maas', 'maas-dhcp', 'maas-dns'])
        logging.info("Done installing MAAS.")
        self.installed = True

    def get_maas_version(self):
        """Return the version of MAAS that is to be installed."""
        _, policy, _ = self.kvm_fixture.run_command(
            ['apt-cache', 'policy', 'maas'], check_call=True)
        return utils.extract_package_version(policy)

    @property
    def maas_admin(self):
        """Return the correct maas admin command.

        If maas-region-admin is available, use it (on trusty or later),
        otherwise fall back to the old 'maas' command.
        """
        if self._maas_admin is None:
            retcode, _, _ = self.kvm_fixture.run_command(
                ['which', 'maas-region-admin'])
            if retcode == 0:
                self._maas_admin = 'maas-region-admin'
            else:
                self._maas_admin = 'maas'
        return self._maas_admin

    def configure_default_maas_url(self):
        """Set `DEFAULT_MAAS_URL` in the virtual machine.

        We'd prefer to do this by preseed, before installing MAAS, but that
        doesn't currently work (bug 1251175).  Instead, we patch up
        configuration after installation.
        """
        assert self.kvm_fixture.direct_ip is not None, (
            "configure_default_maas_url should only be called on a "
            "machine that has a 'direct interface.'")
        rewrite_command = compose_rewrite_for_default_maas_url(
            '/etc/maas/maas_local_settings.py',
            'http://%s/MAAS' % self.kvm_fixture.direct_ip)
        self.kvm_fixture.run_command(
            ['sudo'] + rewrite_command, check_call=True)
        # Restarting apache2 sometimes times out: retry the command a
        # couple of times before giving up.  See bug #1260363.
        args = ['sudo', 'service', 'apache2', 'restart']
        for retry in utils.retries(delay=10, timeout=60):
            retcode, stdout, stderr = self.kvm_fixture.run_command(
                args, check_call=False)
            if retcode == 0:
                break
        else:
            raise utils.make_exception(args, retcode, stdout, stderr)

    def query_api_key(self, username):
        """Return the API key for the given MAAS user."""
        # The "apikey" command prints the user's API key to stdout.
        return_code, stdout, stderr = self.kvm_fixture.run_command([
            'sudo', self.maas_admin, 'apikey',
            '--username=%s' % username,
            ], check_call=True)
        return stdout.strip()

    def create_maas_admin(self):
        """Create a MAAS admin user.

        Invoke this after MAAS has been installed on the virtual machine.

        :return: User name and password for the new admin.
        """
        username = MAAS_ADMIN_USER
        password = ''.join(choice(ascii_letters) for counter in range(8))
        # Email address really does not matter.  MAAS never sends anything
        # there. Its hostname doesn't have to resolve, although it has to
        # look like a FQDN.
        email = 'root@localhost.local'

        return_code, _, stderr = self.kvm_fixture.run_command([
            'sudo', self.maas_admin, 'createadmin',
            '--username=%s' % username,
            '--password=%s' % password,
            '--email=%s' % email,
            ], check_call=True)

        return username, password

    def dump_data(self):
        """Dump the NodeCommissionResult table to stdout.

        This allows us to capture the lshw results for commissioned
        nodes. This method will be called when the fixture get cleaned
        up.
        """
        return_code, stdout, _ = self.kvm_fixture.run_command([
            'sudo', self.maas_admin, 'dumpdata',
            'metadataserver.NodeCommissionResult'
            ])

    def import_maas_images(self, series, architecture,
                           simplestreams_filter=None):
        """Import boot images into the MAAS instance.

        This download gigabytes of data to the virtual machine's disk.

        :param series: The Ubuntu series for which images should be imported.
        :param archtecture: The architecture for which images should be
            imported, e.g. 'amd64', 'arm/hightbank'.  If the
            subarchitecture isn't specified, 'generic' is assumed.
        """
        arch_list = utils.mipf_arch_list(architecture)
        arch_string = ' '.join(arch_list)
        logging.info(
            "Importing boot images series=%s, architectures=%s..." % (
                series, arch_string))
        # Import boot images.
        # XXX jtv 2014-04-03: Configure /etc/maas/bootresources.yaml to import
        # just the right series & architecture.
        self.kvm_fixture.run_command(
            [
                'sudo',
                'http_proxy=%s' % self.proxy_url,
                'https_proxy=%s' % self.proxy_url,
                'maas-import-pxe-files',
            ],
            check_call=True)
        logging.info("Done importing boot images.")

    def wait_until_boot_images_scanned(self):
        """Wait until the master cluster has reported boot images."""
        # Poll until the boot images have been reported.
        timeout = timedelta(minutes=10).total_seconds()
        for _ in utils.retries(timeout=timeout, delay=10):
            boot_images = self.list_boot_images()
            if len(boot_images) > 0:
                return

        raise NoBootImagesError(
            "Boot image download timed out: cluster reported no images.")

    def list_boot_images(self):
        """Return the boot images of the master cluster."""
        ng_uuid = self.get_master_ng_uuid()
        uri = utils.get_uri('nodegroups/%s/boot-images/' % ng_uuid)
        response = self.admin_maas_client.get(uri)
        if response.code != httplib.OK:
            raise Exception(
                "Error getting boot images for cluster '%s'" % ng_uuid)
        return json.loads(response.read())

    def configure(self, name, value):
        """Set a config value in MAAS."""
        uri = utils.get_uri('maas/')
        response = self.admin_maas_client.post(
            uri, op='set_config', name=name, value=value)
        if response.code != httplib.OK:
            # TODO: include the response's content in the exception
            # message, here and in other places in this file.
            raise Exception("Error configuring '%s'" % name)

    def configure_default_series(self, series):
        """Set the default series used for enlistment and commissioning."""
        self.configure('commissioning_distro_series', series)

    def configure_http_proxy(self, proxy_url):
        """Set the proxy to be used by MAAS."""
        self.configure('http_proxy', proxy_url)

    def setUp(self):
        """Install and configure MAAS inside the virtual machine.

        :param kvm_fixture: `KVMFixture` that controls the virtual machine.
        :return: Username, password and API key for the MAAS admin user.
        """
        super(MAASFixture, self).setUp()
        # MAAS installation and configuration.
        self.install_maas()
        if self.kvm_fixture.direct_ip is not None:
            self.configure_default_maas_url()
        # Admin creation.
        admin_user, admin_password = self.create_maas_admin()
        api_key = self.query_api_key(admin_user)
        maas_client = self.get_maas_api_client(api_key)
        self.admin_user = admin_user
        self.admin_password = admin_password
        self.admin_api_key = api_key
        self.admin_maas_client = maas_client
        # MAAS setup.
        if self.proxy_url:  # None or '' indicates no proxy.
            self.configure_http_proxy(self.proxy_url)

        # We should have a working MAAS at this point, so if things break from
        # here on, try to provide relevant information for debugging.
        self.log_connection_details()
        self.addCleanup(self.collect_logs)
        self.addCleanup(self.dump_data)

        # Now do the rest of the setup: import images, configure the
        # controllers, and so on.
        self.import_maas_images(
            series=self.series, architecture=self.architecture,
            simplestreams_filter=self.simplestreams_filter)
        self.wait_until_boot_images_scanned()
        self.configure_default_series(self.series)
        if self.kvm_fixture.direct_ip is not None:
            self.check_cluster_connected()
            self.configure_cluster()

    def log_connection_details(self):
        """Log the details on how to connect to this MAAS server."""
        logging.info(
            "MAAS server URL: http://%s/MAAS/ username:%s, password:%s" % (
                self.kvm_fixture.ip_address(),
                self.admin_user,
                self.admin_password,
            )
        )
        logging.info(
            "SSH login: sudo ssh -i %s %s" % (
                self.kvm_fixture.ssh_private_key_file,
                self.kvm_fixture.identity(),
            )
        )

    def collect_logs(self):
        for filename in LOG_FILES:
            # The result of every command run on the VM goes through
            # testtools.TestCase.addDetail() so we just need to
            # 'cat' the files to get their content included in the
            # report.
            self.kvm_fixture.run_command([
                'sudo', 'cat', filename], check_call=False)

    def get_maas_api_client(self, api_key):
        """Create and return a MAASClient.

        :param kvm_fixture: `KVMFixture` that controls the virtual machine.
        :return: `MAASClient` instance for the MAAS admin user.
        """
        credentials = convert_string_to_tuple(api_key)
        auth = MAASOAuth(*credentials)
        return MAASClient(
            auth, MAASDispatcher(),
            "http://%s/MAAS" % self.kvm_fixture.ip_address())

    def get_master_ng_uuid(self):
        """Get the UUID of the master nodegroup from the API."""
        uri = utils.get_uri('nodegroups/')
        response = self.admin_maas_client.get(uri, op='list')
        if response.code != httplib.OK:
            raise Exception("Error listing the clusters")
        nodegroups = json.loads(response.read())
        if len(nodegroups) != 1:
            raise Exception(
                "Expected exactly 1 nodegroup, but saw %d." % len(nodegroups))
        return nodegroups[0]['uuid']

    def check_cluster_connected(self):
        for retry in utils.retries(timeout=3 * 60):
            name = self.get_master_ng_uuid()
            if name != 'master':
                # The master cluster is connected.  The master nodegroup
                # had its uuid field updated from 'master' to its UUID.
                return
        raise Exception("Master cluster failed to connect.")

    def configure_cluster(self):
        network = self.kvm_fixture.direct_network
        first_ip = self.kvm_fixture.direct_first_available_ip()
        last_ip = "%s" % netaddr.IPAddress(
            self.kvm_fixture.direct_network.last - 1)
        dhcp_config = {
            "ip": self.kvm_fixture.direct_ip,
            "interface": "eth1",
            "subnet_mask": "%s" % network.netmask,
            "broadcast_ip": "%s" % network.broadcast,
            "router_ip": self.kvm_fixture.direct_ip,
            "management": '%s' % NODEGROUPINTERFACE_MANAGEMENT.DHCP_AND_DNS,
            "ip_range_low": first_ip,
            "ip_range_high": last_ip,
        }
        uuid = self.get_master_ng_uuid()
        uri = utils.get_uri(
            'nodegroups/%s/interfaces/%s/' % (uuid, 'eth1'))

        # XXX: rvb 2013-11-14 bug=1251214: workaround a bug affecting
        # 'PUT' requests in urllib2 (which is used by MAASClient).
        patcher = MonkeyPatcher()
        patcher.add_patch(urllib2.Request, "get_method", lambda self: 'PUT')
        response = patcher.run_with_patches(
            self.admin_maas_client.put, uri, **dhcp_config)

        if response.code != httplib.OK:
            raise Exception("Error configure the master cluster")
