#include <iostream>

#include "test_helpers.hxx"

using namespace PGSTD;
using namespace pqxx;


// Simple test program for libpqxx.  Open connection to database, start
// a dummy transaction to gain nontransactional access, and perform a query.

namespace
{
// Define a pqxx::noticer to process warnings generated by the database
// connection and (in this case) pass them to cerr.  This is optional.
struct ReportWarning : noticer
{
  ReportWarning():noticer(){}	// Silences bogus warning in some gcc versions

  virtual void operator()(const char Msg[]) throw ()
  {
    cerr << Msg;
  }
};


void test_014(transaction_base &orgT)
{
  connection_base &C(orgT.conn());
  orgT.abort();

  // Tell C to report its warnings through std::cerr instead of the default
  // (which is to print to stderr).  This is done just to show that the way
  // messages are processed can be changed by the client program.
  noticer *MyNoticer = new ReportWarning;
  // This is not a memory leak: C stores MyNoticer in an auto_ptr that will
  // delete the object on destruction.
  C.set_noticer(auto_ptr<noticer>(MyNoticer));

  PQXX_CHECK(C.get_noticer() == MyNoticer, "Lost noticer.");

  // Now use our noticer to output a diagnostic message.  Note that the
  // message must end in a newline.
  C.process_notice("Opened connection\n");

  // ProcessNotice() can take either a C++ string or a C-style const char *.
  const string HostName = (C.hostname() ? C.hostname() : "<local>");
  C.process_notice(string() +
	"database=" + C.dbname() + ", "
	"username=" + C.username() + ", "
	"hostname=" + HostName + ", "
	"port=" + to_string(C.port()) + ", "
	"options='" + static_cast<connection &>(C).options() + "', "
	"backendpid=" + to_string(C.backendpid()) + "\n");

  // Begin a "non-transaction" acting on our current connection.  This is
  // really all the transactional integrity we need since we're only
  // performing one query which does not modify the database.
  nontransaction T(C, "test14");

  // The Transaction family of classes also has ProcessNotice() functions.
  // These simply pass the notice through to their connection, but this may
  // be more convenient in some cases.  All ProcessNotice() functions accept
  // C++ strings as well as C strings.
  T.process_notice(string("Started nontransaction\n"));

  result R( T.exec("SELECT * FROM pg_tables") );

  // Give some feedback to the test program's user prior to the real work
  T.process_notice(to_string(R.size()) + " "
	"result tuples in transaction " +
	T.name() + "\n");

  for (result::const_iterator c = R.begin(); c != R.end(); ++c)
  {
    string N;
    c[0].to(N);

    cout << '\t' << to_string(c.num()) << '\t' << N << endl;
  }

  // "Commit" the non-transaction.  This doesn't really do anything since
  // NonTransaction doesn't start a backend transaction.
  T.commit();
}
} // namespace

PQXX_REGISTER_TEST_T(test_014, nontransaction)
