/*
 * MimeOutputStream.java - A Filter stream for MIME encoding
 *
 * Copyright (C) 2000,,2003 2002 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */ 


package net.sourceforge.groboutils.util.io.v1;

import java.io.FilterOutputStream;
import java.io.OutputStream;
import java.io.IOException;
import java.awt.TextComponent;

/**
 * java.io.FilterOutputStream implementation for Mime base 64. Not incredibly
 * efficient, but it works and is small.
 * <P>
 * All we need to implement are:
 * <ul>
 *     <li>write(int)</li>
 *     <li>flush()</li>
 * </ul>
 *
 * @author   Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @since    0.9.0 Alpha (early 2000)
 * @version  $Date: 2003/02/10 22:52:45 $
 */
public class MimeOutputStream extends FilterOutputStream
{
    private int bits = 0, spare = 0;

    /**
     * Mime character set translation
     */
    private static final int[] charset = {
       'A','B','C','D','E','F','G','H','I','J','K','L','M','N','O','P','Q','R',
       'S','T','U','V','W','X','Y','Z','a','b','c','d','e','f','g','h','i','j',
       'k','l','m','n','o','p','q','r','s','t','u','v','w','x','y','z','0','1',
       '2','3','4','5','6','7','8','9','+','/' };
    private static final int pad = '=';

    /** Constructor! */
    public MimeOutputStream( OutputStream o )
    { super(o); }

    /**
     * Write the specified <code>byte</code>, performing mime encoding.
     *
     * <p>Override this method, since all other write methods call it.
     *
     * @exception  IOException  If an I/O error occurs
     */
    public void write(int c) throws IOException
    {
       int s, t;
       switch (bits)
       {
          case 0: bits++;
              s = (c >> 2) & 0x3f;
              spare = c & 0x03;
              out.write( charset[s] );
            break;
          case 1: bits++;
              s = (spare << 4) | ((c >> 4) & 0x0f);
              spare = c & 0x0f;
              out.write( charset[s] );
            break;
          case 2: bits = 0;
              s = (spare << 2) | ((c >> 6) & 0x03);
              t = c & 0x3f;
              out.write( charset[s] );
              out.write( charset[t] );
            break;
       }
    }


    /**
     * Flush the stream.  If the stream has saved any characters from the
     * various write() methods in a buffer, write them immediately to their
     * intended destination.  Then, if that destination is another character or
     * byte stream, flush it.  Thus one flush() invocation will flush all the
     * buffers in a chain of Writers and OutputStreams.
     * <P>
     * This version does not buffer, but Mime encoding may have some trailing
     * stuff, which needs padding.
     *
     * @exception  IOException  If an I/O error occurs
     */
    public void flush() throws IOException
    {
       switch (bits)
       {
          case 1:
              out.write( charset[spare << 4] );
              out.write( pad );
              out.write( pad );
            break;
          case 2:
              out.write( charset[spare << 2] );
              out.write( pad );
            break;
       }
       super.flush();
    }
}
