/*
 * @(#)HashCacheUTest.java
 *
 * Copyright (C) 2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.util.datastruct.v1;

import net.sourceforge.groboutils.junit.v1.MultiThreadedTestRunner;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

/**
 * 
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @since     April 17, 2001 (Alpha 0.9.0)
 * @version   $Date: 2003/05/23 20:57:00 $
 */
public class HashCacheUTest extends TestCase
{
    private static final Class THIS_CLASS = HashCacheUTest.class;
    
    public HashCacheUTest( String name )
    {
        super( name );
    }
    
    public static Test suite()
    {
        TestSuite suite = new TestSuite( THIS_CLASS );
        
        return suite;
    }
    
    public static void main( String[] args )
    {
        String[] name = { THIS_CLASS.getName() };
        
        // junit.textui.TestRunner.main( name );
        // junit.swingui.TestRunner.main( name );
        
        junit.textui.TestRunner.main( name );
    }
    
    protected void setUp() throws Exception
    {
        super.setUp();
        
        // set ourself up
    }
    
    
    protected void tearDown() throws Exception
    {
        // tear ourself down
        
        
        super.tearDown();
    }
    
    
    public class OMgr implements HashCache.ObjectManager
    {
        private int count = 0;
        private int closecount = 0;
        public synchronized Object createObject( Object key )
        {
            return "["+key+"] "+(++count);
        }
        
        public synchronized void cleanUpObject( Object key, Object value )
        {
            ++this.closecount;
        }
        
        public int getCreationCount()
        {
            return this.count;
        }
        
        public int getCloseCount()
        {
            return this.closecount;
        }
    }
    
    
    
    public void testConstructor1()
    {
        HashCache oc = new HashCache( new OMgr(), 10 );
        assertEquals( "Size must be instantiation size.",
            10, oc.getMaxSize() );
        assertEquals( "Overflows must be at zero.",
            0, oc.getOverflowCount() );
    }
    
    
    public void testConstructor2()
    {
        try
        {
            new HashCache( null, 10 );
            fail("Did not throw IllegalArgumentException.");
        }
        catch (IllegalArgumentException iae)
        {
            // test exception
        }
    }
    
    
    public void testConstructor3()
    {
        try
        {
            new HashCache( null, 0 );
            fail("Did not throw IllegalArgumentException.");
        }
        catch (IllegalArgumentException iae)
        {
            // test exception
        }
    }
    
    
    public void testConstructor4()
    {
        try
        {
            new HashCache( new OMgr(), 1 );
            fail("Did not throw IllegalArgumentException.");
        }
        catch (IllegalArgumentException iae)
        {
            // test exception
        }
    }
    
    
    public void testConstructor5()
    {
        try
        {
            new HashCache( new OMgr(), -1 );
            fail("Did not throw IllegalArgumentException.");
        }
        catch (IllegalArgumentException iae)
        {
            // test exception
        }
    }
    
    
    public void testGet1()
    {
        OMgr om = new OMgr();
        HashCache hc = new HashCache( om, 10 );
        try
        {
            hc.get( null );
        }
        catch (NullPointerException npe)
        {
            // test exception?
        }
    }
    
    
    public void testGet2()
    {
        OMgr om = new OMgr();
        HashCache hc = new HashCache( om, 10 );
        Object o = hc.get( "a" );
        assertNotNull(
            "get returned null.",
            o );
        assertTrue(
            "incorrect type returned.",
            o instanceof String );
        assertEquals(
            "Incorrect value of returned object.",
            "[a] 1",
            (String)o );
        assertEquals(
            "The mgr reported the creation.",
            1,
            om.getCreationCount() );
        
        Object o2 = hc.get( "a" );
        assertSame(
            "Did not return the same object.",
            o,
            o2 );
        assertEquals(
            "The mgr was not called for the creation.",
            1,
            om.getCreationCount() );
    }
    
    
    public void testGet3()
    {
        OMgr om = new OMgr();
        HashCache hc = new HashCache( om, 2 );
        Object o = hc.get( "a" );
        Object o2 = hc.get( "b" );
        Object o3 = hc.get( "c" );
        assertEquals(
            "The mgr was not called for the creation correctly.",
            3,
            om.getCreationCount() );
        assertEquals(
            "The mgr was not called for the clean-up.",
            1,
            om.getCloseCount() );
        assertEquals(
            "The overflow count is wrong.",
            1,
            hc.getOverflowCount() );
        assertEquals(
            "The size is wrong.",
            2,
            hc.getSize() );
    }
    
    
    public void testGet4()
    {
        System.out.println("Get4:");
        OMgr om = new OMgr();
        HashCache hc = new HashCache( om, 10 );
        int requests[] = {
            0,1,2,3,
                0,2,   // looks like 1,3,0,2
            4,5,6,
                0,3,6, // looks like 1,2,4,5,0,3,6
            7,8,9,10,11,
                1,2, // recreated, looks like 0,3,6,7,8,9,10,11,1,2
            12,13,14,15,16,17,18,19,
                0 // recreated, looks like 2,12,13,14,15,16,17,18,19,0
        };
        for (int i = 0; i < requests.length; ++i)
        {
            hc.get( new Integer( requests[i] ) );
            //hc.printChain();
        }
        assertEquals(
            "The mgr was not called for the creation correctly.",
            23,
            om.getCreationCount() );
        assertEquals(
            "The mgr was not called for the clean-up.",
            13,
            om.getCloseCount() );
        assertEquals(
            "The overflow count is wrong.",
            13,
            hc.getOverflowCount() );
        assertEquals(
            "The size is wrong.",
            10,
            hc.getSize() );
    }
    
    
    public void testResetOverflowCount1()
    {
        System.out.println("Get4:");
        OMgr om = new OMgr();
        HashCache hc = new HashCache( om, 10 );
        hc.resetOverflowCount();
    }
}
