//===----------------------------------------------------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//

// <chrono>

// time_point

// template <class ToDuration, class Clock, class Duration>
//   time_point<Clock, ToDuration>
//   time_point_cast(const time_point<Clock, Duration>& t);

#include <chrono>
#include <cassert>
#include <ratio>
#include <type_traits>

#include "test_macros.h"

template <class T, class = void>
struct has_time_point_cast : std::false_type {};

template <class T>
struct has_time_point_cast<T, decltype((void)std::chrono::time_point_cast<T>(std::chrono::system_clock::now()))>
    : std::true_type {};

static_assert(has_time_point_cast<std::chrono::seconds>::value, "");
static_assert(!has_time_point_cast<int>::value, "");

template <class FromDuration, class ToDuration>
void
test(const FromDuration& df, const ToDuration& d)
{
    typedef std::chrono::system_clock Clock;
    typedef std::chrono::time_point<Clock, FromDuration> FromTimePoint;
    typedef std::chrono::time_point<Clock, ToDuration> ToTimePoint;
    {
    FromTimePoint f(df);
    ToTimePoint t(d);
    typedef decltype(std::chrono::time_point_cast<ToDuration>(f)) R;
    static_assert((std::is_same<R, ToTimePoint>::value), "");
    assert(std::chrono::time_point_cast<ToDuration>(f) == t);
    }
}

#if TEST_STD_VER > 11

template<class FromDuration, long long From, class ToDuration, long long To>
void test_constexpr ()
{
    typedef std::chrono::system_clock Clock;
    typedef std::chrono::time_point<Clock, FromDuration> FromTimePoint;
    typedef std::chrono::time_point<Clock, ToDuration> ToTimePoint;
    {
    constexpr FromTimePoint f{FromDuration{From}};
    constexpr ToTimePoint   t{ToDuration{To}};
    static_assert(std::chrono::time_point_cast<ToDuration>(f) == t, "");
    }

}

#endif

int main(int, char**)
{
    test(std::chrono::milliseconds(7265000), std::chrono::hours(2));
    test(std::chrono::milliseconds(7265000), std::chrono::minutes(121));
    test(std::chrono::milliseconds(7265000), std::chrono::seconds(7265));
    test(std::chrono::milliseconds(7265000), std::chrono::milliseconds(7265000));
    test(std::chrono::milliseconds(7265000), std::chrono::microseconds(7265000000LL));
    test(std::chrono::milliseconds(7265000), std::chrono::nanoseconds(7265000000000LL));
    test(std::chrono::milliseconds(7265000),
         std::chrono::duration<double, std::ratio<3600> >(7265./3600));
    test(std::chrono::duration<int, std::ratio<2, 3> >(9),
         std::chrono::duration<int, std::ratio<3, 5> >(10));
#if TEST_STD_VER > 11
    {
    test_constexpr<std::chrono::milliseconds, 7265000, std::chrono::hours,    2> ();
    test_constexpr<std::chrono::milliseconds, 7265000, std::chrono::minutes,121> ();
    test_constexpr<std::chrono::milliseconds, 7265000, std::chrono::seconds,7265> ();
    test_constexpr<std::chrono::milliseconds, 7265000, std::chrono::milliseconds,7265000> ();
    test_constexpr<std::chrono::milliseconds, 7265000, std::chrono::microseconds,7265000000LL> ();
    test_constexpr<std::chrono::milliseconds, 7265000, std::chrono::nanoseconds,7265000000000LL> ();
    typedef std::chrono::duration<int, std::ratio<3, 5>> T1;
    test_constexpr<std::chrono::duration<int, std::ratio<2, 3>>, 9, T1, 10> ();
    }
#endif

  return 0;
}
