/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <sys/types.h>
#include <time.h>

#include <gssapi.h>

#include "llgt_config.h"

#include "llgt_globus_internal.h"

#include "llgt_utils.h"

static int llgt_debugging_mode = 0; /* disabled by default */

static int llgt_log_to_file = -1; /* uninitialized */

static FILE *llgt_logfile=NULL;	    /* FILE used for logging (when llgt_log_to_file == 1) */
static char *llgt_log_ident=NULL;   /* log ident used when logging to file */

char *llgt_loglevel[]={
    "LOG_EMERG",
    "LOG_ALERT",
    "LOG_CRIT",
    "LOG_ERR",
    "LOG_WARNING",
    "LOG_NOTICE",
    "LOG_INFO",
    "LOG_DEBUG"};

static int llgt_setup_syslog(void);

static int llgt_setup_syslog(void)
{
    int   syslog_facility       = LOG_DAEMON;
    char *syslog_log_prefix     = NULL;
    char *llgt_log_facility     = NULL;

    /* Enable debugging */
    if (getenv("LLGT_ENABLE_DEBUG")) {
        llgt_enable_debugging_mode();
    }


    /* Check for a specific Syslog IDENT */
    if ((syslog_log_prefix = getenv("LLGT_LOG_IDENT"))) {
        /* If the string length is zero, then nothing is really specified and
         * then we just NULLify the string */
        if (strlen(syslog_log_prefix) == 0) {
            syslog_log_prefix = NULL;
        }
    }

    /* Set an alternative SysLog facility */
    if ((llgt_log_facility = getenv("LLGT_LOG_FACILITY"))) {
        if      (strcmp("LOG_DAEMON",   llgt_log_facility) == 0) { syslog_facility = LOG_DAEMON; }
        else if (strcmp("LOG_AUTH",     llgt_log_facility) == 0) { syslog_facility = LOG_AUTH; }
        else if (strcmp("LOG_CRON",     llgt_log_facility) == 0) { syslog_facility = LOG_CRON; }
        else if (strcmp("LOG_FTP",      llgt_log_facility) == 0) { syslog_facility = LOG_FTP; }
        else if (strcmp("LOG_KERN",     llgt_log_facility) == 0) { syslog_facility = LOG_KERN; }
        else if (strcmp("LOG_LPR",      llgt_log_facility) == 0) { syslog_facility = LOG_LPR; }
        else if (strcmp("LOG_MAIL",     llgt_log_facility) == 0) { syslog_facility = LOG_MAIL; }
        else if (strcmp("LOG_NEWS",     llgt_log_facility) == 0) { syslog_facility = LOG_NEWS; }
        else if (strcmp("LOG_SYSLOG",   llgt_log_facility) == 0) { syslog_facility = LOG_SYSLOG; }
        else if (strcmp("LOG_USER",     llgt_log_facility) == 0) { syslog_facility = LOG_USER; }
        else if (strcmp("LOG_UUCP",     llgt_log_facility) == 0) { syslog_facility = LOG_UUCP; }

        else if (strcmp("LOG_LOCAL0", llgt_log_facility) == 0) { syslog_facility = LOG_LOCAL0; }
        else if (strcmp("LOG_LOCAL1", llgt_log_facility) == 0) { syslog_facility = LOG_LOCAL1; }
        else if (strcmp("LOG_LOCAL2", llgt_log_facility) == 0) { syslog_facility = LOG_LOCAL2; }
        else if (strcmp("LOG_LOCAL3", llgt_log_facility) == 0) { syslog_facility = LOG_LOCAL3; }
        else if (strcmp("LOG_LOCAL4", llgt_log_facility) == 0) { syslog_facility = LOG_LOCAL4; }
        else if (strcmp("LOG_LOCAL5", llgt_log_facility) == 0) { syslog_facility = LOG_LOCAL5; }
        else if (strcmp("LOG_LOCAL6", llgt_log_facility) == 0) { syslog_facility = LOG_LOCAL6; }
        else if (strcmp("LOG_LOCAL7", llgt_log_facility) == 0) { syslog_facility = LOG_LOCAL7; }
        else {
            syslog(LOG_ERR, "The Syslog facility is configured with $LLGT_LOG_FACILITY and set to the unknown facility: \"%s\". Overriding to LOG_DAEMON. Please fix the setting.", llgt_log_facility);
        }
    } else {
        /* Default - Nothing specified */
        syslog_facility = LOG_DAEMON;
    }

    /* When either LLGT_LOG_FACILITY or LLGT_LOG_IDENT is given, call
     * openlog(), otherwise, skip it */
    if (llgt_log_facility || syslog_log_prefix)	{
	/* initialize Syslog */
	openlog (syslog_log_prefix, LOG_CONS | LOG_PID | LOG_NDELAY,
		 syslog_facility);
    }

    return 0;
}

void llgt_enable_debugging_mode(void) {
    llgt_debugging_mode = 1;
}

int llgt_is_debugmode_enabled(void) {
    return llgt_debugging_mode;
}


/* Opens log, eeither syslog or logfile */
void llgt_open_log(void)	{
    char *llgt_logfilename=getenv("LLGT_LOG_FILE");
    int errno_fopen;

    /* Only continue when we aren't initialized */
    if (llgt_log_to_file>=0)
	return;

    /* Do we have a LLGT_LOG_FILE? */
    if (llgt_logfilename==NULL) {	/* log to syslog */
	llgt_log_to_file=0;
	llgt_setup_syslog();
    } else { /* Log to file */
	llgt_logfile=fopen(llgt_logfilename, "a");
	if (llgt_logfile)  { /* Succesfully opened logfile */
	    llgt_log_to_file=1;
	    if ( (llgt_log_ident=getenv("LLGT_LOG_IDENT")) == NULL)
		/* Default log ident */
		llgt_log_ident="llgt";

	    /* Make sure LCAS and LCMAPS use the same logfile if no default has
	     * been set for them. */
	    if (getenv("LCMAPS_LOG_FILE")==NULL)
		setenv("LCMAPS_LOG_FILE", llgt_logfilename, 1);
#ifndef DISABLE_LCAS
	    if (getenv("LCAS_LOG_FILE")==NULL)
		setenv("LCAS_LOG_FILE", llgt_logfilename, 1);
#endif
	} else {/* Switch to syslog and log error */
	    errno_fopen=errno;
	    llgt_log_to_file=0;
	    llgt_setup_syslog();
	    llgt_logmsg(LOG_ERR,"Cannot open %s, using syslog: %s\n",
		        llgt_logfilename,strerror(errno_fopen));
	}
    }
}

/* Close the logfile when opened */
void llgt_close_log(void)   {
    if (llgt_logfile)	{
	fclose(llgt_logfile);
	llgt_logfile=NULL;
	llgt_log_to_file=-1;
    }
}

/* Log function for lcas-lcmaps-gt4-interface messages, logs to either file or
 * syslog. When needed the log will be opened first via llgt_open_log() */
void llgt_logmsg (int priority, const char* format, ...)
{
    va_list     args;
    char	buf[MAX_ERRORBUF_LEN];
    char	datetime[21];
    int		res,i;
    time_t      mclock;
    struct tm * tmp;


    /* Check if the module was enabled with debugging enabled.
     * If so, pass the LOG_DEBUG level message though */
    if (priority == LOG_DEBUG && !llgt_is_debugmode_enabled()) {
        /* Skip debugging messages */
        return;
    }

    /* check whether log destination is initialized */
    if (llgt_log_to_file<0)
	llgt_open_log();

    /* Put logstring in buffer */
    va_start(args, format);
    res=vsnprintf(buf,MAX_ERRORBUF_LEN,format,args);
    va_end(args);

    /* Handle too large msg */
    if (res>0 && (size_t)res>= MAX_ERRORBUF_LEN)	{
	sprintf(&(buf[MAX_ERRORBUF_LEN-5]),"...\n");
	res=MAX_ERRORBUF_LEN-1;
    }

    /* Replace unprintable characters with ? */
    for (i=0; buf[i]!='\0'; i++)    {
	if (buf[i]!='\n' && !isprint(buf[i]))
	    buf[i]='?';
    }
    /* Guarantee a newline at the end of the logstring */
    buf[res-1]='\n';

    if (llgt_log_to_file==0)
	/* Log to syslog */
	syslog(priority, "%s", buf);
    else {
	/* Log to file: get time */
	time(&mclock);
	/* Log in GMT, no exceptions */
	if ( (tmp = gmtime(&mclock))==NULL )
	    datetime[0]='\0';
	else
	    snprintf(datetime, (size_t)21, "%04d-%02d-%02d.%02d:%02d:%02dZ",
			      tmp->tm_year + 1900, tmp->tm_mon + 1,
			      tmp->tm_mday,
			       tmp->tm_hour, tmp->tm_min, tmp->tm_sec);
	/* print in logfile */
	fprintf(llgt_logfile,"%s[%d]: %11s: %s: %s",
		llgt_log_ident, (int)getpid(),
		llgt_loglevel[priority], datetime, buf);
    }
}



char* llgt_get_client_name(gss_ctx_id_t context)
{
	int initiator;
	unsigned int major_status;
	unsigned int minor_status;
	gss_name_t peer;
	gss_buffer_desc peer_name_buffer;
	char *client_name = NULL;

	major_status = gss_inquire_context(&minor_status,
                                       context,
                                       GLOBUS_NULL,
                                       GLOBUS_NULL,
                                       GLOBUS_NULL,
                                       GLOBUS_NULL,
                                       GLOBUS_NULL,
                                       &initiator,
                                       GLOBUS_NULL);
    if(GSS_ERROR(major_status))
    {
        /* GLOBUS_GRIDMAP_CALLOUT_GSS_ERROR(result, major_status, minor_status); */
        /* goto error; */
    	return NULL;
    }

	major_status = gss_inquire_context(&minor_status,
                                       context,
                                       initiator ? GLOBUS_NULL : &peer,
                                       initiator ? &peer : GLOBUS_NULL,
                                       GLOBUS_NULL,
                                       GLOBUS_NULL,
                                       GLOBUS_NULL,
                                       GLOBUS_NULL,
                                       GLOBUS_NULL);
    if(GSS_ERROR(major_status))
    {
		return NULL;
    }
    major_status = gss_display_name(&minor_status,
                                    peer,
                                    &peer_name_buffer,
                                    GLOBUS_NULL);
    if(GSS_ERROR(major_status))
    {
        /* GLOBUS_GRIDMAP_CALLOUT_GSS_ERROR(result, major_status, minor_status); */
        gss_release_name(&minor_status, &peer);
		return NULL;
    }
    gss_release_name(&minor_status, &peer);
	if((client_name = (char*)malloc(peer_name_buffer.length + 1)))
	{
		memcpy(client_name, peer_name_buffer.value, peer_name_buffer.length);
		client_name[peer_name_buffer.length] = 0;
	}
	gss_release_buffer(&minor_status, &peer_name_buffer);
	return client_name;
}



gss_cred_id_t llgt_get_user_cred_handle(gss_ctx_id_t context_handle)
{
    llgt_gss_ctx_id_desc *local_handle=(llgt_gss_ctx_id_desc*)context_handle;
    return local_handle->peer_cred_handle;
}



int llgt_create_jobid(void)
{
    /*
     * Create the unique Job Repository ID and export it as an environment variable
     * the ID is the same one as for the Job Manager with the addition of microseconds.
     * We moved this part here so that the LCMAPS Job Repository plugin
     * can use this environment variable to uniquely identify the job
     * To have a really unique id we also add the microseconds to the time string
     */
    struct tm  * jr_tmp;
    const char * jr_id_var = "JOB_REPOSITORY_ID";
    char         jr_id[71];
    const char * gk_jm_id_var = "GATEKEEPER_JM_ID";
    char         gatekeeper_jm_id[64];
    const pid_t  gatekeeper_pid = getpid();
    static unsigned int   reqnr = 0;

#ifdef HAVE_GETTIMEOFDAY
    {
    struct timeval  jr_clock;

    /* get time in microseconds ! */
    gettimeofday(&jr_clock,NULL);
    jr_tmp = gmtime(&(jr_clock.tv_sec));

    snprintf(jr_id, (size_t)70, "%04d-%02d-%02d.%02d:%02d:%02d.%06d.%010u.%010u",
        jr_tmp->tm_year + 1900, jr_tmp->tm_mon + 1, jr_tmp->tm_mday,
        jr_tmp->tm_hour, jr_tmp->tm_min, jr_tmp->tm_sec, (int)(jr_clock.tv_usec),
        (unsigned)gatekeeper_pid & 0xFFFFFFFF, reqnr & 0xFFFFFFFF);
    }
#else
    {
    time_t       myclock;

    time(&myclock);
    jr_tmp = gmtime(&myclock);
    }

    snprintf(jr_id, (size_t)70, "%04d-%02d-%02d.%02d:%02d:%02d.%010u.%010u",
        jr_tmp->tm_year + 1900, jr_tmp->tm_mon + 1, jr_tmp->tm_mday,
        jr_tmp->tm_hour, jr_tmp->tm_min, jr_tmp->tm_sec,
        (unsigned)gatekeeper_pid & 0xFFFFFFFF, reqnr & 0xFFFFFFFF);
#endif
    snprintf(gatekeeper_jm_id, (size_t)63, "%04d-%02d-%02d.%02d:%02d:%02d.%010u.%010u",
        jr_tmp->tm_year + 1900, jr_tmp->tm_mon + 1, jr_tmp->tm_mday,
        jr_tmp->tm_hour, jr_tmp->tm_min, jr_tmp->tm_sec,
        (unsigned)gatekeeper_pid & 0xFFFFFFFF, reqnr & 0xFFFFFFFF);


    setenv(jr_id_var, jr_id, 1);
    setenv(gk_jm_id_var, gatekeeper_jm_id, 1);

    reqnr++;
    return 0;
}
