type ColorSpace
    CIEprimaries::Matrix
    whitepoint::Vector
    rgbxyz::Matrix
    rgbxyzScaled::Matrix
end

function colorspace(xr, yr, xg, yg, xb, yb, xw, yw)
    zr = 1 - (xr+yr)
    zg = 1 - (xg+yg)
    zb = 1 - (xb+yb)
    zw = 1 - (xw+yw)

    rgbxyz = [(yg*zb)-(yb*zg) (xb*zg)-(xg*zb) (xg*yb)-(xb*yg);
              (yb*zr)-(yr*zb) (xr*zb)-(xb*zr) (xb*yr)-(xr*yb);
              (yr*zg)-(yg*zr) (xg*zr)-(xr*zg) (xr*yg)-(xg*yr)]
    mat = [xr yr zr;
           xg yg zg;
           xb yb zb]
    whi = [xw; yw; zw]

    rgbw = rgbxyz*whi ./ yw

    rgbxyzsc = [(rgbxyz[1,:] ./ rgbw[1])';
                (rgbxyz[2,:] ./ rgbw[2])';
                (rgbxyz[3,:] ./ rgbw[3])']

    return ColorSpace(mat, whi, rgbxyz, rgbxyzsc)
end

CIEsystem = colorspace(0.7355, 0.2645, 0.2658, 0.7243, 0.1669, 0.0085,
                       0.3333, 0.3333)

sRGBsystem = colorspace(.64, .33, .3, .6, .15, .06, .3127, .3290)

rgb(r, g, b) = int32(65535*clamp(r,0,255) +
                     256  *clamp(g,0,255) +
                           clamp(b,0,255))
rgbf(r, g, b) = rgb(255*r, 255*g, 255*b)

function vec2pixel(v)
    w = -min(0, v[1], v[2], v[3])
    if w > 0
        rgbf(v[1]+w, v[2]+w, v[3]+w)
    else
        rgbf(v[1], v[2], v[3])
    end
end

typealias Pixel Int32

redval(p) = div(p,65536)
greenval(p) = div(p,256)&0xff
blueval(p) = p&0xff

pixel2vec(p) = [redval(p); greenval(p); blueval(p)]./255

xyz2rgb(sys::ColorSpace, xyz) = sys.rgbxyzScaled * xyz

rgb2yiq(rgb) = [.299  .587  .114;
                .596 -.275 -.321;
                .212 -.523  .311] * rgb

yiq2rgb(yiq) = [1   .956  .621;
                1  -.272 -.647;
                1 -1.105 1.702] * yiq

rgb2gray(rgb::Vector) = 0.3*v[1] + 0.59*v[2] + 0.11*v[3]
rgb2gray(p::Pixel) =
    (y = 0.3*redval(p) + 0.59*greenval(p) + 0.11*blueval(p);
     rgb(y,y,y))

const cie_color_match =
    [0.001368 3.8999999999999999e-005 0.006450001;
     0.00150205 4.2826400000000000e-005 0.007083216;
     0.001642328 4.6914600000000000e-005 0.007745488;
     0.001802382 5.1589599999999998e-005 0.008501152;
     0.001995757 5.7176399999999997e-005 0.009414544;
     0.002236 6.3999999999999997e-005 0.01054999;
     0.002535385 7.2344209999999998e-005 0.0119658;
     0.002892603 8.2212239999999995e-005 0.01365587;
     0.003300829 9.3508159999999998e-005 0.01558805;
     0.003753236 1.0613610000000000e-004 0.01773015;
     0.004243 1.2000000000000000e-004 0.02005001;
     0.004762389 0.000134984 0.02251136;
     0.005330048 0.000151492 0.02520288;
     0.005978712 0.000170208 0.02827972;
     0.006741117 0.000191816 0.03189704;
     0.00765 0.000217 0.03621;
     0.008751373 0.0002469067 0.04143771;
     0.01002888 0.00028124 0.04750372;
     0.0114217 0.00031852 0.05411988;
     0.01286901 0.0003572667 0.06099803;
     0.01431 0.000396 0.06785001;
     0.01570443 0.0004337147 0.07448632;
     0.01714744 0.000473024 0.08136156;
     0.01878122 0.000517876 0.08915364;
     0.02074801 0.0005722187 0.09854048;
     0.02319 0.00064 0.1102;
     0.02620736 0.00072456 0.1246133;
     0.02978248 0.0008255 0.1417017;
     0.03388092 0.00094116 0.1613035;
     0.03846824 0.00106988 0.1832568;
     0.04351 0.00121 0.2074;
     0.0489956 0.001362091 0.2336921;
     0.0550226 0.001530752 0.2626114;
     0.0617188 0.001720368 0.2947746;
     0.069212 0.001935323 0.3307985;
     0.07763 0.00218 0.3713;
     0.08695811 0.0024548 0.4162091;
     0.09717672 0.002764 0.4654642;
     0.1084063 0.0031178 0.5196948;
     0.1207672 0.0035264 0.5795303000000001;
     0.13438 0.004 0.6456;
     0.1493582 0.00454624 0.7184838;
     0.1653957 0.00515932 0.7967133;
     0.1819831 0.00582928 0.8778459;
     0.198611 0.00654616 0.959439;
     0.21477 0.0073 1.0390501000000001;
     0.2301868 0.008086507 1.1153673;
     0.2448797 0.00890872 1.1884971;
     0.2587773 0.00976768 1.2581233000000001;
     0.2718079 0.01066443 1.3239296;
     0.2839 0.0116 1.3855999999999999;
     0.2949438 0.01257317 1.4426352;
     0.3048965 0.01358272 1.4948035;
     0.3137873 0.01462968 1.5421902999999999;
     0.3216454 0.01571509 1.5848807;
     0.3285 0.01684 1.62296;
     0.3343513 0.01800736 1.6564048;
     0.3392101 0.01921448 1.6852959000000001;
     0.3431213 0.02045392 1.7098745;
     0.3461296 0.02171824 1.7303820999999999;
     0.34828 0.023 1.7470600000000001;
     0.3495999 0.02429461 1.7600446000000001;
     0.3501474 0.02561024 1.7696232999999999;
     0.350013 0.02695857 1.7762636999999999;
     0.349287 0.02835125 1.7804333999999999;
     0.34806 0.0298 1.7826;
     0.3463733 0.03131083 1.7829682;
     0.3442624 0.03288368 1.7816997999999999;
     0.3418088 0.03452112 1.7791982;
     0.3390941 0.03622571 1.7758670999999999;
     0.3362 0.038 1.7721100000000001;
     0.3331977 0.03984667 1.7682589;
     0.3300411 0.041768 1.7640389999999999;
     0.3266357 0.043766 1.7589437999999999;
     0.3228868 0.04584267 1.7524663;
     0.3187 0.048 1.7441;
     0.3140251 0.05024368 1.7335594999999999;
     0.308884 0.05257304 1.7208581000000001;
     0.3032904 0.05498056 1.7059369;
     0.2972579 0.05745872 1.6887372;
     0.2908 0.06 1.6692;
     0.2839701 0.06260197 1.6475287000000001;
     0.2767214 0.06527752 1.6234127;
     0.2689178 0.06804208 1.5960223;
     0.2604227 0.07091109 1.5645279999999999;
     0.2511 0.0739 1.5281;
     0.2408475 0.077016 1.4861114;
     0.2298512 0.0802664 1.4395214999999999;
     0.2184072 0.0836668 1.3898798999999999;
     0.2068115 0.0872328 1.3387362;
     0.19536 0.09098 1.2876399999999999;
     0.1842136 0.09491755 1.2374223;
     0.1733273 0.09904584 1.1878242999999999;
     0.1626881 0.1033674 1.1387611;
     0.1522833 0.1078846 1.0901479999999999;
     0.1421 0.1126 1.0419;
     0.1321786 0.117532 0.9941976;
     0.1225696 0.1226744 0.9473473;
     0.1132752 0.1279928 0.9014531;
     0.1042979 0.1334528 0.8566193;
     0.09564 0.13902 0.8129501;
     0.08729955 0.1446764 0.7705173;
     0.07930804 0.1504693 0.7294448;
     0.07171776 0.1564619 0.6899136;
     0.06458099 0.1627177 0.6521049;
     0.05795001 0.1693 0.6162;
     0.05186211 0.1762431 0.5823286;
     0.04628152 0.1835581 0.5504162;
     0.04115088 0.1912735 0.5203376;
     0.03641283 0.199418 0.4919673;
     0.03201 0.20802 0.46518;
     0.0279172 0.2171199 0.4399246;
     0.0241444 0.2267345 0.4161836;
     0.020687 0.2368571 0.3938822;
     0.0175404 0.2474812 0.3729459;
     0.0147 0.2586 0.3533;
     0.01216179 0.2701849 0.3348578;
     0.00991996 0.2822939 0.3175521;
     0.00796724 0.2950505 0.3013375;
     0.006296346 0.308578 0.2861686;
     0.0049 0.323 0.272;
     0.003777173 0.3384021 0.2588171;
     0.00294532 0.3546858 0.2464838;
     0.00242488 0.3716986 0.2347718;
     0.002236293 0.3892875 0.2234533;
     0.0024 0.4073 0.2123;
     0.00292552 0.4256299 0.2011692;
     0.00383656 0.4443096 0.1901196;
     0.00517484 0.4633944 0.1792254;
     0.00698208 0.4829395 0.1685608;
     0.0093 0.503 0.1582;
     0.01214949 0.5235693 0.1481383;
     0.01553588 0.544512 0.1383758;
     0.01947752 0.56569 0.1289942;
     0.02399277 0.5869653 0.1200751;
     0.0291 0.6082 0.1117;
     0.03481485 0.6293456 0.1039048;
     0.04112016 0.6503068 0.09666748;
     0.04798504 0.6708752 0.08998272;
     0.05537861 0.6908424 0.08384531;
     0.06327 0.71 0.07824999;
     0.07163501 0.7281852 0.07320899;
     0.08046224 0.7454636 0.06867816;
     0.08973996 0.7619694 0.06456784;
     0.09945645 0.7778368 0.06078835;
     0.1096 0.7932 0.05725001;
     0.1201674 0.8081104 0.05390435;
     0.1311145 0.8224962 0.05074664;
     0.1423679 0.8363068 0.04775276;
     0.1538542 0.8494916 0.04489859;
     0.1655 0.862 0.04216;
     0.1772571 0.8738108 0.03950728;
     0.18914 0.8849624 0.03693564;
     0.2011694 0.8954936 0.03445836;
     0.2133658 0.9054432 0.03208872;
     0.2257499 0.9148501 0.02984;
     0.2383209 0.9237348 0.02771181;
     0.2510668 0.9320924 0.02569444;
     0.2639922 0.9399226000000001 0.02378716;
     0.2771017 0.9472252 0.02198925;
     0.2904 0.954 0.0203;
     0.3038912 0.9602561 0.01871805;
     0.3175726 0.9660074 0.01724036;
     0.3314384 0.9712606 0.01586364;
     0.3454828 0.9760225 0.01458461;
     0.3597 0.9803 0.0134;
     0.3740839 0.9840924 0.01230723;
     0.3886396 0.9874182 0.01130188;
     0.4033784 0.9903128 0.01037792;
     0.4183115 0.9928116 0.009529306;
     0.4334499 0.9949501 0.008749999;
     0.4487953 0.9967108 0.0080352;
     0.464336 0.9980983 0.0073816;
     0.480064 0.999112 0.0067854;
     0.4959713 0.9997482 0.0062428;
     0.5120501 1.0 0.005749999;
     0.5282959 0.9998567 0.0053036;
     0.5446916000000001 0.9993046 0.0048998;
     0.5612094 0.9983255 0.0045342;
     0.5778215 0.9968987 0.0042024;
     0.5945 0.995 0.0039;
     0.6112209 0.9926005 0.0036232;
     0.6279758 0.9897426 0.0033706;
     0.6447602 0.9864444 0.0031414;
     0.6615697 0.9827241 0.0029348;
     0.6784 0.9786 0.002749999;
     0.6952392 0.9740837 0.0025852;
     0.7120586 0.9691712 0.0024386;
     0.7288284 0.9638568 0.0023094;
     0.7455188 0.9581349 0.0021968;
     0.7621 0.952 0.0021;
     0.7785432 0.9454504 0.002017733;
     0.7948256 0.9384992 0.0019482;
     0.8109264000000001 0.9311628 0.0018898;
     0.8268248 0.9234576 0.001840933;
     0.8425 0.9154 0.0018;
     0.8579325 0.9070064 0.001766267;
     0.8730816 0.8982772 0.0017378;
     0.8878944 0.8892048 0.0017112;
     0.9023181 0.8797816000000001 0.001683067;
     0.9163 0.87 0.001650001;
     0.9297995 0.8598613000000001 0.001610133;
     0.9427984 0.849392 0.0015644;
     0.9552776 0.838622 0.0015136;
     0.9672179 0.8275813 0.001458533;
     0.9786 0.8163 0.0014;
     0.9893856 0.8047947 0.001336667;
     0.9995488 0.793082 0.00127;
     1.0090892 0.781192 0.001205;
     1.0180064 0.7691547 0.001146667;
     1.0263 0.757 0.0011;
     1.0339826999999999 0.7447541 0.0010688;
     1.040986 0.7324224 0.0010494;
     1.047188 0.7200036 0.0010356;
     1.0524667000000001 0.7074965 0.0010212;
     1.0567 0.6949 0.001;
     1.0597943999999999 0.6822192 0.00096864;
     1.0617992000000001 0.6694715999999999 0.00092992;
     1.0628067999999999 0.6566744 0.00088688;
     1.0629096 0.6438448 0.00084256;
     1.0622 0.631 0.0008;
     1.0607352000000001 0.6181555 0.00076096;
     1.0584435999999999 0.6053144 0.00072368;
     1.0552244 0.5924756 0.00068592;
     1.0509767999999999 0.5796379 0.00064544;
     1.0456000000000001 0.5668 0.0006;
     1.0390368999999999 0.5539611 0.0005478667;
     1.0313608000000001 0.5411372 0.0004916;
     1.0226662 0.5283528 0.0004354;
     1.0130477 0.5156323 0.0003834667;
     1.0025999999999999 0.503 0.00034;
     0.9913675 0.4904688 0.0003072533;
     0.9793314 0.4780304 0.00028316;
     0.9664916 0.4656776 0.00026544;
     0.9528479 0.4534032 0.0002518133;
     0.9384 0.4412 0.00024;
     0.923194 0.42908 0.0002295467;
     0.9072440000000001 0.417036 0.00022064;
     0.890502 0.405032 0.00021196;
     0.87292 0.393032 0.0002021867;
     0.8544499 0.381 0.00019;
     0.8350840000000001 0.3689184 0.0001742133;
     0.814946 0.3568272 0.00015564;
     0.794186 0.3447768 0.00013596;
     0.772954 0.3328176 1.1685329999999999e-004;
     0.7514 0.321 1e-004;
     0.7295836 0.3093381 8.6133330000000006e-005;
     0.7075888 0.2978504 7.4599999999999997e-005;
     0.6856022000000001 0.2865936 6.4999999999999994e-005;
     0.6638104 0.2756245 5.6933330000000001e-005;
     0.6424 0.265 4.9999990000000002e-005;
     0.6215149 0.2547632 4.4159999999999997e-005;
     0.6011138 0.2448896 3.9480000000000001e-005;
     0.5811052 0.2353344 3.5719999999999997e-005;
     0.5613977 0.2260528 3.2639999999999999e-005;
     0.5419000000000001 0.217 3.0000000000000001e-005;
     0.5225995 0.2081616 2.7653330000000000e-005;
     0.5035464 0.1995488 2.5559999999999999e-005;
     0.4847436 0.1911552 2.3640000000000001e-005;
     0.4661939 0.1829744 2.1813329999999999e-005;
     0.4479 0.175 2.0000000000000002e-005;
     0.4298613 0.1672235 1.8133330000000000e-005;
     0.412098 0.1596464 1.6200000000000001e-005;
     0.394644 0.1522776 1.4200000000000000e-005;
     0.3775333 0.1451259 1.2133330000000000e-005;
     0.3608 0.1382 1e-005;
     0.3444563 0.1315003 7.7333329999999996e-006;
     0.3285168 0.1250248 5.4000000000000000e-006;
     0.3130192 0.1187792 3.1999999999999999e-006;
     0.2980011 0.1127691 1.3333330000000001e-006;
     0.2835 0.107 0.0;
     0.2695448 0.1014762 0.0;
     0.2561184 0.09618864 0.0;
     0.2431896 0.09112296 0.0;
     0.2307272 0.08626485 0.0;
     0.2187 0.0816 0.0;
     0.2070971 0.07712064 0.0;
     0.1959232 0.07282552 0.0;
     0.1851708 0.06871008 0.0;
     0.1748323 0.06476976 0.0;
     0.1649 0.061 0.0;
     0.1553667 0.05739621 0.0;
     0.14623 0.05395504 0.0;
     0.13749 0.05067376 0.0;
     0.1291467 0.04754965 0.0;
     0.1212 0.04458 0.0;
     0.1136397 0.04175872 0.0;
     0.106465 0.03908496 0.0;
     0.09969044 0.03656384 0.0;
     0.09333061 0.03420048 0.0;
     0.0874 0.032 0.0;
     0.08190096 0.02996261 0.0;
     0.07680428 0.02807664 0.0;
     0.07207712 0.02632936 0.0;
     0.06768664 0.02470805 0.0;
     0.0636 0.0232 0.0;
     0.05980685 0.02180077 0.0;
     0.05628216 0.02050112 0.0;
     0.05297104 0.01928108 0.0;
     0.04981861 0.01812069 0.0;
     0.04677 0.017 0.0;
     0.04378405 0.01590379 0.0;
     0.04087536 0.01483718 0.0;
     0.03807264 0.01381068 0.0;
     0.03540461 0.01283478 0.0;
     0.0329 0.01192 0.0;
     0.03056419 0.01106831 0.0;
     0.02838056 0.01027339 0.0;
     0.02634484 0.009533311 0.0;
     0.02445275 0.008846157 0.0;
     0.0227 0.00821 0.0;
     0.02108429 0.007623781 0.0;
     0.01959988 0.007085424 0.0;
     0.01823732 0.006591476 0.0;
     0.01698717 0.006138485 0.0;
     0.01584 0.005723 0.0;
     0.01479064 0.005343059 0.0;
     0.01383132 0.004995796 0.0;
     0.01294868 0.004676404 0.0;
     0.0121292 0.004380075 0.0;
     0.01135916 0.004102 0.0;
     0.01062935 0.003838453 0.0;
     0.009938846 0.003589099 0.0;
     0.009288422 0.003354219 0.0;
     0.008678854 0.003134093 0.0;
     0.008110916 0.002929 0.0;
     0.007582388 0.002738139 0.0;
     0.007088746 0.002559876 0.0;
     0.006627313 0.002393244 0.0;
     0.006195408 0.002237275 0.0;
     0.005790346 0.002091 0.0;
     0.005409826 0.001953587 0.0;
     0.005052583 0.00182458 0.0;
     0.004717512 0.00170358 0.0;
     0.004403507 0.001590187 0.0;
     0.004109457 0.001484 0.0;
     0.003833913 0.001384496 0.0;
     0.003575748 0.001291268 0.0;
     0.003334342 0.001204092 0.0;
     0.003109075 0.001122744 0.0;
     0.002899327 0.001047 0.0;
     0.002704348 0.0009765896 0.0;
     0.00252302 0.0009111088 0.0;
     0.002354168 0.0008501332 0.0;
     0.002196616 0.0007932384 0.0;
     0.00204919 0.00074 0.0;
     0.00191096 0.0006900827 0.0;
     0.001781438 0.00064331 0.0;
     0.00166011 0.000599496 0.0;
     0.001546459 0.0005584547 0.0;
     0.001439971 0.00052 0.0;
     0.001340042 0.0004839136 0.0;
     0.001246275 0.0004500528 0.0;
     0.001158471 0.0004183452 0.0;
     0.00107643 0.0003887184 0.0;
     0.0009999493 0.0003611 0.0;
     0.0009287358 0.0003353835 0.0;
     0.0008624332 0.0003114404 0.0;
     0.0008007503 0.0002891656 0.0;
     0.000743396 0.0002684539 0.0;
     0.0006900786 0.0002492 0.0;
     0.0006405156 0.0002313019 0.0;
     0.0005945021 0.0002146856 0.0;
     0.0005518646 0.0001992884 0.0;
     0.000512429 0.0001850475 0.0;
     0.0004760213 0.0001719 0.0;
     0.0004424536 0.0001597781 0.0;
     0.0004115117 0.0001486044 0.0;
     0.0003829814 0.0001383016 0.0;
     0.0003566491 0.0001287925 0.0;
     0.0003323011 1.2000000000000000e-004 0.0;
     0.0003097586 1.1185950000000000e-004 0.0;
     0.0002888871 1.0432240000000000e-004 0.0;
     0.0002695394 9.7335600000000004e-005 0.0;
     0.0002515682 9.0845869999999995e-005 0.0;
     0.0002348261 8.4800000000000001e-005 0.0;
     0.000219171 7.9146669999999997e-005 0.0;
     0.0002045258 7.3857999999999997e-005 0.0;
     0.0001908405 6.8916000000000002e-005 0.0;
     0.0001780654 6.4302670000000001e-005 0.0;
     0.0001661505 6.0000000000000002e-005 0.0;
     0.0001550236 5.5981869999999998e-005 0.0;
     0.0001446219 5.2225599999999997e-005 0.0;
     0.0001349098 4.8718399999999998e-005 0.0;
     0.000125852 4.5447469999999999e-005 0.0;
     1.1741300000000000e-004 4.2400000000000001e-005 0.0;
     1.0955150000000000e-004 3.9561039999999999e-005 0.0;
     1.0222450000000000e-004 3.6915120000000003e-005 0.0;
     9.5394450000000006e-005 3.4448680000000000e-005 0.0;
     8.9023900000000000e-005 3.2148160000000002e-005 0.0;
     8.3075270000000005e-005 3.0000000000000001e-005 0.0;
     7.7512689999999993e-005 2.7991250000000001e-005 0.0;
     7.2313040000000000e-005 2.6113560000000001e-005 0.0;
     6.7457780000000002e-005 2.4360240000000000e-005 0.0;
     6.2928439999999995e-005 2.2724610000000000e-005 0.0;
     5.8706519999999999e-005 2.1200000000000000e-005 0.0;
     5.4770279999999998e-005 1.9778550000000001e-005 0.0;
     5.1099179999999999e-005 1.8452850000000001e-005 0.0;
     4.7676539999999999e-005 1.7216869999999999e-005 0.0;
     4.4485669999999998e-005 1.6064590000000001e-005 0.0;
     4.1509940000000003e-005 1.4990000000000000e-005 0.0]'

# compute the perceived color of a spectral distribution function
function spectrum2xyz(intensity::Function)
    Me = [ intensity(i) for i=380:780 ]
    X = dot(Me, cie_color_match[1, :])
    Y = dot(Me, cie_color_match[2, :])
    Z = dot(Me, cie_color_match[3, :])
    [X; Y; Z] ./ (X+Y+Z)
end

# convert a wavelength to an RGB approximation of its appearance
function wavelength2rgb(lambda::Real)
    idx = lambda - 380 + 1
    frac = idx - truncate(idx)
    idx = int32(idx)
    if idx < 1 || idx >= size(cie_color_match,2)
        return [0;0;0]
    end

    xyz = (1-frac)*cie_color_match[:,idx] + frac*cie_color_match[:,idx+1]
    xyz /= 1.25
    xyz2rgb(sRGBsystem, xyz)
end
