"""OpenStack provider file storage on Swift

Basically a limited wrapper around the underlying api calls, with a few added
quirks. There's some specific handling for 404 responses, raises FileNotFound
on GET, and on PUT attempts to create the container then retries.

Expects file-like objects for data. This isn't terribly useful as it doesn't
fit well with the twisted model for chunking http data and most objects are
small anyway.

The main complication is the get_url method, which requires the generation of
link that can be used by any http client to fetch a particular object without
authentication. This is not possible in the general case in swift, however
there are a few ways around the problem:

* The stub nova-objectstore service doesn't check access anyway, see lp:947374
* The tempurl swift middleware if enabled can do this with some advance setup
* A container can have a more permissive ACL applying to all objects within

All of these require touching the network to at least get the swift endpoint
from the identity service, which as get_url doesn't return a deferred is
problematic. In practice it's only used after putting an object however, so
just raising if client has not yet been authenticated is good enough.
"""

from cStringIO import StringIO

from twisted.internet.defer import inlineCallbacks, returnValue

from juju import errors


class FileStorage(object):
    """Swift-backed :class:`FileStorage` abstraction"""

    def __init__(self, swift, container):
        self._swift = swift
        self._container = container

    def get_url(self, name):
        return self._swift.public_object_url(self._container, name)

    @inlineCallbacks
    def get(self, name):
        """Get a file object from Swift.

        :param unicode name: S3 key for the desired file

        :return: an open file object
        :rtype: :class:`twisted.internet.defer.Deferred`

        :raises: :exc:`juju.errors.FileNotFound` if the file doesn't exist
        """
        response, body = yield self._swift.get_object(self._container, name)
        if response.code == 404:
            raise errors.FileNotFound(name)
        if response.code != 200:
            raise errors.ProviderInteractionError(
                "Couldn't fetch object %r %r" % (response.code, body))
        returnValue(StringIO(body))

    @inlineCallbacks
    def put(self, remote_path, file_object):
        """Upload a file to Swift.

        :param unicode remote_path: key on which to store the content

        :param file_object: open file object containing the content

        :rtype: :class:`twisted.internet.defer.Deferred`
        """
        data = file_object.read()
        response, body = yield self._swift.put_object(self._container,
            remote_path, data)
        if response.code == 404:
            response, body = yield self._swift.put_container(self._container)
            if response.code != 201:
                raise errors.ProviderInteractionError(
                    "Couldn't create container %r" % (self._container,))
            response, body = yield self._swift.put_object(self._container,
                remote_path, data)
        if response.code != 201:
            raise errors.ProviderInteractionError(
                "Couldn't create object %r %r" % (response.code, remote_path))
