/*******************************************************************************
  Copyright (c) 2023, Intel Corporation

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

      * Redistributions of source code must retain the above copyright notice,
        this list of conditions and the following disclaimer.
      * Redistributions in binary form must reproduce the above copyright
        notice, this list of conditions and the following disclaimer in the
        documentation and/or other materials provided with the distribution.
      * Neither the name of Intel Corporation nor the names of its contributors
        may be used to endorse or promote products derived from this software
        without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*******************************************************************************/

#include "intel-ipsec-mb.h"

#ifndef SM3_H
#define SM3_H

/**
 * @brief Authenticates message with SM3 hash algorithm
 *
 * @param tag pointer to store computed SM3 digest
 * @param tag_length output tag length in bytes
 * @param msg pointer to input message to compute digest over
 * @param msg_length length of the message in bytes
 */
IMB_DLL_LOCAL void
sm3_msg(void *tag, const uint64_t tag_len, const void *msg, const uint64_t msg_length);

/**
 * @brief Authenticates one block with SM3 hash algorithm
 *
 * @param tag pointer to store computed SM3 digest
 * @param msg pointer to input block compute digest over
 */
IMB_DLL_LOCAL void
sm3_one_block(void *tag, const void *msg);

/**
 * @brief Authenticates message with SM3-HMAC hash algorithm
 *
 * @param tag pointer to store computed SM3 digest
 * @param tag_length output tag length in bytes
 * @param msg pointer to input message to compute digest over
 * @param msg_length length of the message in bytes
 * @param ipad_hash block-sized inner padding
 * @param opad_hash block-sized outer padding
 */
IMB_DLL_LOCAL void
sm3_hmac_msg(void *tag, const uint64_t tag_len, const void *msg, const uint64_t msg_length,
             const void *ipad_hash, const void *opad_hash);

#endif /* SM3_H */
