/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef GLOBUS_GSI_AUTHZ_CALLOUT_ERROR_H
#define GLOBUS_GSI_AUTHZ_CALLOUT_ERROR_H

/**
 * @file globus_gsi_authz_callout_error.h
 * @brief GSI Authorization Callout Error API
 */

#include "globus_common.h"

#ifdef __cplusplus
extern "C" {
#endif

#ifndef GLOBUS_GLOBAL_DOCUMENT_SET
/**
 * @mainpage GSI Authorization Callout Error API
 * @copydoc globus_gsi_authz_callout_error 
 * For more details, see the @ref globus_gsi_authz_callout_error documentation
 */
#endif

/**
 * @defgroup globus_gsi_authz_callout_error GSI Authorization Callout Error API
 * @brief Authorization Callout Error Objects
 *
 * @details
 * The GSI Authorization Callout Error API defines a set of Globus error 
 * codes which are used by the error objects generated by the GSI Authorization
 * API. Applications or libraries that wish to use this interface must include
 * the globus_gsi_authz_callout_error.h header.
 */

/**
 * @brief Error codes
 * @ingroup globus_gsi_authz_callout_error
 * @details
 * Error code types used by the GSI Authorization Callout API
 */
typedef enum
{
    /** Authz callout error */
    GLOBUS_GSI_AUTHZ_CALLOUT_AUTHZ_CALLOUT_ERROR = 0,
    /** Authorization denied by callout */
    GLOBUS_GSI_AUTHZ_CALLOUT_AUTHZ_DENIED_BY_CALLOUT = 1,
    /** Configuration Error */
    GLOBUS_GSI_AUTHZ_CALLOUT_CONFIGURATION_ERROR = 2,
    /** System Error */
    GLOBUS_GSI_AUTHZ_CALLOUT_SYSTEM_ERROR = 3,
    /** Credentials Error */
    GLOBUS_GSI_AUTHZ_CALLOUT_CREDENTIAL_ERROR = 4,
    /** A invalid parameter was detected */
    GLOBUS_GSI_AUTHZ_CALLOUT_BAD_ARGUMENT_ERROR = 5,
    GLOBUS_GSI_AUTHZ_CALLOUT_ERROR_LAST = 6
}
globus_gsi_authz_callout_error_t;

extern globus_module_descriptor_t globus_gsi_authz_callout_error_module;

/**
 * @brief Module descriptor
 * @ingroup globus_gsi_authz_callout_error
 */
#define GLOBUS_GSI_AUTHZ_CALLOUT_ERROR_MODULE (&globus_gsi_authz_callout_error_module)

extern char * globus_gsi_authz_callout_error_strings[];

/**
 * @brief Create an authorization callout error object
 * @ingroup globus_gsi_authz_callout_error
 * @details
 * Update the __RESULT parameter to contain a new globus_result_t
 * value pointing to an authorization error of type __TYPE with the
 * detail string __ERRSTR.
 * @hideinitializer
 * @param __RESULT
 *     A globus_result_t value pointing to a causal error. This will be
 *     modified by this macro to point to the new error object.
 * @param __TYPE
 *     The authorization error value from the globus_gsi_authz_callout_error_t
 *     enumeration.
 * @param __ERRSTR
 *     A parenthesized printf string containing the description of the error
 */
#define GLOBUS_GSI_AUTHZ_CALLOUT_ERROR(__RESULT, __TYPE, __ERRSTR)       \
{                                                                        \
    char *                          _tmp_str_ =                          \
        globus_common_create_string __ERRSTR;                            \
    (__RESULT) = globus_error_put(                                       \
        globus_error_construct_error(                                    \
            GLOBUS_GSI_AUTHZ_CALLOUT_ERROR_MODULE,                       \
            (__RESULT) ? globus_error_get(__RESULT) : NULL,              \
            __TYPE,                                                      \
            __FILE__, \
            "Authz Callout", \
            __LINE__, \
            "%s%s%s",                                                    \
            globus_gsi_authz_callout_error_strings[__TYPE],              \
            _tmp_str_ ? ": " : "",                                       \
            _tmp_str_ ? _tmp_str_ : ""));                                \
    if(_tmp_str_) free(_tmp_str_);                                       \
}

/**
 * @brief Create an authorization callout error object with an errno
 * @ingroup globus_gsi_authz_callout_error
 * @details
 * Update the __RESULT parameter to contain a new globus_result_t
 * value pointing to an authorization error with the errno.
 * @hideinitializer
 * @param __RESULT
 *     A globus_result_t value pointing to a causal error. This will be
 *     modified by this macro to point to the new error object.
 * @param __ERRNO
 *     The errno value of this new error object.
 */
#define GLOBUS_GSI_AUTHZ_CALLOUT_ERRNO_ERROR(__RESULT, __ERRNO)          \
{                                                                        \
    (__RESULT) = globus_error_put(                                       \
        globus_error_construct_errno_error(                              \
            GLOBUS_GSI_AUTHZ_CALLOUT_ERROR_MODULE,                       \
            (__RESULT) ? globus_error_get(__RESULT) : NULL,              \
            __ERRNO));                                                   \
}


#ifdef __cplusplus
}
#endif

#endif
