static const char* op_c_source =
"/* This file is an image processing operation for GEGL                        \n"
" *                                                                            \n"
" * GEGL is free software; you can redistribute it and/or                      \n"
" * modify it under the terms of the GNU Lesser General Public                 \n"
" * License as published by the Free Software Foundation; either               \n"
" * version 3 of the License, or (at your option) any later version.           \n"
" *                                                                            \n"
" * GEGL is distributed in the hope that it will be useful,                    \n"
" * but WITHOUT ANY WARRANTY; without even the implied warranty of             \n"
" * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          \n"
" * Lesser General Public License for more details.                            \n"
" *                                                                            \n"
" * You should have received a copy of the GNU Lesser General Public           \n"
" * License along with GEGL; if not, see <http://www.gnu.org/licenses/>.       \n"
" *                                                                            \n"
" * Copyright (C) 1995 Spencer Kimball and Peter Mattis                        \n"
" *                                                                            \n"
" * This plug-in uses the algorithm described by John Schlag, \"Fast           \n"
" * Embossing Effects on Raster Image Data\" in Graphics Gems IV (ISBN         \n"
" * 0-12-336155-9).  It takes a grayscale image to be applied as a             \n"
" * bump-map to another image, producing a nice embossing effect.              \n"
" *                                                                            \n"
" * Ported to gegl by Dov Grobgeld <dov.grobgeld@gmail.com>                    \n"
" *                                                                            \n"
" */                                                                           \n"
"                                                                              \n"
"#include \"config.h\"                                                         \n"
"#include <glib/gi18n-lib.h>                                                   \n"
"                                                                              \n"
"#ifdef GEGL_PROPERTIES                                                        \n"
"                                                                              \n"
"enum_start (gegl_bump_map_type)                                               \n"
"  enum_value (GEGL_BUMP_MAP_TYPE_LINEAR,     \"linear\",     N_(\"Linear\"))  \n"
"  enum_value (GEGL_BUMP_MAP_TYPE_SPHERICAL,  \"spherical\",  N_(\"Spherical\"))\n"
"  enum_value (GEGL_BUMP_MAP_TYPE_SINUSOIDAL, \"sinusodial\", N_(\"Sinusoidal\"))\n"
"enum_end (GeglBumpMapType)                                                    \n"
"                                                                              \n"
"property_enum (type, _(\"Type\"), GeglBumpMapType, gegl_bump_map_type,        \n"
"               GEGL_BUMP_MAP_TYPE_LINEAR)                                     \n"
"  description (_(\"Type of map\"))                                            \n"
"                                                                              \n"
"property_boolean (compensate, _(\"Compensate\"), TRUE)                        \n"
"  description (_(\"Compensate for darkening\"))                               \n"
"                                                                              \n"
"property_boolean (invert, _(\"Invert\"), FALSE)                               \n"
"  description (_(\"Invert bumpmap\"))                                         \n"
"                                                                              \n"
"property_boolean (tiled, _(\"Tiled\"), FALSE)                                 \n"
"  description (_(\"Tiled bumpmap\"))                                          \n"
"                                                                              \n"
"property_double  (azimuth, _(\"Azimuth\"), 135.0)                             \n"
"  value_range (0.0, 360.0)                                                    \n"
"  ui_meta     (\"unit\", \"degree\")                                          \n"
"                                                                              \n"
"property_double  (elevation, _(\"Elevation\"), 45.0)                          \n"
"  value_range (0.5, 90.0)                                                     \n"
"                                                                              \n"
"property_int  (depth, _(\"Depth\"), 3)                                        \n"
"  value_range (1, 65)                                                         \n"
"                                                                              \n"
"property_int  (offset_x, _(\"Offset X\"), 0)                                  \n"
"  value_range (-20000, 20000)                                                 \n"
"  ui_range (-1000, 1000)                                                      \n"
"  ui_meta  (\"axis\", \"x\")                                                  \n"
"  ui_meta  (\"unit\", \"pixel-coordinate\")                                   \n"
"                                                                              \n"
"property_int  (offset_y, _(\"Offset Y\"), 0)                                  \n"
"  value_range (-20000, 20000)                                                 \n"
"  ui_range (-1000, 1000)                                                      \n"
"  ui_meta  (\"axis\", \"y\")                                                  \n"
"  ui_meta  (\"unit\", \"pixel-coordinate\")                                   \n"
"                                                                              \n"
"property_double  (waterlevel, _(\"Waterlevel\"), 0.0)                         \n"
"  description(_(\"Level that full transparency should represent\"))           \n"
"  value_range (0.0, 1.0)                                                      \n"
"                                                                              \n"
"property_double  (ambient, _(\"Ambient lighting factor\"), 0.0)               \n"
"  value_range (0.0, 1.0)                                                      \n"
"                                                                              \n"
"#else                                                                         \n"
"                                                                              \n"
"#define GEGL_OP_COMPOSER                                                      \n"
"#define GEGL_OP_C_SOURCE bump-map.c                                           \n"
"                                                                              \n"
"#include \"gegl-op.h\"                                                        \n"
"#include <math.h>                                                             \n"
"                                                                              \n"
"#define LUT_SIZE 2048                                                         \n"
"                                                                              \n"
"typedef struct                                                                \n"
"{                                                                             \n"
"  gdouble lx, ly;           /* X and Y components of light vector */          \n"
"  gdouble nz2, nzlz;        /* nz^2, nz*lz */                                 \n"
"  gdouble background;       /* Shade for vertical normals */                  \n"
"  gdouble compensation;     /* Background compensation */                     \n"
"  gdouble lut[LUT_SIZE];    /* Look-up table for modes - should be made interpolated*/\n"
"                                                                              \n"
"  gboolean in_has_alpha;    /* babl formats info */                           \n"
"  gboolean bm_has_alpha;                                                      \n"
"  gint     in_components;                                                     \n"
"  gint     bm_components;                                                     \n"
"} bumpmap_params_t;                                                           \n"
"                                                                              \n"
"static void                                                                   \n"
"bumpmap_init_params (GeglProperties *o,                                       \n"
"                     const Babl     *in_format,                               \n"
"                     const Babl     *bm_format)                               \n"
"{                                                                             \n"
"  bumpmap_params_t *params = (bumpmap_params_t *) o->user_data;               \n"
"  gdouble lz, nz;                                                             \n"
"  gint    i;                                                                  \n"
"                                                                              \n"
"  gint scale = LUT_SIZE - 1;                                                  \n"
"                                                                              \n"
"  /* Convert to radians */                                                    \n"
"  const gdouble azimuth   = G_PI * o->azimuth / 180.0;                        \n"
"  const gdouble elevation = G_PI * o->elevation / 180.0;                      \n"
"                                                                              \n"
"  /* Calculate the light vector */                                            \n"
"  params->lx = cos (azimuth) * cos (elevation);                               \n"
"  params->ly = sin (azimuth) * cos (elevation);                               \n"
"  lz         = sin (elevation);                                               \n"
"                                                                              \n"
"  /* Calculate constant Z component of surface normal */                      \n"
"  nz           = 6.0 / o->depth;                                              \n"
"  params->nz2  = nz * nz;                                                     \n"
"  params->nzlz = nz * lz;                                                     \n"
"                                                                              \n"
"  /* Optimize for vertical normals */                                         \n"
"  params->background = lz;                                                    \n"
"                                                                              \n"
"  /* Calculate darkness compensation factor */                                \n"
"  params->compensation = sin (elevation);                                     \n"
"                                                                              \n"
"  /* Create look-up table for map type */                                     \n"
"  for (i = 0; i < LUT_SIZE; i++)                                              \n"
"    {                                                                         \n"
"      gdouble n;                                                              \n"
"                                                                              \n"
"      switch (o->type)                                                        \n"
"        {                                                                     \n"
"        case GEGL_BUMP_MAP_TYPE_SPHERICAL:                                    \n"
"          n = (gdouble) i / scale - 1.0;                                      \n"
"          params->lut[i] = sqrt (1.0 - n * n) + 0.5;                          \n"
"          break;                                                              \n"
"                                                                              \n"
"        case GEGL_BUMP_MAP_TYPE_SINUSOIDAL:                                   \n"
"          n = (gdouble) i / scale;                                            \n"
"          params->lut[i] = (sin ((-G_PI / 2.0) + G_PI * n) + 1.0) / 2.0 + 0.5;\n"
"          break;                                                              \n"
"                                                                              \n"
"        case GEGL_BUMP_MAP_TYPE_LINEAR:                                       \n"
"        default:                                                              \n"
"          params->lut[i] = (gdouble) i / scale;                               \n"
"        }                                                                     \n"
"                                                                              \n"
"      if (o->invert)                                                          \n"
"        params->lut[i] = 1.0 - params->lut[i];                                \n"
"    }                                                                         \n"
"                                                                              \n"
"  /* babl format stuff */                                                     \n"
"  params->in_has_alpha  = babl_format_has_alpha (in_format);                  \n"
"  params->bm_has_alpha  = babl_format_has_alpha (bm_format);                  \n"
"  params->in_components = babl_format_get_n_components (in_format);           \n"
"  params->bm_components = babl_format_get_n_components (bm_format);           \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"bumpmap_row (gfloat         *row,                                             \n"
"             gint            width,                                           \n"
"             const gfloat   *bm_row1,                                         \n"
"             const gfloat   *bm_row2,                                         \n"
"             const gfloat   *bm_row3,                                         \n"
"             gint            bm_width,                                        \n"
"             gboolean        row_in_bumpmap,                                  \n"
"             gint            roix,                                            \n"
"             GeglProperties *o)                                               \n"
"{                                                                             \n"
"  bumpmap_params_t *params = (bumpmap_params_t *) o->user_data;               \n"
"  gint xofs1, xofs2, xofs3;                                                   \n"
"  gint x, b;                                                                  \n"
"  gfloat *buf = row;                                                          \n"
"                                                                              \n"
"  for (x = 0; x < width; x++)                                                 \n"
"    {                                                                         \n"
"      gdouble shade;                                                          \n"
"      gdouble nx, ny;                                                         \n"
"                                                                              \n"
"      /* Calculate surface normal from bump map */                            \n"
"                                                                              \n"
"      if (o->tiled || (row_in_bumpmap &&                                      \n"
"            roix + x >= - o->offset_x && roix + x < - o->offset_x + bm_width))\n"
"        {                                                                     \n"
"          xofs2 = (x + 1) * params->bm_components;                            \n"
"          xofs1 = CLAMP (x * params->bm_components, 0, (width + 1) * params->bm_components);\n"
"          xofs3 = CLAMP ((x + 2) * params->bm_components, 0, (width + 1) * params->bm_components);\n"
"                                                                              \n"
"          nx = (bm_row1[xofs1] + bm_row2[xofs1] + bm_row3[xofs1] -            \n"
"                bm_row1[xofs3] - bm_row2[xofs3] - bm_row3[xofs3]);            \n"
"                                                                              \n"
"          ny = (bm_row3[xofs1] + bm_row3[xofs2] + bm_row3[xofs3] -            \n"
"                bm_row1[xofs1] - bm_row1[xofs2] - bm_row1[xofs3]);            \n"
"        }                                                                     \n"
"      else                                                                    \n"
"        {                                                                     \n"
"          nx = ny = 0.0;                                                      \n"
"        }                                                                     \n"
"                                                                              \n"
"      if ((nx == 0.0) && (ny == 0.0))                                         \n"
"        {                                                                     \n"
"          shade = params->background;                                         \n"
"        }                                                                     \n"
"      else                                                                    \n"
"        {                                                                     \n"
"          gdouble ndotl = nx * params->lx + ny * params->ly + params->nzlz;   \n"
"                                                                              \n"
"          if (ndotl < 0.0)                                                    \n"
"            {                                                                 \n"
"              shade = params->compensation * o->ambient;                      \n"
"            }                                                                 \n"
"          else                                                                \n"
"            {                                                                 \n"
"              shade = ndotl / sqrt (nx * nx + ny * ny + params->nz2);         \n"
"              shade += MAX(0, (params->compensation - shade)) * o->ambient;   \n"
"            }                                                                 \n"
"        }                                                                     \n"
"                                                                              \n"
"      /* Paint */                                                             \n"
"      if (o->compensate)                                                      \n"
"        {                                                                     \n"
"          for (b = 0 ; b < 3; b++)                                            \n"
"            buf[b] = (buf[b] * shade) / params->compensation;                 \n"
"        }                                                                     \n"
"      else                                                                    \n"
"        {                                                                     \n"
"          for (b = 0 ; b < 3; b++)                                            \n"
"            buf[b] = buf[b] * shade;                                          \n"
"        }                                                                     \n"
"                                                                              \n"
"      if (params->in_has_alpha)                                               \n"
"        buf[3] = buf[3];                                                      \n"
"                                                                              \n"
"      buf += params->in_components;                                           \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"bumpmap_convert (gfloat         *buffer,                                      \n"
"                 glong           n_pixels,                                    \n"
"                 GeglProperties *o)                                           \n"
"{                                                                             \n"
"  bumpmap_params_t *params   = (bumpmap_params_t *) o->user_data;             \n"
"  gint    idx;                                                                \n"
"  gint    scale = LUT_SIZE - 1;                                               \n"
"  gfloat  *p = buffer;                                                        \n"
"                                                                              \n"
"  while (n_pixels--)                                                          \n"
"    {                                                                         \n"
"      gfloat value = CLAMP (p[0], 0.0, 1.0);                                  \n"
"                                                                              \n"
"      if (params->bm_has_alpha)                                               \n"
"        {                                                                     \n"
"          gfloat alpha = CLAMP (p[1], 0.0, 1.0);                              \n"
"          idx = (gint) ((o->waterlevel + (value - o->waterlevel) * alpha) * scale);\n"
"        }                                                                     \n"
"      else                                                                    \n"
"        idx = (gint) (value * scale);                                         \n"
"                                                                              \n"
"      p[0] = params->lut[idx];                                                \n"
"      p += params->bm_components;                                             \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"prepare (GeglOperation *operation)                                            \n"
"{                                                                             \n"
"  GeglProperties *o = GEGL_PROPERTIES (operation);                            \n"
"  const Babl *in_format = gegl_operation_get_source_format (operation, \"input\");\n"
"  const Babl *bm_format = gegl_operation_get_source_format (operation, \"aux\");\n"
"                                                                              \n"
"  if (!o->user_data)                                                          \n"
"    o->user_data = g_slice_new0 (bumpmap_params_t);                           \n"
"                                                                              \n"
"  if (in_format)                                                              \n"
"    {                                                                         \n"
"      if (babl_format_has_alpha (in_format))                                  \n"
"        in_format = babl_format (\"R'G'B'A float\");                          \n"
"      else                                                                    \n"
"        in_format = babl_format (\"R'G'B' float\");                           \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    in_format = babl_format (\"R'G'B' float\");                               \n"
"                                                                              \n"
"  if (bm_format)                                                              \n"
"    {                                                                         \n"
"      if (babl_format_has_alpha (bm_format))                                  \n"
"        bm_format = babl_format (\"Y'A float\");                              \n"
"      else                                                                    \n"
"        bm_format = babl_format (\"Y' float\");                               \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    bm_format = babl_format (\"Y' float\");                                   \n"
"                                                                              \n"
"  bumpmap_init_params (o, in_format, bm_format);                              \n"
"                                                                              \n"
"  gegl_operation_set_format (operation, \"input\",  in_format);               \n"
"  gegl_operation_set_format (operation, \"aux\",    bm_format);               \n"
"  gegl_operation_set_format (operation, \"output\", in_format);               \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"finalize (GObject *object)                                                    \n"
"{                                                                             \n"
"  GeglOperation *op = (void*) object;                                         \n"
"  GeglProperties *o = GEGL_PROPERTIES (op);                                   \n"
"                                                                              \n"
"  if (o->user_data)                                                           \n"
"    {                                                                         \n"
"      g_slice_free (bumpmap_params_t, o->user_data);                          \n"
"      o->user_data = NULL;                                                    \n"
"    }                                                                         \n"
"                                                                              \n"
"  G_OBJECT_CLASS (gegl_op_parent_class)->finalize (object);                   \n"
"}                                                                             \n"
"                                                                              \n"
"static gboolean                                                               \n"
"process (GeglOperation       *operation,                                      \n"
"         GeglBuffer          *input,                                          \n"
"         GeglBuffer          *aux,                                            \n"
"         GeglBuffer          *output,                                         \n"
"         const GeglRectangle *roi,                                            \n"
"         gint                 level)                                          \n"
"{                                                                             \n"
"  GeglProperties   *o        = GEGL_PROPERTIES (operation);                   \n"
"  bumpmap_params_t *params   = (bumpmap_params_t *) o->user_data;             \n"
"                                                                              \n"
"  const Babl *in_format = gegl_operation_get_format (operation, \"input\");   \n"
"  const Babl *bm_format = gegl_operation_get_format (operation, \"aux\");     \n"
"                                                                              \n"
"  GeglAbyssPolicy repeat_mode = o->tiled ? GEGL_ABYSS_LOOP : GEGL_ABYSS_CLAMP;\n"
"                                                                              \n"
"  GeglRectangle bm_rect, bm_extent;                                           \n"
"                                                                              \n"
"  gfloat  *src_row, *src_buffer, *bm_buffer;                                  \n"
"  gfloat  *bm_row1, *bm_row2, *bm_row3;                                       \n"
"  gint     y;                                                                 \n"
"  gboolean row_in_bumpmap;                                                    \n"
"                                                                              \n"
"  src_buffer = g_new (gfloat, roi->width * roi->height * params->in_components);\n"
"                                                                              \n"
"  gegl_buffer_get (input, roi, 1.0, in_format, src_buffer,                    \n"
"                   GEGL_AUTO_ROWSTRIDE, GEGL_ABYSS_NONE);                     \n"
"                                                                              \n"
"  if (aux)                                                                    \n"
"    {                                                                         \n"
"      bm_extent = *gegl_buffer_get_extent (aux);                              \n"
"                                                                              \n"
"      bm_rect = *roi;                                                         \n"
"      bm_rect.x += o->offset_x - 1;                                           \n"
"      bm_rect.y += o->offset_y - 1;                                           \n"
"      bm_rect.width += 2;                                                     \n"
"      bm_rect.height += 2;                                                    \n"
"                                                                              \n"
"      bm_buffer = g_new (gfloat, bm_rect.width * bm_rect.height * params->bm_components);\n"
"                                                                              \n"
"      gegl_buffer_get (aux, &bm_rect, 1.0, bm_format,                         \n"
"                       bm_buffer, GEGL_AUTO_ROWSTRIDE, repeat_mode);          \n"
"                                                                              \n"
"      bumpmap_convert (bm_buffer, bm_rect.width * bm_rect.height, o);         \n"
"                                                                              \n"
"      bm_row1 = bm_buffer;                                                    \n"
"      bm_row2 = bm_buffer + (bm_rect.width * params->bm_components);          \n"
"      bm_row3 = bm_buffer + (2 * bm_rect.width * params->bm_components);      \n"
"                                                                              \n"
"      for (y = roi->y; y < roi->y + roi->height; y++)                         \n"
"        {                                                                     \n"
"          row_in_bumpmap = (y >= - o->offset_y && y < - o->offset_y + bm_extent.height);\n"
"                                                                              \n"
"          src_row = src_buffer + ((y - roi->y) * roi->width * params->in_components);\n"
"                                                                              \n"
"          bumpmap_row (src_row, roi->width,                                   \n"
"                       bm_row1, bm_row2, bm_row3, bm_extent.width,            \n"
"                       row_in_bumpmap, roi->x, o);                            \n"
"                                                                              \n"
"          bm_row1 = bm_row2;                                                  \n"
"          bm_row2 = bm_row3;                                                  \n"
"          bm_row3 = bm_row2 + (bm_rect.width * params->bm_components);        \n"
"       }                                                                      \n"
"                                                                              \n"
"       g_free (bm_buffer);                                                    \n"
"    }                                                                         \n"
"                                                                              \n"
"  gegl_buffer_set (output, roi, level, in_format,                             \n"
"                   src_buffer, GEGL_AUTO_ROWSTRIDE);                          \n"
"                                                                              \n"
"  g_free (src_buffer);                                                        \n"
"                                                                              \n"
"  return TRUE;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"static GeglRectangle                                                          \n"
"get_bounding_box (GeglOperation *operation)                                   \n"
"{                                                                             \n"
"  GeglRectangle  result  = { 0, 0, 0, 0 };                                    \n"
"  GeglRectangle *in_rect = gegl_operation_source_get_bounding_box (operation, \"input\");\n"
"                                                                              \n"
"  if (! in_rect)                                                              \n"
"    return result;                                                            \n"
"                                                                              \n"
"  return *in_rect;                                                            \n"
"}                                                                             \n"
"                                                                              \n"
"static GeglRectangle                                                          \n"
"get_required_for_output (GeglOperation       *operation,                      \n"
"                         const gchar         *input_pad,                      \n"
"                         const GeglRectangle *roi)                            \n"
"{                                                                             \n"
"  if (!strcmp (input_pad, \"aux\"))                                           \n"
"    {                                                                         \n"
"      GeglRectangle bm_rect = *gegl_operation_source_get_bounding_box (operation,\n"
"                                                                       \"aux\");\n"
"      if (gegl_rectangle_is_empty (&bm_rect))                                 \n"
"        return *roi;                                                          \n"
"                                                                              \n"
"      return bm_rect;                                                         \n"
"    }                                                                         \n"
"                                                                              \n"
"  return *roi;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"gegl_op_class_init (GeglOpClass *klass)                                       \n"
"{                                                                             \n"
"  GObjectClass               *object_class;                                   \n"
"  GeglOperationClass         *operation_class;                                \n"
"  GeglOperationComposerClass *composer_class;                                 \n"
"                                                                              \n"
"  object_class    = G_OBJECT_CLASS (klass);                                   \n"
"  operation_class = GEGL_OPERATION_CLASS (klass);                             \n"
"  composer_class  = GEGL_OPERATION_COMPOSER_CLASS (klass);                    \n"
"                                                                              \n"
"  object_class->finalize                   = finalize;                        \n"
"                                                                              \n"
"  operation_class->prepare                 = prepare;                         \n"
"  operation_class->get_bounding_box        = get_bounding_box;                \n"
"  operation_class->get_required_for_output = get_required_for_output;         \n"
"  operation_class->opencl_support          = FALSE;                           \n"
"                                                                              \n"
"  composer_class->process                  = process;                         \n"
"                                                                              \n"
"  gegl_operation_class_set_keys (operation_class,                             \n"
"    \"name\",        \"gegl:bump-map\",                                       \n"
"    \"categories\",  \"light\",                                               \n"
"    \"title\",       _(\"Bump Map\"),                                         \n"
"    \"license\",     \"GPL3+\",                                               \n"
"    \"description\", _(\"This plug-in uses the algorithm described by John \" \n"
"                     \"Schlag, \\\"Fast Embossing Effects on Raster Image \"  \n"
"                     \"Data\\\" in Graphics GEMS IV (ISBN 0-12-336155-9). \"  \n"
"                     \"It takes a buffer to be applied as a bump \"           \n"
"                     \"map to another buffer and produces a nice embossing \" \n"
"                     \"effect.\"),                                            \n"
"    NULL);                                                                    \n"
"}                                                                             \n"
"                                                                              \n"
"#endif                                                                        \n"
;
