/*
 * This file is part of din.
 *
 * din is copyright (c) 2006 - 2012 S Jagannathan <jag@dinisnoise.org>
 * For more information, please visit http://dinisnoise.org
 *
 * din is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * din is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with din.  If not, see <http://www.gnu.org/licenses/>.
 *
*/
#include "field.h"
#include "font.h"
#include "utils.h"
#include "input.h"
#include "key_consts.h"
#include "utils.h"

#include <string>
#include <sstream>
#include <list>
#include <cstdlib>

#include <iostream>
#include <iomanip>

using namespace std;

extern int lmb;
extern int mousex, mouseyy;

void accept_keypress (int);

int field::handle_input () {

  widget::handle_input ();

  update ();

  extern int xkey;

  // locate mouse in field
  if (lmb && !clicked_lmb) {
    hittest (mousex, mouseyy);
    accept_keypress (focus);
    if (focus) widget::focus = this; else widget::focus = 0;
    clicked_lmb = 1;
  } else clicked_lmb = 0;

  if (focus) { // edit text

    if ((xkey > 9) && (xkey < 127) && (xkey != 13)) { // char from X

      if (mode == pushback)
        text.push_back (xkey);
      else
        text.insert (text.begin() + cursor, xkey);

      ++len;
      ++last;
      if (++cursor >= len) {
        cursor = len;
        mode = pushback;
      }

      calc_view ();

    }

    if (keypressed (k_enter) || keypressed (k_esc)) { // quit editing
      if (lsnr) lsnr->changed (*this);
      focus = 0;
      accept_keypress (focus);
      widget::focus = 0;
      return 1;
    }

    else if (keypressedd (k_left)) { // move cursor left

      mode = insert;
      if (--cursor < 0) cursor = 0;
      calc_view ();

    } else if (keypressedd (k_right)) { // move cursor right

      mode = insert;
      if (++cursor >= len) {
        cursor = len;
        mode = pushback;
      }
      calc_view ();

    } else if (keypressedd (k_backspace)) { // delete

      if (cursor) {

        --cursor;

        if (--left < 0) {
          left = 0;
          right = width - 1;
        } else --right;

        text.erase (text.begin () + cursor);

        --len;
        --last;

        calc_view ();

      }

    }

    xkey = 0;
  }

  return focus;

}

void field::set_text (const string& txt) {

  text = txt;

  len = text.length ();
  last = len - 1;
  cursor = len;

  left = 0;
  right = width - 1;

  calc_view_text ();

  mode = pushback;

}

void field::set_text (int i) {
  sprintf (sbuf, "%d", i);
  set_text (sbuf);
}

void field::set_text (float f) {
  sprintf (sbuf, "%.2f", f);
  set_text (sbuf);
}

void field::calc_view () {
  if (cursor < left) {
    --left;
    --right;
  } else if (cursor > right) {
    ++left;
    ++right;
  }
  calc_view_text ();
  calc_cursor ();
}

void field::calc_view_text () {

  int si = max (0, left);
  int sj = min (right, last);
  int sd = sj - si + 1;
  view_text = text.substr (si, sd);

}

void field::calc_cursor () {
  int cd = cursor - left;
  string cstr (view_text.substr (0, cd));
  offset = get_char_width (cstr);
}

void field::update () {
  const box<int>& e = get_extents ();
  set_extents (e.left, e.bottom, e.left + get_char_width (view_text) + fnt.max_char_width, e.bottom + fnt.avg_char_height);
}

void field::draw_cursor (int x, int y) {
  int cx = x + offset;
  int cy = y - fnt.headroom;
  const color& clr = get_color ();
  glColor3f (clr.r, clr.g, clr.b);
  glBegin (GL_LINES);
    glVertex2i (cx, y);
    glVertex2i (cx - fnt.avg_char_width, cy);
    glVertex2i (cx, y);
    glVertex2i (cx + fnt.avg_char_width, cy);
  glEnd ();
}

void field::draw () {

  widget::draw ();
  int x, y; get_pos (x, y);
  draw_string (view_text, x, y);
  if (focus) draw_cursor (x, y);

  //draw_bbox ();

  /*const box<int>& e = get_extents ();
  glBegin (GL_LINES);
    glVertex2i (e.left, e.midy + 5);
    glVertex2i (e.left, e.midy - 5);
    glVertex2i (e.right, e.midy + 5);
    glVertex2i (e.right, e.midy - 5);
  glEnd ();*/

}

field::field (int w) : width (w) {

  set_text ("");

  update ();

  clicked_lmb = 0;
  focus = 0;

}

field::field (int x, int y, const string& txt, int w) : width (w) {

  set_text (txt);

  update ();

  set_pos (x, y);

  clicked_lmb = 0;

}

field::operator int() const {
  return (atoi (text.c_str()));
}

field::operator float() const {
  return ((float) atof (text.c_str()));
}

void field::set_listener (change_listener<field> *fl) {
  lsnr = fl;
}

int field::hittest (int x, int y) {

  const box<int>& e = get_extents ();

  if (inbox (e, x, y)) { // in the box

    focus = 1; // so got focus

    int n = view_text.length ();
    if (n == 0) {
      cursor = 0;
      mode = pushback;
      calc_view ();
      return focus;
    }

    // locate cursor in text
    int l = e.left, r;
    for (int i = 0; i < n; ++i) {
      r = e.left + get_char_width (view_text.substr (0, i + 1));
      if (inrange (l, x, r)) { // found!
        int xl = x - l, xr = r - x;
        if (xl < xr) cursor = left + i; else cursor = left + i + 1;
        mode = insert;
        calc_view ();
        return focus;
      }
      l = r;
    }

    // not found so place after last char
    cursor = len;
    mode = pushback;
    calc_view ();
    return focus;

  }

  focus = 0;

  return focus;

}
